#include "menuitem.h"

//############################################################################
// Construction ##############################################################
//############################################################################

/** Default constructor.
 * @param n Name of this item.
 * @param func Command function.
 */
MenuItem::MenuItem(std::string n, void (*func)())
  : parent(NULL), index(0)
{
  this->name = n;
  this->command = func;
}

/** Default destructor.
 */
MenuItem::~MenuItem()
{
  for(std::vector<MenuItem*>::iterator i = this->items.begin(),
      e = this->items.end(); (i != e); ++i)
  {
    delete *i;
  }
}

//############################################################################
// Other methods #############################################################
//############################################################################

/** Adds a new menu item.
 * @param op New menu item.
 */
void MenuItem::add_item(MenuItem *op)
{
  int idx = this->num_items();

  this->items.push_back(op);
  op->parent = this;
  op->index = idx;
}

/** Executes the command within this menu item or returns the submenu within.
 * Returning the submenu is interpreted as returning the first item from the
 * list.
 * @return Menu to descend or NULL if command was executed instead.
 */
MenuItem* MenuItem::execute()
{
  if(this->command)
  {
    this->command();
    return NULL;
  }
  else if(this->items.size() > 0)
  {
    return this->items[0];
  }
  else
  {
    return this->parent;
  }
}

/** Get the index of this menuitem.
 * @return The index as integer.
 */
int MenuItem::get_index() const
{
  return this->index;
}

/** Get the item inside this menu of the given index.
 * @param idx Index within this menu.
 * @return A pointer to MenuItem of the given index.
 */
MenuItem* MenuItem::get_item(int idx)
{
  return this->items[idx];
}

/** Get the name of this menuitem.
 * @return Name as const string reference.
 */
const std::string& MenuItem::get_name() const
{
  return this->name;
}

/** Get the parent of this item. This is neccessary when displaying menus, we
 * want to get the parent menu and iterate through it.
 * @return Pointer to the parent.
 */
MenuItem* MenuItem::get_parent()
{
  return this->parent;
}

/** Get the number of menuitems within this menu.
 * @return Number of sub-menuitems or 0 if none (a command).
 */
int MenuItem::num_items()
{
  return static_cast<int>(this->items.size());
}

//############################################################################
// End #######################################################################
//############################################################################

