#include "challenge.h"
#include "constants.h"
#include "data.h"
#include "game.h"
#include "globals.h"
#include "mainmenu.h"
#include "options.h"

#include "SDL_mixer.h"

#include <sstream>
#include <time.h>

#include "SDL.h"

//############################################################################
// Globals ###################################################################
//############################################################################

static const int
  DEFAULT_SCREEN_W = 640,
  DEFAULT_SCREEN_H = 480,
  DEFAULT_SCREEN_B = 32,
  DEFAULT_SOUND_RATE = 44100,
  DEFAULT_SOUND_CHANNELS = 2,
  DEFAULT_SOUND_BUFFER = 4096,
  DEFAULT_SOUND_POLYPHONY = 64;

#define FPS_CAP

//############################################################################
// Event handler #############################################################
//############################################################################

/** Event loop is a function that takes as it's parameter an object that can
 * receive events. It will call (virtual) methods of that object. That is:
 * keydown(int) - Inputs a command (usually, a keypress) to the object.
 * keyup(int) - As before, but release of key instead of press.
 * All commands are given in sequence before the next transmissions.
 * tick() - An order to perform all calculation that must be done every frame.
 * draw() - An order to output everything to the screen. Will be done if there
 * is time.
 * @param op The event handler object as a pointer.
 * @param screen The screen to render to.
 */ 
void event_loop(EventHandler *op, libfhi::Surface *screen)
{
  // Constants.
  static const uint32_t TIMER_MS = TIMER_MILLISECONDS;
  static const int TIMER_MAXFPS = 1000 / TIMER_MS,
	           MAX_SKIP = 4;

  // Statics.
  static uint32_t lasttime = SDL_GetTicks();
  static bool frames_drawn[TIMER_MAXFPS],
	      *frames_ptr = frames_drawn;

  // Variables.
  uint32_t currtime;
  int skip, fps = 0;
  std::ostringstream skipstr;

#ifdef FPS_CAP
  // Sleep loop.
  while(true)
  {
    currtime = SDL_GetTicks() - lasttime;

    // If we have time, sleep.
    if(currtime < TIMER_MS)
    {
      // Timer granularity may cause problems here.
      SDL_Delay(TIMER_MS - currtime);
    }
    // Otherwise check if need to cancel spin.
    else if(currtime >= TIMER_MS)
    {
      break;
    }
  }

  // Iterate for the missed parts.
  for(skip = 0; (currtime >= TIMER_MS); ++skip)
  {
    // If skip is 0, enter true into frame array, otherwise enter false.
    if(skip)
    {
      *frames_ptr = false;
    }
    else
    {
      *frames_ptr = true;
    }
    
    // Increment frames drawn counter.
    if(++frames_ptr >= frames_drawn + TIMER_MAXFPS)
    {
      frames_ptr = frames_drawn;
    }

    // Prevent skipping in excess, only allow up to MAX_SKIP frames without
    // drawing in between.
    if(skip <= MAX_SKIP)
    {
#endif
      SDL_Event event;
  
      while(SDL_PollEvent(&event))
      {
	switch(event.type)
	{
	  case SDL_KEYDOWN:
	    op->keydown(event.key.keysym.sym);
	    break;
	  case SDL_KEYUP:
	    op->keyup(event.key.keysym.sym);
	    break;
	  default:
	    break;
	}
      }

      // Assign tick command.
      op->tick();
#ifdef FPS_CAP
    }

    currtime -= TIMER_MS;
    lasttime += TIMER_MS;
  }
#endif

  // Assign update command.
  screen->lock();
  op->draw(screen);

#ifdef FPS_CAP
  // Count all drawn frames. This is only accurate for successive periods
  // of one second.
  for(int i = 0; (i < TIMER_MAXFPS); ++i)
  {
    fps += frames_drawn[i];
  }

  if(op == Game::instance)
  {
    // Skip was always one greater than the actual number of skipped frames.
    skipstr << "FPS: " << fps << ((skip > MAX_SKIP + 1) ? " (slow)" : "");

    libfhi::Font *font = Menu::get_font();

    libfhi::Surface::draw_text(
	screen->get_w() - 5 - font->get_string_width(skipstr.str().c_str()),
	Menu::get_text_bottom_y(screen),
	Menu::get_color_text(),
	font,
	skipstr.str().c_str());
  }
#endif

  // Flip contents.
  screen->unlock();
  screen->flip();
}

//############################################################################
// Main ######################################################################
//############################################################################

int main(int argc, char **argv)
{
  // Globals.
  static const char *usage = "Usage: absorb [options]\n"
    "Execute the binary only from places where data directory can be found\n"
    "in a relative path data/.\n\n"
    "Command line options with arguments:\n"
    "  --channels, -c      Sound polyphony (default 64)\n"
    "  --resolution, -r    Use resolution ?x?x? (e.x. 640x480x32)\n"
    "  --sndrate, -s       Sound rate (default 44100)\n"
    "\nCommand line options without arguments:\n"
    "  --fullscreen, -f    Fullscreen mode.\n"
    "  --help, -h          Display this help.\n"
    "  --mono, -m          Mono audio (most sounds are mono so can safely be "
    "enabled).\n"
    "  --no-opengl, -n     No opengl, software mode incomplete (only for "
    "testing).\n";
  int screen_w = 640,
      screen_h = 480,
      screen_b = 32,
	  sound_rate = DEFAULT_SOUND_RATE,
	  sound_channels = DEFAULT_SOUND_CHANNELS,
	  sound_buffer = DEFAULT_SOUND_BUFFER,
	  sound_polyphony = DEFAULT_SOUND_POLYPHONY;
  bool fullscreen = false,
       opengl = true;

  // Create options struct.
  libfhi::GetOpt *opt = new libfhi::GetOpt(argc, argv);
  opt->add_option("resolution", true, 'r', "r");
  opt->add_option("fullscreen", false, 'f', "f");
  opt->add_option("help", false, 'h', "h");
  opt->add_option("no-opengl", false, 'n', "n");
  opt->add_option("sndrate", true, 's', "s");
  opt->add_option("mono", false, 'm', "m");
  opt->add_option("channels", true, 'c', "c");

  // Get the options.
  while(true)
  {
    int option_result = opt->get_opt();

    if(option_result == 0)
    {
      break;
    }

    switch(option_result)
    {
      case 'f':
	fullscreen = true;
	break;

      case 'h':
	std::cout << usage;
	exit(0);
	break;

      case 'n':
	opengl = false;
	break;

      case 'r':
	if(sscanf(opt->get_arg(), "%ix%ix%i", &screen_w, &screen_h, &screen_b)
	    != 3)
	{
	  std::cout << "Invalid resolution string: \"" << opt->get_arg() <<
	    "\"\n";
	  screen_w = DEFAULT_SCREEN_W;
	  screen_h = DEFAULT_SCREEN_H;
	  screen_b = DEFAULT_SCREEN_B;
	}
	break;

      case 's':
	{
	  std::istringstream iss(opt->get_arg());
	  iss>>sound_rate;
	  std::cout << "Sound rate set to " << sound_rate << " Hz." <<
	    std::endl;
	}
	break;

      case 'm':
	sound_channels=1;
	std::cout << "Mono sound selected." << std::endl;
	break;

      case 'c':
	{
	  std::istringstream iss(opt->get_arg());
	  iss>>sound_polyphony;
	  std::cout<<"Polyphony: "<<sound_polyphony<<" channels"<<std::endl;
	}
	break;

      case 0:
      case '?':
	// Error message already printed.
	break;

      default:
	exit(1);
	break;
    }
  }

  // Free options struct.
  delete opt;

  // Initialize random number generator.
  srand(static_cast<unsigned int>(time(0)));

  // Initialize libfhi (will initialize SDL and Freetype).
  libfhi::init_libfhi();

  // Set graphics mode.
  if(opengl)
  {
    g_screen = libfhi::glsurface_new(screen_w, screen_h, screen_b,
	(fullscreen ? SDL_FULLSCREEN : 0));
  }
  else
  {
    g_screen = libfhi::set_mode(screen_w, screen_h, screen_b,
	libfhi::Surface::FLAG_ZBUFFER,
	libfhi::sdlSurface::DEFAULT_SDL_FLAGS |
	(fullscreen ? SDL_FULLSCREEN : 0));
  }
  SDL_ShowCursor(SDL_DISABLE);
  g_screen->select();
  g_screen->lock();
  g_screen->clear();
  g_screen->unlock();

  // Initialize SDL_mixer.
  if(Mix_OpenAudio(sound_rate, MIX_DEFAULT_FORMAT, sound_channels,
	sound_buffer)==-1)
  {
    std::cerr<<"SDL_mixer failed: "<<Mix_GetError()<<std::endl;
    return 1;
  }
  Mix_AllocateChannels(sound_polyphony);

  // Screen is initialized, initialize other stuff.
  Options::init();
  Menu::init(g_screen);

  // Change directory to allow future loads.
  Data::chdir_to_datadir();
  
  // Enter event loop.
  g_handler = MainMenu::instance_get();
  while(g_handler)
  {
    event_loop(g_handler, g_screen);
  }

  // Close the challenge singleton. This will also write the high scores.
  Challenge::instance_delete();

  // Perform exit.
  Mix_CloseAudio();
  // TODO: Add canonical database clearing of non-libfhi types here.
  delete g_screen;
  libfhi::quit_libfhi();
  exit(0);
}

//############################################################################
// End #######################################################################
//############################################################################

