#include "data.h"

#include "collisionmap.h"
#include "effect.h"
#include "entityarchtype.h"
#include "particlesystem.h"
#include "sound.h"
#include "weaponmount.h"

// For some reason, there is disparity between MSVS and unix in the matter.
// This fixes it.
#ifdef WIN32
#include <direct.h>
#define chdir _chdir
#else
#include <unistd.h>
#endif

// Define data directory if not already specified.
#ifndef DATADIR
#define DATADIR "/usr/local/share/absorb"
#endif

//############################################################################
// Static data ###############################################################
//############################################################################

bool Data::in_datadir = false;

//############################################################################
// Class #####################################################################
//############################################################################

/** Change to data directory so successive loaders need not know about it.
 * @return False if no data directory found, true otherwise.
 */
void Data::chdir_to_datadir()
{
  if(Data::in_datadir)
  {
    return;
  }

  // Try local data directory.
  if(chdir("data") == 0)
  {
    std::cout << "Opened local data directory \"data/\"\n";
    Data::in_datadir = true;
    return;
  }

  // Try default data directory.
  if(chdir(DATADIR) == 0)
  {
    std::cout << "Opened default data directory \"" << DATADIR << "\"\n";
    Data::in_datadir = true;
    return;
  }

  std::cout << "ERROR: no data directory found.\n";
}

/** Open a collision map.
 * @param str Filename to open from.
 * @return New CollisionMap or NULL on error.
 */
CollisionMap* Data::load_cmap(const char *str)
{
  return Data::load_template<CollisionMap>(str);
}

/** Open an enffect file, save to canonical database.
 * @param str Filename to open from.
 * @return New Effect or NULL on error.
 */
Effect* Data::load_eff(const char *str)
{
  return Data::load_template<Effect>(str);
}

/** Open an entity file (which may in turn open some other files), save to
 * canonical database.
 * @param str Filename to open from.
 * @return New PhysicalModel or NULL on error.
 */
EntityArchtype* Data::load_ent(const char *str)
{
  return Data::load_template<EntityArchtype>(str);
}

/** Open a mesh, save to canonical database.
 * @param str Filename to open from.
 * @return New Mesh or NULL on error.
 */
libfhi::Mesh* Data::load_msh(const char *str)
{
  libfhi::Mesh *ret = libfhi::Mesh::canon_get(str);
  if(ret)
  {
    return ret;
  }

  chdir_to_datadir();

  if(!libfhi::fexists(str))
  {
    std::cerr << "Warning: file not found: \"" << str << "\n";
    return NULL;
  }

  // Load, compile, delete and return.
  libfhi::PreModel *mdl = new libfhi::PreModel(str);
  libfhi::Mesh *msh = mdl->compile(str);
  delete mdl;
  return msh;
}

/** Open a weapon mount (weapon data), save to canonical database.
 * @param str Filename to open from.
 * @return New WeaponMount or NULL on error.
 */
WeaponMount* Data::load_mount(const char *str)
{
  return Data::load_template<WeaponMount>(str);
}

/** Open a particle system, save to canonical database.
 * @param str Filename to open from.
 * @return New ParticleSystem or NULL on error.
 */
ParticleSystem* Data::load_part(const char *str)
{
  return Data::load_template<ParticleSystem>(str);
}

/** Open a sample, save to canonical database.
 * @param str Filename to open from.
 * @return New Sample or NULL on error.
 */
Sample* Data::load_sample(const char *str)
{
  return Data::load_template<Sample>(str);
}

/** Open a new texture.
 * @param str Filename to open from.
 * @return New Texture or NULL on error.
 */
libfhi::Texture* Data::load_texture(const char *str)
{
  return Data::load_template<libfhi::Texture>(str);
}

/** Open a weapon.
 * @param str Filename to open from.
 * @return New Weapon or NULL on error.
 */
Weapon* Data::load_weap(const char *str)
{
  return Data::load_template<Weapon>(str);
}

//############################################################################
// End #######################################################################
//############################################################################

