#include "collisionmap.h"

#include "entity.h"

//############################################################################
// Variables #################################################################
//############################################################################

libfhi::Vector2 CollisionMap::dot_difference;

CollisionMap::CollisionLine
  *CollisionMap::line_lhs = NULL,
  *CollisionMap::line_rhs = NULL;

libfhi::Vector2 CollisionMap::collision_position;

//############################################################################
// Template specializations ##################################################
//############################################################################

/** Tell if two dots are colliding.
 * @param op1 Dot 1.
 * @param dpos Relative difference from op1 to op2.
 * @param op2 Dot 2.
 * @return True if collides, false if no.
 */
template<>
bool CollisionMap::collides_template<CollisionMap::CollisionDot,
  CollisionMap::CollisionDot>
(CollisionDot *op1, const libfhi::Vector2 &dpos, CollisionDot *op2)
{
  CollisionMap::dot_difference = (op2->tp + dpos) - op1->tp;
  float r_2 = op1->r + op2->r;

  return (CollisionMap::dot_difference.length_sqr() <= (r_2 * r_2));
}

/** Tell if a dot and a line are colliding.
 * @param op1 Dot 1.
 * @param dpos Relative difference from op1 to op2.
 * @param op2 Line 2.
 * @return True if collides, false if no.
 */
template<>
bool CollisionMap::collides_template<CollisionMap::CollisionDot,
  CollisionMap::CollisionLine>
(CollisionDot *op1, const libfhi::Vector2 &dpos, CollisionLine *op2)
{
  libfhi::Vector2 from = op1->tp - (op2->tp1 + dpos);

  // Calculate length along line.
  float len = from.product_dot(op2->tdir),
	r_2 = op1->r + op2->r;

  // Outside, calc. distance to last point.
  if(len > op2->len)
  {
    from = op1->tp - (op2->tp2 + dpos);
  }
  // Inside, create point inside.
  else if(len > 0.0f)
  {
    from = (op2->tdir * len) - from;
  }
  if(((from.xf * from.xf) + (from.yf * from.yf)) <= (r_2 * r_2))
  {
    CollisionMap::collision_position = op2->tdir * len + op2->tp1;
    return true;
  }
  return false;
}

/** Tell if line and dot are colliding. Inverse of previous, do not modify
 * this, instead backport possible changes from former.
 * @param op1 Line 1.
 * @param dpos Relative difference from op1 to op2.
 * @param op2 Dot 2.
 * @return True if collides, false if no.
 */
template<>
bool CollisionMap::collides_template<CollisionMap::CollisionLine,
  CollisionMap::CollisionDot>
(CollisionLine *op1, const libfhi::Vector2 &dpos, CollisionDot *op2)
{
  /*std::cout << "Line at " << op1->tp1 << " -> " << op1->tp2 << "\nDot at " <<
    op2->tp << ", R: " << op2->r << "\n";*/

  libfhi::Vector2 from = (op2->tp + dpos) - op1->tp1;

  // Calculate length along line.
  float len = from.product_dot(op1->tdir),
	r_2 = op1->r + op2->r;

  // Outside, calc. distance to last point.
  if(len > op1->len)
  {
    from = (op2->tp + dpos) - op1->tp2;
  }
  // Inside, create point inside.
  else if(len > 0.0f)
  {
    from = (op1->tdir * len) - from;
  }
  if (((from.xf * from.xf) + (from.yf * from.yf)) <= (r_2 * r_2))
  {
    CollisionMap::collision_position = op1->tdir * len + op1->tp1;
    return true;
  }
  return false;
}

/** Tell if two lines are colliding.
 * @param op1 Line 1.
 * @param dpos Relative difference from op1 to op2.
 * @param op2 Line 2.
 * @return True if collides, false if no.
 */
template<>
bool CollisionMap::collides_template<CollisionMap::CollisionLine,
  CollisionMap::CollisionLine>
(CollisionLine *op1, const libfhi::Vector2 &dpos, CollisionLine *op2)
{
  // Solution taken from an article by Paul Brouke in 1989:
  // http://astronomy.swin.edu.au/~pbourke/geometry/lineline2d/
  float x1 = op1->tp1.xf,
	y1 = op1->tp1.yf,
	x2 = op1->tp2.xf,
	y2 = op1->tp2.yf,
	x3 = op2->tp1.xf + dpos.xf,
	y3 = op2->tp1.yf + dpos.yf,
	x4 = op2->tp2.xf + dpos.xf,
	y4 = op2->tp2.yf + dpos.yf;

  // Needed multiple times.
  float x4_x3 = x4 - x3,
        y1_y3 = y1 - y3,
	y4_y3 = y4 - y3,
	x1_x3 = x1 - x3,
	x2_x1 = x2 - x1,
	y2_y1 = y2 - y1;

  // Solve stuff.
  float uatop = x4_x3 * y1_y3 - y4_y3 * x1_x3,
    	uabottom = y4_y3 * x2_x1 - x4_x3 * y2_y1;

  // Check for fpu errors.
  if(fabsf(uabottom) <= FLT_EPSILON)
  {
    return false;
  }

  // Get intersection point.
  float ua = uatop / uabottom,
	xi = x1 + ua * (x2 - x1),
	yi = y1 + ua * (y2 - y1);

  // Algorithm to check if collision point is within a line from:
  // http://www.geog.ubc.ca/courses/klink/gis.notes/ncgia/u32.html
  // David H. Douglas, University of Ottawa
  // David M. Mark, State University of New York at Buffalo
  if((((x1 - xi) * (xi - x2)) >= 0.0f) &&
      (((x3 - xi) * (xi - x4)) >= 0.0f))
  {
    CollisionMap::collision_position.set(xi, yi);
    return true;
  }
  return false;
}

/** Tell if line and dot are colliding. Simple version, no translations.
 * Do not modify this, backport possible changes from translating version.
 * @param op1 Line 1.
 * @param op2 Dot 2.
 * @return True if collides, false if no.
 */
template<>
bool CollisionMap::collides_template<CollisionMap::CollisionLine,
  CollisionMap::CollisionDot>
(CollisionLine *op1, CollisionDot *op2)
{
  libfhi::Vector2 from = op2->tp - op1->tp1;

  // Calculate length along line.
  float len = from.product_dot(op1->tdir),
	r_2 = op1->r + op2->r;

  // Outside, calc. distance to last point.
  if(len > op1->len)
  {
    from = op2->tp - op1->tp2;
  }
  // Inside, create point inside.
  else if(len > 0.0f)
  {
    from = (op1->tdir * len) - from;
  }
  if(((from.xf * from.xf) + (from.yf * from.yf)) <= (r_2 * r_2))
  {
    CollisionMap::collision_position = op1->tdir * len + op1->tp1;
    return true;
  }
  return false;
}

/** Tell if two lines are colliding.
 * Do not modify this, backport possible changes from translating version.
 * @param op1 Line 1.
 * @param dpos Relative difference from op1 to op2.
 * @param op2 Line 2.
 * @return True if collides, false if no.
 */
template<>
bool CollisionMap::collides_template<CollisionMap::CollisionLine,
  CollisionMap::CollisionLine>
(CollisionLine *op1, CollisionLine *op2)
{
  float x1 = op1->tp1.xf,
	y1 = op1->tp1.yf,
	x2 = op1->tp2.xf,
	y2 = op1->tp2.yf,
	x3 = op2->tp1.xf,
	y3 = op2->tp1.yf,
	x4 = op2->tp2.xf,
	y4 = op2->tp2.yf;

  // Needed multiple times.
  float x4_x3 = x4 - x3,
        y1_y3 = y1 - y3,
	y4_y3 = y4 - y3,
	x1_x3 = x1 - x3,
	x2_x1 = x2 - x1,
	y2_y1 = y2 - y1;

  // Solve stuff.
  float uatop = x4_x3 * y1_y3 - y4_y3 * x1_x3,
    	uabottom = y4_y3 * x2_x1 - x4_x3 * y2_y1;

  // Check for fpu errors.
  if(fabsf(uabottom) <= FLT_EPSILON)
  {
    return false;
  }

  // Get intersection point.
  float ua = uatop / uabottom,
	xi = x1 + ua * (x2 - x1),
	yi = y1 + ua * (y2 - y1);

  // Algorithm to check if collision point is within a line from:
  // http://www.geog.ubc.ca/courses/klink/gis.notes/ncgia/u32.html
  // David H. Douglas, University of Ottawa
  // David M. Mark, State University of New York at Buffalo
  if((((x1 - xi) * (xi - x2)) >= 0.0f) &&
      (((x3 - xi) * (xi - x4)) >= 0.0f))
  {
    CollisionMap::collision_position.set(xi, yi);
    return true;
  }
  return false;
}

//############################################################################
// Construction ##############################################################
//############################################################################

/** Default constructor.
 * @param mdl Host model.
 */
CollisionMap::CollisionMap()
  : Attributable(DEFAULT_ATTR), square(NULL), host(NULL),
  faction(FACTION_ENEMY), array_dots(NULL), num_dots(0), array_lines(NULL),
  num_lines(0)
{
  // Do nothing.
}

/** Load constructor for opening new collision maps.
 * @param filename The filename to load from.
 */
CollisionMap::CollisionMap(const char *filename)
  : Attributable(DEFAULT_ATTR), square(NULL), host(NULL),
  faction(FACTION_ENEMY), array_dots(NULL), num_dots(0), array_lines(NULL),
  num_lines(0)
{
  libfhi::ConfigFile filu(filename);

  if(!filu.is_ok())
  {
    std::cout << "ERROR: could not load collision map from \"" << filename <<
      "\"\n";
    return;
  }

  // These last for the whole loop.
  float scale = 1.0f,
	cr = 1.0f,
	sr = 0.0f;

  for(int line = 1; (filu.advance()); ++line)
  {
    // Scale.
    if(filu.has_id_arg("scale", 1))
    {
      scale = filu.get_float(0);
    }
    // Rotate
    else if(filu.has_id_arg("rotate", 1))
    {
      float pi = libfhi::uint2pi(static_cast<uint16_t>(filu.get_int(0)));

      // Set new rotation shite.
      cr = cosf(pi);
      sr = sinf(pi);
    }
    // One line.
    else if(filu.has_id("line"))
    {
      if(filu.get_num_arg() < 5)
      {
	filu.warn("need at least five arguments.");
      }
      else
      {
	// Get radius.
	float r = filu.get_float(0) * scale;

	for(size_t i = 1; (i + 3 < filu.get_num_arg()); i += 2)
	{
	  libfhi::Vector2
	    p1(filu.get_float(i), filu.get_float(i + 1)),
	    p2(filu.get_float(i + 2), filu.get_float(i + 3)),
	    tp1,
	    tp2;

	  tp1.rotate(p1, cr, sr);
	  tp2.rotate(p2, cr, sr);

	  this->add_line(tp1 * scale, tp2 * scale, r);
	}
      }
    }
    // One dot.
    else if(filu.has_id_arg("dot", 3))
    {
      libfhi::Vector2
	p1(filu.get_float(0), filu.get_float(1)),
	tp1;
      tp1.rotate(p1, cr, sr);

      this->add_dot(tp1 * scale, filu.get_float(2) * scale);
    }
    else
    {
      filu.warn_empty();
    }
  }

  this->precalculate();

  std::cout << "Loaded collision map \"" << filename << "\" with D(" <<
    this->num_dots << ") L(" << this->num_lines << ")\n";
}

/** Copy constructor. Note that the base maps should only be calculated
 * once, after this new objects are created by copying them so there is no
 * need for precalculation. All parameters need not be copied as they are
 * recalculated each frame.
 * @param src Source model.
 * @param mdl The host model of this.
 */
CollisionMap::CollisionMap(CollisionMap *src, EntityBase *mdl)
  : Attributable(src->get_attr()), square(NULL), host(mdl), array_dots(NULL),
  array_lines(NULL)
{
  // Copy data from source
  this->relative = src->relative;
  this->longest_radius = src->longest_radius;
  this->num_dots = src->num_dots;
  this->num_lines = src->num_lines;

  // Copy faction from host.
  if(this->host)
  {
    this->faction = host->get_faction();
  }

  // Copy the line and dot data.
  if(this->num_dots)
  {
    this->array_dots = static_cast<CollisionDot*>(
	malloc(sizeof(CollisionDot) * this->num_dots));
    memcpy(this->array_dots, src->array_dots,
	sizeof(CollisionDot) * this->num_dots);
  }
  if(this->num_lines)
  {
    this->array_lines = static_cast<CollisionLine*>(
	malloc(sizeof(CollisionLine) * this->num_lines));
    memcpy(this->array_lines, src->array_lines,
	sizeof(CollisionLine) * this->num_lines);
  }
}

/** Default destructor.
 */
CollisionMap::~CollisionMap()
{
  if(this->array_dots)
  {
    free(this->array_dots);
  }
  if(this->array_lines)
  {
    free(this->array_lines);
  }
}

//############################################################################
// Methods ###################################################################
//############################################################################

/** Add one dot element.
 * @param dotcenter Center of the dot.
 * @param radius Radius.
 */
void CollisionMap::add_dot(const libfhi::Vector2 &dotcenter, float radius)
{
  if(this->array_dots == NULL)
  {
    this->array_dots = static_cast<CollisionDot*>(
	malloc(sizeof(CollisionDot)));
    num_dots = 1;
  }
  else
  {
    this->array_dots = static_cast<CollisionDot*>(realloc(this->array_dots,
	  sizeof(CollisionDot) * (++num_dots)));
  }

  CollisionDot *tgt = this->array_dots + (num_dots - 1);
  tgt->p = dotcenter;
  tgt->r = radius;
}

/** Add one line element.
 * @param dot1 First line point.
 * @param dot2 Second line point.
 * @param radius Line radius.
 */
void CollisionMap::add_line(const libfhi::Vector2 &dot1,
    const libfhi::Vector2 &dot2, float radius)
{
  libfhi::Vector2 direction = dot2 - dot1;
  float len = (dot2 - dot1).length();

  // Check for illegal lines.
  if(len <= 0.0f)
  {
    std::cout << "WARNING: not creating collision line of length 0.\n";
    return;
  }
  // Otherwise can make unit vector.
  direction /= len;

  if(this->array_lines == NULL)
  {
    this->array_lines = static_cast<CollisionLine*>(
	malloc(sizeof(CollisionLine)));
    num_lines = 1;
  }
  else
  {
    this->array_lines = static_cast<CollisionLine*>(realloc(this->array_lines,
	  sizeof(CollisionLine) * (++num_lines)));
  }

  CollisionLine* tgt = this->array_lines + (num_lines - 1);
  tgt->p1 = dot1;
  tgt->p2 = dot2;
  tgt->dir = direction;
  tgt->len = len;
  tgt->r = radius;
}

/** Precalculate the longest radius, set relative_x and relative_y to correct
 * values based on the maximum and miminum values of collision data, then
 * translate all the collision data on this.
 */
void CollisionMap::precalculate()
{
  // Minimum values need some kind of "big" initialization that will be larger
  // than any possible object.
  float xmin = FLT_MAX,
	xmax = -FLT_MAX,
	ymin = FLT_MAX,
	ymax = -FLT_MAX;

  // Get the max and min values of all points.
  for(int i = 0; (i < num_dots); ++i)
  {
    CollisionDot *dot = this->array_dots + i;

    xmin = stdmin(xmin, dot->p.xf - dot->r);
    xmax = stdmax(xmax, dot->p.xf + dot->r);
    ymin = stdmin(ymin, dot->p.yf - dot->r);
    ymax = stdmax(ymax, dot->p.yf + dot->r);
  }

  // Continue with lines.
  for(int i = 0; (i < num_lines); ++i)
  {
    CollisionLine *line = this->array_lines + i;

    xmin = stdmin(stdmin(xmin, line->p1.xf), line->p2.xf);
    xmax = stdmax(stdmax(xmax, line->p1.xf), line->p2.xf);
    ymin = stdmin(stdmin(ymin, line->p1.yf), line->p2.yf);
    ymax = stdmax(stdmax(ymax, line->p1.yf), line->p2.yf);
  }

  // We now know the boundaries. Calculate the center.
  this->relative.xf = (xmax + xmin) / 2.0f;
  this->relative.yf = (ymax + ymin) / 2.0f;
  this->longest_radius = 0.0f;

  // Iterate through all dots, and calculate the maximum distance they may
  // have from the center of this map. Do not translate the dots themselves,
  // we need to keep them in absolute object coordinates (not collision map
  // coordinates).
  for(int i = 0; (i < num_dots); ++i)
  {
    CollisionDot *dot = this->array_dots + i;

    libfhi::Vector2 diff = dot->p - this->relative;

    this->longest_radius = stdmax(this->longest_radius,
	diff.length() + dot->r);
  }

  // Same for lines.
  for(int i = 0; (i < num_lines); ++i)
  {
    CollisionLine *line = this->array_lines + i;

    libfhi::Vector2 diff1 = line->p1 - this->relative,
      diff2 = line->p2 - this->relative;

    this->longest_radius = stdmax(this->longest_radius,
	stdmax(diff1.length(), diff2.length()));
  }
}

/** Section is used to clip large collision maps into multiple smaller ones
 * to ease the creation of collision data. The algothm:
 * 1) Select one dot or line.
 * 2) If the dot or line can be fully implanted in one map already in the save
 * list, do it.
 * 3) Otherwise if the dot or line can be fully implanted into one new
 * collision map, do it.
 * 4) Otherwise, create a new collision map and implant as much of the dot or
 * line there as possible.
 * The result is in no way optimal and the process itself takes way too long a
 * time, but happily this is not for real-time use.
 * @param dst Target std::vector.
 */
void CollisionMap::section(std::vector<CollisionMap*> *dst)
{
  //std::cout << "Dot count: " << this->num_dots << "\n";

  // Section all dots.
  for(int i = 0; (i < this->num_dots); ++i)
  {
    CollisionDot *dot = this->array_dots + i;

    // If the size of the dot exceeds maximum, warn and truncate.
    if(dot->r > 0.25f)
    {
      std::cerr << "Warning: collision dot of radius " << dot->r <<
	"found, truncating.\n";
      dot->r = 0.25f;
    }

    // Set this to true when we find what we are looking for.
    bool endsearch = false;

    // Loop the existing collision maps and check for successful fit.
    for(std::vector<CollisionMap*>::iterator iter = dst->begin(),
	iterend = dst->end(); (iter != iterend); ++iter)
    {
      if(((*iter)->get_attr() == this->get_attr()) &&
	  ((((*iter)->relative - dot->p).length() + dot->r) <= 0.25f))
      {
	(*iter)->add_dot(dot->p, dot->r);
	(*iter)->precalculate();
	endsearch = true;
	break;
      }
    }

    if(endsearch)
    {
      continue;
    }

    // Was not found, implant as a new collision map.
    CollisionMap *newmap = new CollisionMap();
    newmap->set_attr(this->get_attr());
    newmap->add_dot(dot->p, dot->r);
    newmap->precalculate();
    dst->push_back(newmap);
  }

  // Section all lines.
  for(int i = 0; (i < this->num_lines); ++i)
  {
    CollisionLine *line = this->array_lines + i;

    // Set this to true when we find what we are looking for.
    bool endsearch = false;

    // Loop the existing collision maps and check for successful fit.
    for(std::vector<CollisionMap*>::iterator iter = dst->begin(),
	iterend = dst->end(); (iter != iterend); ++iter)
    {
      if(((*iter)->get_attr() == this->get_attr()) &&
	  ((((*iter)->relative - line->p1).length() + line->r) <= 0.25f) &&
	  ((((*iter)->relative - line->p2).length() + line->r) <= 0.25f))
      {
	(*iter)->add_line(line->p1, line->p2, line->r);
	(*iter)->precalculate();
	endsearch = true;
	break;
      }
    }

    if(endsearch)
    {
      continue;
    }

    // Check if we can implant this line as a whole new collision map.
    if(line->len + 2.0f * line->r <= 0.5f)
    {
      CollisionMap *newmap = new CollisionMap();
      newmap->set_attr(this->get_attr());
      newmap->add_line(line->p1, line->p2, line->r);
      newmap->precalculate();
      dst->push_back(newmap);
      continue;
    }

    // Can not. Split the line.
    libfhi::Vector2 middle = line->dir;
    middle.unit();
    middle = middle * (0.5f - 2 * line->r) + line->p1;

    CollisionMap *newmap = new CollisionMap();
    newmap->set_attr(this->get_attr());
    newmap->add_line(line->p1, middle, line->r);
    newmap->precalculate();
    dst->push_back(newmap);

    // Rest goes here.
    this->add_line(middle, line->p2, line->r);
  }
}

/** Transform all collision elements inside this map.
 * @param hpos Host object position.
 * @param cr Cosine of rotation.
 * @param sr Sine of rotation.
 */
void CollisionMap::transform(const libfhi::Vector2 &hpos, float cr, float sr)
{
  // Calculate the new absolute center point of this collision map
  // (will be used to calculate the lower left corner).
  this->center.rotate_translate(this->relative, cr, sr, hpos);

  // Rotate all dots. The dots are already in relative object space, transform
  // them to absolute space in the same manner as the map center.
  if(this->array_dots)
  {
    int i = this->num_dots;
    CollisionDot *dot = this->array_dots;
    do {
      dot->tp.rotate_translate(dot->p, cr, sr, hpos);

      ++dot;
    } while(--i);
  }

  // Rotate all lines. Works like dots.
  if(this->array_lines)
  {
    int i = this->num_lines;
    CollisionLine *line = this->array_lines;
    do {
      line->tp1.rotate_translate(line->p1, cr, sr, hpos);
      line->tp2.rotate_translate(line->p2, cr, sr, hpos);
      line->tdir.rotate(line->dir, cr, sr);

      // Line normals ALWAYS point right from their direction. Because of
      // this, you will have to define lines in counter-clockwise fashion, so
      // that outside of the object is on the right side of any given line.
      line->tnor.xf = line->tdir.yf;
      line->tnor.yf = -line->tdir.xf;

      ++line;
    } while(--i);
  }
}

/** Translate all collision elements inside this map. Similar to transform,
 * but does not perform rotation. Note, that in this point, the center no
 * longer needs translating, as it is only used for fetching the lower left
 * corner.
 * @param diff Difference in positions.
 */
void CollisionMap::translate(const libfhi::Vector2 &diff)
{
  // Translate dots.
  if(this->array_dots)
  {
    int i = this->num_dots;
    CollisionDot *dot = this->array_dots;
    do {
      //std::cout << "Dot " << i << "\n";
      dot->tp += diff;
      ++dot;
    } while(--i);
  }

  // Translate lines.
  if(this->array_lines)
  {
    int i = this->num_lines;
    CollisionLine *line = this->array_lines;
    do {
      //std::cout << "Line " << i << "\n";
      line->tp1 += diff;
      line->tp2 += diff;
      ++line;
    } while(--i);
  }
}

//############################################################################
// Sub-collision methods #####################################################
//############################################################################

/** Collide one dot against any number of dots in target map.
 * @param dot Tested dot.
 * @param dpos Distance to second map.
 * @param op Second map.
 * @return True if collision happened. False otherwise.
 */
bool CollisionMap::collision_batch_dot_dots(CollisionDot *dot,
    const libfhi::Vector2 &dpos, CollisionMap *op)
{
  // Against dots of target.
  int i = op->num_dots;
  if(i)
  {
    CollisionDot *tgt_dot = op->array_dots;
    do {
      if(collides_template(dot, dpos, tgt_dot))
      {
	CollisionMap::line_lhs = NULL;
	CollisionMap::line_rhs = NULL;
	return true;
      }
      ++tgt_dot;
    } while(--i);
  }

  return false;
}

/** Collide one dot against any number of lines in target map.
 * @param dot Tested dot.
 * @param dpos Distance to second map.
 * @param op Second map.
 * @return True if collision happened. False otherwise.
 */
bool CollisionMap::collision_batch_dot_lines(CollisionDot *dot,
    const libfhi::Vector2 &dpos, CollisionMap *op)
{
  // Against lines of target.
  int i = op->num_lines;
  if(i)
  {
    CollisionLine *tgt_line = op->array_lines;
    do {
      if(collides_template(dot, dpos, tgt_line))
      {
	CollisionMap::line_lhs = NULL;
	CollisionMap::line_rhs = tgt_line;
	return true;
      }
      ++tgt_line;
    } while(--i);
  }

  return false;
}

/** Collide one line against any number of dots in target map.
 * @param line Tested line.
 * @param dpos Distance to second map.
 * @param op Second map.
 * @return True if collision happened. False otherwise.
 */
bool CollisionMap::collision_batch_line_dots(CollisionLine *line,
    const libfhi::Vector2 &dpos, CollisionMap *op)
{
  // Against dots of target.
  int i = op->num_dots;
  if(i)
  {
    CollisionDot *tgt_dot = op->array_dots;
    do {
      if(collides_template(line, dpos, tgt_dot))
      {
	CollisionMap::line_lhs = line;
	CollisionMap::line_rhs = NULL;
	return true;
      }
      ++tgt_dot;
    } while(--i);
  }

  return false;
}

/** Collide one dot against any number of lines in target map.
 * @param dot Tested line.
 * @param dpos Distance to second map.
 * @param op Second map.
 * @return True if collision happened. False otherwise.
 */
bool CollisionMap::collision_batch_line_lines(CollisionLine *line,
    const libfhi::Vector2 &dpos, CollisionMap *op)
{
  // Against lines of target.
  int i = op->num_lines;
  if(i)
  {
    CollisionLine *tgt_line = op->array_lines;
    do {
      if(collides_template(line, dpos, tgt_line))
      {
	CollisionMap::line_lhs = line;
	CollisionMap::line_rhs = tgt_line;
	return true;
      }
      ++tgt_line;
    } while(--i);
  }

  return false;
}

//############################################################################
// Collision methods #########################################################
//############################################################################

/** Collide with another map. Since these functions do not happen only between
 * collision lines, they save the collision lines into the global variables
 * upon collision. The globals will be set to null if the corresponding
 * comparator was not a line.
 * @param op1 Source map.
 * @param dpos Relative x difference to the square of target.
 * @param op2 Map to test against.
 * @return True if collision happened, false if not.
 */
bool CollisionMap::collides_freeform_freeform(CollisionMap *op1,
    const libfhi::Vector2 &dpos, CollisionMap *op2)
{
  // Do not collide with own faction.
  if(!is_faction_collidable(op1->get_faction(), op2->get_faction()))
  {
    return false;
  }

  // Dots of this.
  int i = op1->num_dots;

  if(i)
  {
    CollisionDot *dot = op1->array_dots;

    do {
      if(CollisionMap::collision_batch_dot_dots(dot, dpos, op2))
      {
	return true;
      }

      if(CollisionMap::collision_batch_dot_lines(dot, dpos, op2))
      {
	return true;
      }
  
      ++dot;
    } while(--i);
  }

  // Lines of this.
  i = op1->num_lines;

  if(i)
  {
    CollisionLine *line = op1->array_lines;

    do {
      if(CollisionMap::collision_batch_line_dots(line, dpos, op2))
      {
	return true;
      }

      if(CollisionMap::collision_batch_line_lines(line, dpos, op2))
      {
	return true;
      }

      ++line;
    } while(--i);
  }

  return false;
}

/** Collide all lines of this against target freeform. This is used by
 * terrain, basically, chack all lines of this against anything of target,
 * no need for translations.
 * @param op1 Source map.
 * @param op2 Map to test against.
 * @return The line against which collision happened or NULL if not.
 */
CollisionMap::CollisionLine*
CollisionMap::collides_lines_freeform(CollisionMap *op1,
    CollisionMap *op2)
{
  // Do not collide with own faction.
  if(!is_faction_collidable(op1->get_faction(), op2->get_faction()))
  {
    return NULL;
  }

  int i = op1->num_lines;
  if(!i)
  {
    return NULL;
  }

  CollisionLine *line = op1->array_lines;
  do {
    int j = op2->num_dots;
    if(j)
    {
      CollisionDot *tgt_dot = op2->array_dots;
      do {
	if(CollisionMap::collides_template(line, tgt_dot))
	{
	  return line;
	}
	++tgt_dot;
      } while(--j);
    }

    j = op2->num_lines;
    if(j)
    {
      CollisionLine *tgt_line = op2->array_lines;
      do {
	if(CollisionMap::collides_template(line, tgt_line))
	{
	  return line;
	}
	++tgt_line;
      } while(--j);
    }

    ++line;
  } while(--i);

  // End.
  return NULL;
}

/** Collide all lines of this agains single target dot. This is used by
 * terrain in collisions against bullets. No need for translations.
 * @param op1 Source map.
 * @param op2 Map to test against.
 * @return The line against which collision happened or NULL if not.
 */
CollisionMap::CollisionLine*
CollisionMap::collides_lines_single_dot(CollisionMap *op1,
    CollisionMap *op2)
{
  // Do not collide with own faction.
  if(!is_faction_collidable(op1->get_faction(), op2->get_faction()))
  {
    return NULL;
  }

  int i = op1->num_lines;
  if(!i)
  {
    return NULL;
  }

  CollisionLine *line = op1->array_lines;
  do {
    // Requires the single dot in opponent. Will crash otherwise.
    if(CollisionMap::collides_template(line, op2->array_dots))
    {
      return line;
    }
    ++line;
  } while(--i);

  // End.
  return NULL;
}

/** Collide all lines of this agains single target line. This is used by
 * terrain in collisions against instant weapons. No need for translations.
 * @param op1 Source map.
 * @param op2 Map to test against.
 * @return The line against which collision happened or NULL if not.
 */
CollisionMap::CollisionLine*
CollisionMap::collides_lines_single_line(CollisionMap *op1,
    CollisionMap *op2)
{
  // Do not collide with own faction.
  if(!is_faction_collidable(op1->get_faction(), op2->get_faction()))
  {
    return NULL;
  }

  int i = op1->num_lines;
  if(!i)
  {
    return NULL;
  }

  CollisionLine *line = op1->array_lines;
  do {
    // Requires the single dot in opponent. Will crash otherwise.
    if(CollisionMap::collides_template(line, op2->array_lines))
    {
      return line;
    }
    ++line;
  } while(--i);

  // End.
  return NULL;
}

/** Collide the single dot in this against a freeform map.
 * @param op1 Source map.
 * @param dpos Relative x difference to the square of target.
 * @param op2 Map to test against.
 * @return True if collides, false if not.
 */
bool CollisionMap::collides_single_dot_freeform(CollisionMap *op1,
    const libfhi::Vector2 &dpos, CollisionMap *op2)
{
  // Do not collide with own faction.
  if(!is_faction_collidable(op1->get_faction(), op2->get_faction()))
  {
    return false;
  }

  return
    (CollisionMap::collision_batch_dot_dots(op1->array_dots, dpos, op2) ||
     CollisionMap::collision_batch_dot_lines(op1->array_dots, dpos, op2));
}

/** Collide the single line in this against a freeform map.
 * @param op1 Source map.
 * @param dpos Relative x difference to the square of target.
 * @param op2 Map to test against.
 * @return True if collides, false if not.
 */
bool CollisionMap::collides_single_line_freeform(CollisionMap *op1,
    const libfhi::Vector2 &dpos, CollisionMap *op2)
{
  // Do not collide with own faction.
  if(!is_faction_collidable(op1->get_faction(), op2->get_faction()))
  {
    return false;
  }

  return
    (CollisionMap::collision_batch_line_dots(op1->array_lines, dpos, op2) ||
     CollisionMap::collision_batch_line_lines(op1->array_lines, dpos, op2));
}

//############################################################################
// End #######################################################################
//############################################################################

