/************************************************************************/
/*                                                                      */
/* Include this file in your dll code when writing your own fx dll.     */
/* It makes a couple of important/useful Hairy Harry functions          */
/* available.                                                           */
/* Note that these functions are simply helpers, none of them have      */
/* to be used. This means that e.g. for rendering, you can just simply  */
/* issue OpenGL commands and do everything yourself (just make sure     */
/* to clean up the states correctly! ;) ).                              */
/*                                                                      */
/* NOTE:                                                                */
/* All 'filename' parameters imply relative paths from the current      */
/* directory, which is the directory of harry.exe.                      */
/*                                                                      */
/* You should leave this file unchanged.                                */
/*                                                                      */
/************************************************************************/

#ifndef FXDLL_INCLUDED
#define FXDLL_INCLUDED

/************************************************************************/
/* Some misc. defines/declarations                                      */
/************************************************************************/
// The game's screen resolution. (0/0) is the top left corner.
#define RES_X           1024
#define RES_Y            768

// Rect data type (for the fonts)
struct Rect {
    Rect() : top(0),bottom(0),left(0),right(0)
    {}
    Rect(float _top, float _bottom, float _left, float _right)
        : top(_top), bottom(_bottom), left(_left), right(_right)
    {}
    float   top,bottom,left,right;
};


//
// IMAGE FUNCTIONS
//
/************************************************************************
Load an image from a file. Many common file formats are supported
(DevIL is used for loading).
Parms:      filename:               Filename of the Image
Returns:    Handle to image. Zero if the image could not be loaded.
/************************************************************************/
typedef unsigned (*fp_image_load)(const char* filename);

/************************************************************************
Release the image.
Parms:      h:                      Handle to the image (from image_load)
Returns:    -
/************************************************************************/
typedef void (*fp_image_free)(unsigned h);

/************************************************************************
Get information on an image.
Parms:      h:                      Handle to the image
            widht, height:          Returned width/height of the image
            format:                 Returned image format: 0=RGB,1=RGBA,2=BGR,3=BGRA,255=unknown
Returns:    -
/************************************************************************/
typedef void (*fp_image_getinfo)(unsigned h, int *width, int *height, int *format);

/************************************************************************
Get a pointer to image data. This pointer can be used e.g. for generating
OpenGL textures. Do not free this pointer or write to the data.
Parms:      h:                      Handle to the image
Returns:    Pointer to the image data
/************************************************************************/
typedef unsigned char* (*fp_image_data)(unsigned h);



//
// SPRITE FUNCTIONS
//
/************************************************************************
Load a sprite from an image file.
Parms:      filename:               Filename of the image file
            transparent:            true if transparency is needed, false otherwise
            t_r,t_g,t_b:            Transparent color (ignored if transparent==false)
Returns:    Handle to the sprite. Zero if the sprite could not be loaded.
/************************************************************************/
typedef unsigned (*fp_sprite_load)(const char* filename, bool transparent,
    unsigned char t_r, unsigned char t_g, unsigned char t_b);

/************************************************************************
Release the sprite.
Parms:      h:                      Handle to the sprite (from sprite_load)
Returns:    -
/************************************************************************/
typedef void (*fp_sprite_free)(unsigned h);

/************************************************************************
Render a sprite.
Parms:      h:                      Handle to the sprite
            x,y:                    Pixel position of the top left sprite corner
            col_[r,g,b]:            Sprite overlay (multiply) color
            angle:                  Rotation angle, in degrees, counterclockwise
            scalex,scaley:          Horizontal/vertical scale factors
            blend_fact:             0-1, alpha blending parameter
            flip:                   bit 0 set: flip horizontal, bit 1 set: flip vertical
Returns:    -
/************************************************************************/
typedef void (*fp_sprite_render)(unsigned h, int x, int y,
    unsigned char col_r, unsigned char col_g, unsigned char col_b,
    float angle, float scalex, float scaley, float blend_fact, int flip);

/************************************************************************
Get Information on a sprite.
Parms:      h:                      Handle to the sprite
            width,height:           Width and heigth of the sprite, filled
                                    by the function.
Returns:    -
/************************************************************************/
typedef void (*fp_sprite_getinfo)(unsigned h, int *width, int *height);



//
// FONT FUNCTIONS
//
/************************************************************************
Load a font from an image file. Note that in every theme, at least one
font is loaded per default. The handle of this font is zero and can be
used for rendering without having to manually load a font.
Parms:      filename:               Name of the image file
            rects:                  Rect data for each character (see font_gen_char_rects)
Returns:    Handle to the font. Zero if the font could not be loaded.
/************************************************************************/
typedef unsigned (*fp_font_load)(const char* filename, const Rect rects[256]);

/************************************************************************
Release a font. The theme's main font (handle 0) can not be released.
Parms:      h:                      Handle to the font
Returns:    -
/************************************************************************/
typedef void (*fp_font_free)(unsigned h);

/************************************************************************
Render a piece of text.
Parms:      h:                      Handle to the font (0=theme's main font)
            x,y:                    Pixel position of the text
            size:                   Font size. 1.0 = original size
            text:                   Text to be written
            col_[r,g,b]:            Color of the text
Returns:    -
/************************************************************************/
typedef void (*fp_font_render)(unsigned h, int x, int y, float size, const char *text,
    unsigned char col_r, unsigned char col_g, unsigned char col_b);

/************************************************************************
Generate rect data for font loading. Useful if all chars have the same size.
The rects for which there are no entries in the character_string are left
untouched, this way the function can be called multiple times, e.g. if there
are multiple lines of chars in the image.
Parms:      [x,y]_offset:           Offset in the image
            char_width:             Width of the chars
            char_height:            Height of the chars
            rects:                  The rect array to be filled
            character_string:       String containing the chars to be processed
Returns:    -
/************************************************************************/
typedef void (*fp_font_gen_char_rects)(float x_offset, float y_offset, float char_width, 
    float char_height, Rect rects[256], const char* character_string );

/************************************************************************
Compute the space a text will take up on the screen.
Parms:      h:                      Handle to the font (0=theme's main font)
            width,height:           Results of the operation (each can be NULL if not needed)
            size:                   Size of the text
            text:                   The text to be tested
Returns:    -
/************************************************************************/
typedef void (*fp_font_get_text_extent)(unsigned h, int *width, int *height,
    float size, const char* text);



//
// SOUND FUNCTIONS
//
/************************************************************************
Load a sound from a file (wav,ogg,mp3).
Parms:      filename:               Name of the sound file
Returns:    Handle to the sound. Zero if the sound could not be loaded.
/************************************************************************/
typedef unsigned (*fp_sound_load)(const char* filename);

/************************************************************************
Release a sound.
Parms:      h:                      Handle to the sound
Returns:    -
/************************************************************************/
typedef void (*fp_sound_free)(unsigned h);

/************************************************************************
Play a sound.
Parms:      h:                      Handle to the sound
Returns:    -
/************************************************************************/
typedef void (*fp_sound_play)(unsigned h);



//
// KEYBOARD/MOUSE QUERYING
//
/************************************************************************
Get whether a specific key is down. The key can be a keyboard key or
mouse button.
Parms:      ctrl:                   key constant to be queried. see dll_common.h
                                    for a list of possible values.
Returns:    true if the specified key is down, false otherwise.
/************************************************************************/
typedef bool (*fp_key_down)(int ctrl);

/************************************************************************
Get whether a specific key has been "pressed", that is, the key is reported
down only with the specified repeat and delay rate. If both are zero, 
the key is reported exactly once until it has been released and pressed again.
Parms:      ctrl:                   like key_down
            delay:                  milliseconds until the second keypress is reported
            repeat:                 milliseconds until subsequent keypresses are reported
Returns:    true if the specified key has been pressed, false otherwise.
/************************************************************************/
typedef bool (*fp_key_pressed)(int ctrl,double delay, double repeat);

/************************************************************************
Get mouse movement, relative and/or absolute.
The parameters are pointers that are filled by the function. If NULL
is passed for a parameter, the parameter is ignored.
Parms:      x:                      mouse x position
            y:                      mouse y position
            dx:                     mouse x change since last call
            dy:                     mouse y change since last call
Returns:    -
/************************************************************************/
typedef void (*fp_get_mouse_move)(float *x, float *y, float *dx, float *dy);


//
// MISC. FUNCTIONS
//
/************************************************************************
Log to the Game's logfile(s).
Parms:      type:                   0=standard, 1=warning, 2=error
            s:                      Format string (like printf)
Returns:    -
/************************************************************************/
typedef void (*fp_hhlog)(int type, const char *s, ...);

/************************************************************************
Get a (per theme constant) physics parameter.
Parms:      pname:      Name of the parameter. This can be any of the
                        physics parameters in the theme.cfg file. If a
                        (valid) parameter is not found in theme.cfg, it's
                        default value is used.
            result:     The parameter value.
Returns:    true if result has been filled and pname was valid, false
            if parameter pname could not be found.
/************************************************************************/
typedef bool (*fp_get_physparm)(const char* pname, float *result);



/************************************************************************/
/************************************************************************/
/* Function pointers                                                    */
/************************************************************************/
/************************************************************************/
extern fp_image_load                    image_load;
extern fp_image_free                    image_free;
extern fp_image_getinfo                 image_getinfo;
extern fp_image_data                    image_data;
extern fp_sprite_load                   sprite_load;
extern fp_sprite_free                   sprite_free;
extern fp_sprite_render                 sprite_render;
extern fp_sprite_getinfo                sprite_getinfo;
extern fp_font_load                     font_load;
extern fp_font_free                     font_free;
extern fp_font_render                   font_render;
extern fp_font_gen_char_rects           font_gen_char_rects;
extern fp_font_get_text_extent          font_get_text_extent;
extern fp_sound_load                    sound_load;
extern fp_sound_free                    sound_free;
extern fp_sound_play                    sound_play;
extern fp_key_down                      key_down;
extern fp_key_pressed                   key_pressed;
extern fp_get_mouse_move                get_mouse_move;
extern fp_hhlog                         hhlog;
extern fp_get_physparm                  get_physparm;

// DLL Export macro for Microsoft C/C++. For different
// compilers, this most likely needs to be changed.
#define DLL_EXPORT extern "C" __declspec(dllexport)

#endif