// created by eL http://eL.zde.cz eL@email.cz
// vec2, vec3
// simple vector, matrix algebra, openGL compatible matrix
// last update:: 21.3.2003

#ifndef __ALGEBRA_HEADER__
#define __ALGEBRA_HEADER__

#include <iostream>

using namespace std;

#  ifndef M_PI
#    define M_PI            3.14159265358979323846
#    define M_PI_2          1.57079632679489661923
#    define M_PI_4          0.78539816339744830962
#    define M_1_PI          0.31830988618379067154
#    define M_2_PI          0.63661977236758134308
#  endif // !M_PI

#  ifndef M_SQRT2
#    define M_SQRT2         1.41421356237309504880
#    define M_SQRT1_2       0.70710678118654752440
#  endif // !M_SQRT2

// random functions

inline float frnd () { return (float)rand()/RAND_MAX; } //returns float <0,1)
inline int irnd(int max) { return rand()%max; } //returns int <0,max)

// sign func

inline float fsign (float x) 
{
    if (x==0.0) return 0.0;
    if (x>0.0) return 1.0;
    return -1.0;
}

inline void fswap (float &a, float &b)
{
    float t = a;
    a=b;
    b=t;
}

inline float lerp(float f, const float& v1, const float& v2) { return v1 + ((v2 - v1) * f); }

#define DEG_TO_RAD(deg) ((float)(deg)*M_PI/180)
#define RAD_TO_DEG(rad) ((float)(rad)*(180.0f / M_PI))

// 2d vector math

class vec2
{
	public:
		float n[2];
		
		vec2();
		vec2(const vec2&);
		vec2(float, float);
		
		//array access
		float& operator [](int);
		const float& operator [](int) const;
		
		//assignment
		vec2& assign(float, float);
		vec2& operator =(const vec2&);
		
		//math operators
		vec2& operator +=(const vec2&);
		vec2& operator -=(const vec2&);
		vec2& operator *=(float);
		vec2& operator /=(float);
		
		vec2 operator +(const vec2&) const;
		vec2 operator -(const vec2&) const;
		vec2 operator -() const;
		vec2 operator *(float) const;
		vec2 operator /(float) const;
		
		friend vec2 operator *(float, const vec2&);
		
		bool operator ==(const vec2&) const;
		bool operator !=(const vec2&) const;
		
		//operations
		float length() const;
		float lengthSquared() const;
		vec2& normalize();
		
		vec2& rotate (float);
		
        //misc and usefull
		void copyTo(float [2]) const;
		//float * copyFrom ();
		
        //friends
		friend float distance(const vec2&, const vec2&);
		friend float distanceSquared(const vec2&, const vec2&);
		friend float dot(const vec2&, const vec2&);
		friend vec2 lerp(float, const vec2&, const vec2&);
		
		// output
		friend ostream & operator << ( ostream &, const vec2 & );
};

//3d vector

class vec3
{
	public:
		float n[3];
		
		vec3();
		vec3(const vec3&);
		vec3(float, float, float);
		vec3(float, float);           //z==0.0
		
		//array access
		float& operator [](int);
		const float& operator [](int) const;
		
		//assignment
		vec3& assign(float, float, float);
		vec3& operator =(const vec3&);
		vec3& operator =(const vec2&);
		
		//math operators
		vec3& operator +=(const vec3&);
		vec3& operator -=(const vec3&);
		vec3& operator *=(float);
		vec3& operator /=(float);
		
		vec3 operator +(const vec3&) const;
		vec3 operator -(const vec3&) const;
		vec3 operator -() const;
		vec3 operator *(float) const;
		vec3 operator /(float) const;
		
		friend vec3 operator *(float, const vec3&);
		
		bool operator ==(const vec3&) const;
		bool operator !=(const vec3&) const;
		
		//operations
		float length() const;
		float lengthSquared() const;
		vec3& normalize();
		
		vec3& rotateX (float);
		vec3& rotateY (float);
		vec3& rotateZ (float);
		
        //misc and usefull
		void copyTo(float [3]) const;
		//float * copyFrom ();
		
        //friends
		friend vec3 cross(const vec3&, const vec3&);
		friend float distance(const vec3&, const vec3&);
		friend float distanceSquared(const vec3&, const vec3&);
		friend float dot(const vec3&, const vec3&);
		friend vec3 lerp(float, const vec3&, const vec3&);
		
		// output
		friend ostream & operator << ( ostream &, const vec3 & );
		friend istream & operator >> ( istream &, vec3 & );
};


/*
OpengGL compatible matrix:

float[16]
[0 4 8  12]
[1 5 9  13]
[2 6 10 14]
[3 7 11 15]

[rightX upX outX    X]
[rightY upY outY    Y]
[rightZ upZ outZ    Z]
[0      0   0       1]

*/

// CONSTRUCTORS

inline vec2::vec2()
{
    n[0] = n[1] = 0;
}

inline vec2::vec2(const vec2& v)
{
    n[0] = v.n[0]; n[1] = v.n[1];
}

inline vec2::vec2(float x, float y)
{
    n[0] = x; n[1] = y;
}

// ARRAY ACCESS

inline float& vec2::operator [](int i) 
{
    //assert(i == 0 || i == 1);
    return n[i];
}

inline const float& vec2::operator [](int i) const
{
    //assert(i == 0 || i == 1);
    return n[i];
}

// ASSIGNMENT

inline vec2& vec2::assign(float x, float y)
{
    n[0] = x; n[1] = y;
    return *this;
}

inline vec2& vec2::operator =(const vec2& v)
{
    n[0] = v[0]; n[1] = v[1];
    return *this;
}

// MATH OPERATORS

inline vec2& vec2::operator +=(const vec2& v)
{
    n[0] += v[0]; n[1] += v[1];
    return *this;
}

inline vec2& vec2::operator -=(const vec2& v)
{
    n[0] -= v[0]; n[1] -= v[1]; 
    return *this;
}

inline vec2& vec2::operator *=(float c)
{
    n[0] *= c; n[1] *= c;
    return *this;
}

inline vec2& vec2::operator /=(float c)
{
    //assert(!(c==0.0));
    if (c==0.0) return *this;
    n[0] /= c; n[1] /= c;
    return *this;
}

inline vec2 vec2::operator +(const vec2& v) const
{
    return vec2(n[0] + v[0], n[1] + v[1]);
}

inline vec2 vec2::operator -(const vec2& v) const
{
    return vec2(n[0] - v[0], n[1] - v[1]);
}

inline vec2 vec2::operator -() const
{
    return vec2(-n[0], -n[1]);
}

inline vec2 vec2::operator *(float c) const
{
    return vec2(n[0] * c, n[1] * c);
}

inline vec2 vec2::operator /(float c) const
{
    //assert(!(c==0.0));
    if (c==0.0) return vec2(0,0);
    return vec2(n[0] / c, n[1] / c);
}


inline bool vec2::operator ==(const vec2& v) const
{
    return ((n[0]==v[0]) && (n[1]==v[1]));
}

inline bool vec2::operator !=(const vec2& v) const
{
    return (!(*this == v));
}

// OPERATIONS


inline float vec2::length() const
{
    return sqrt((n[0]*n[0]) + (n[1]*n[1]));
}

inline float vec2::lengthSquared() const
{
    return (n[0]*n[0]) + (n[1]*n[1]);
}

inline vec2& vec2::normalize()
{
    float len = length();
    //assert(!(len==0.0));
    if (len==0.0) return *this;
    *this /= len;
    return *this;
}

inline void vec2::copyTo(float f[2]) const
{
    f[0] = n[0]; f[1] = n[1]; 
}

/*
inline float * vec2::copyFrom ()
{
    return n;
}
*/

inline vec2& vec2 :: rotate(float amnt)
{
    float s = sin(amnt);
    float c = cos(amnt);
    float x = n[0];
    float y = n[1];
  
    n[0] = (x * c) - (y * s);
    n[1] = (y * c) + (x * s);
    
    return *this;
}

inline float distance(const vec2& v1, const vec2& v2)
{
    return sqrt( (v1[0]-v2[0])*(v1[0]-v2[0]) 
                +(v1[1]-v2[1])*(v1[1]-v2[1]));
}

inline float distanceSquared(const vec2& v1, const vec2& v2)
{
    return  (v1[0] - v2[0])*(v1[0] - v2[0])
           +(v1[1] - v2[1])*(v1[1] - v2[1]);
}

inline float dot(const vec2& v1, const vec2& v2)
{
    return v1[0] * v2[0] + v1[1] * v2[1];
}

inline vec2 lerp(float f, const vec2& v1, const vec2& v2)
{
    return v1 + ((v2 - v1) * f);
}

// OUTPUT

inline ostream & operator << ( ostream& os, const vec2& v)
{
    os << v[0] << " " << v[1] << " ";
    return os;
}

inline vec2 operator *(float c, const vec2& v)
{
    return vec2(c * v[0], c * v[1]);
}


// CONSTRUCTORS

inline vec3::vec3()
{
    n[0] = n[1] = n[2] = 0;
}

inline vec3::vec3(const vec3& v)
{
    n[0] = v.n[0]; n[1] = v.n[1]; n[2] = v.n[2];
}

inline vec3::vec3(float x, float y, float z)
{
    n[0] = x; n[1] = y; n[2] = z;
}

inline vec3 :: vec3 (float x, float y)
{
    n[0]=x; n[1]=y; n[2]=0.0;
}

// ARRAY ACCESS

inline float& vec3::operator [](int i) 
{
    //assert(i == 0 || i == 1 || i == 2);
    return n[i];
}

inline const float& vec3::operator [](int i) const
{
    //assert(i == 0 || i == 1 || i == 2);
    return n[i];
}

// ASSIGNMENT

inline vec3& vec3::assign(float x, float y, float z)
{
    n[0] = x; n[1] = y; n[2] = z;
    return *this;
}

inline vec3& vec3::operator =(const vec3& v)
{
    n[0] = v[0]; n[1] = v[1]; n[2] = v[2];
    return *this;
}

inline vec3& vec3::operator =(const vec2& v)
{
    n[0]=v[0]; n[1]=v[1]; n[2]=0.0;
    return *this;
}

// MATH OPERATORS

inline vec3& vec3::operator +=(const vec3& v)
{
    n[0] += v[0]; n[1] += v[1]; n[2] += v[2];
    return *this;
}

inline vec3& vec3::operator -=(const vec3& v)
{
    n[0] -= v[0]; n[1] -= v[1]; n[2] -= v[2];
    return *this;
}

inline vec3& vec3::operator *=(float c)
{
    n[0] *= c; n[1] *= c; n[2] *= c;
    return *this;
}

inline vec3& vec3::operator /=(float c)
{
    //assert(!(c==0.0));
    if (c==0.0) return *this;
    n[0] /= c; n[1] /= c; n[2] /= c;
    return *this;
}

inline vec3 vec3::operator +(const vec3& v) const
{
    return vec3(n[0] + v[0], n[1] + v[1], n[2] + v[2]);
}

inline vec3 vec3::operator -(const vec3& v) const
{
    return vec3(n[0] - v[0], n[1] - v[1], n[2] - v[2]);
}

inline vec3 vec3::operator -() const
{
    return vec3(-n[0], -n[1], -n[2]);
}

inline vec3 vec3::operator *(float c) const
{
    return vec3(n[0] * c, n[1] * c, n[2] * c);
}

inline vec3 vec3::operator /(float c) const
{
    //assert(!(c==0.0));
    if (c==0.0) return vec3(0,0,0);
    return vec3(n[0] / c, n[1] / c, n[2] / c);
}


inline bool vec3::operator ==(const vec3& v) const
{
    return ((n[0]==v[0]) && (n[1]==v[1]) && (n[2]==v[2]));
}

inline bool vec3::operator !=(const vec3& v) const
{
    return (!(*this == v));
}

// OPERATIONS


inline float vec3::length() const
{
    return sqrt((n[0]*n[0]) + (n[1]*n[1]) + (n[2]*n[2]));
}

inline float vec3::lengthSquared() const
{
    return (n[0]*n[0]) + (n[1]*n[1]) + (n[2]*n[2]);
}

inline vec3& vec3::normalize()
{
    float len = length();
    //assert(!(len==0.0));
    if (len==0.0) return *this;
    *this /= len;
    return *this;
}

inline void vec3::copyTo(float f[3]) const
{
    f[0] = n[0]; f[1] = n[1]; f[2] = n[2];
}

/*
inline float * vec3::copyFrom ()
{
    return n;
}
*/

inline vec3& vec3 :: rotateX(float amnt)
{
    float s = sin(amnt);
    float c = cos(amnt);
    float y = n[1];
    float z = n[2];
  
    n[1] = (y * c) - (z * s);
    n[2] = (y * s) + (z * c);
    
    return *this;
}

inline vec3& vec3 :: rotateY(float amnt)
{
    float s = sin(amnt);
    float c = cos(amnt);
    float x = n[0];
    float z = n[2];

    n[0] = (x * c) + (z * s);
    n[2] = (z * c) - (x * s);
    
    return *this;
}

inline vec3& vec3 :: rotateZ(float amnt)
{
    float s = sin(amnt);
    float c = cos(amnt);
    float x = n[0];
    float y = n[1];
  
    n[0] = (x * c) - (y * s);
    n[1] = (y * c) + (x * s);
    
    return *this;
}

inline vec3 cross(const vec3& v1, const vec3& v2)
{
    return vec3(v1[1] * v2[2] - v1[2] * v2[1],
                v1[2] * v2[0] - v1[0] * v2[2],
                v1[0] * v2[1] - v1[1] * v2[0]);
}

inline float distance(const vec3& v1, const vec3& v2)
{
    return sqrt( (v1[0]-v2[0])*(v1[0]-v2[0]) 
                +(v1[1]-v2[1])*(v1[1]-v2[1])
                +(v1[2]-v2[2])*(v1[2]-v2[2]));
}

inline float distanceSquared(const vec3& v1, const vec3& v2)
{
    return  (v1[0] - v2[0])*(v1[0] - v2[0])
           +(v1[1] - v2[1])*(v1[1] - v2[1])
           +(v1[2] - v2[2])*(v1[2] - v2[2]);
}

inline float dot(const vec3& v1, const vec3& v2)
{
    return v1[0] * v2[0] + v1[1] * v2[1] + v1[2] * v2[2];
}

inline vec3 lerp(float f, const vec3& v1, const vec3& v2)
{
    return v1 + ((v2 - v1) * f);
}

// OUTPUT

inline ostream & operator << ( ostream& os, const vec3& v)
{
    os << v[0] << " " << v[1] << " " << v[2] << " ";
    return os;
}

inline istream & operator >> (istream& os, vec3& v)
{
	os >> v[0] >> v[1] >> v[2];
	return os;
}

inline vec3 operator *(float c, const vec3& v)
{
    return vec3(c * v[0], c * v[1], c * v[2]);
}



#endif		
