import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;

/**
 * Example03 uses Java 2D rendering hints to request antialiasing.
 * <p>
 * IMPORTANT:  This source code example is offered under the license at:
 * http://wireless.java.sun.com/berkeley_license.html
 *
 * @author Bill Day <bill.day@sun.com>
 * @version 1.1
 * @see java.awt.Font
 * @see java.awt.Graphics2D
 * @see java.awt.RenderingHints
 * @see java.awt.geom.AffineTransform
 * @see java.awt.geom.GeneralPath
**/

public class Example03 extends Frame {
  /**
   * Instantiates an Example03 object.
   **/
  public static void main(String args[]) {
    new Example03();
  }

  /**
   * Our Example03 constructor sets the frame's size, adds the
   * visual components, and then makes them visible to the user.
   * It uses an adapter class to deal with the user closing
   * the frame.
   **/
  public Example03() {
    //Title our frame.
    super("Java 2D Example03, Curves and Antialiasing");

    //Set the size for this frame.
    setSize(330,270); 

    //We need to turn on the visibility of our frame
    //by setting the Visible parameter to true.
    setVisible(true);

    //Now, we want to be sure we properly dispose of resources 
    //this frame is using when the window is closed.  We use 
    //an anonymous inner class adapter for this.
    addWindowListener(new WindowAdapter() 
      {public void windowClosing(WindowEvent e) 
         {dispose(); System.exit(0);}  
      }
    );
  }

  /**
   * The paint method is where the real magic is.  In previous
   * examples, we saw some jagged edges due to aliasing.
   * Example03 illustrates how to use rendering hints to request
   * an anti-aliased render from Graphics2D.
   **/
  public void paint(Graphics g) {
    Graphics2D g2d = (Graphics2D) g;

    //This time, we want to use anti-aliasing if possible
    //to avoid the jagged edges that were so prominent in
    //our last example.  We ask the Java 2D rendering
    //engine (Graphics2D) to do this using a "rendering hint".
    g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
       RenderingHints.VALUE_ANTIALIAS_ON);

    //We reuse our GeneralPath filled shape.  We translate
    //and rotate this shape as we did before.
    GeneralPath path = new GeneralPath(GeneralPath.WIND_EVEN_ODD);
    path.moveTo(0.0f,0.0f);
    path.lineTo(0.0f,125.0f);
    path.quadTo(100.0f,100.0f,225.0f,125.0f);
    path.curveTo(260.0f,100.0f,130.0f,50.0f,225.0f,0.0f);
    path.closePath();

    AffineTransform at = new AffineTransform();
    at.setToRotation(-Math.PI/8.0);
    g2d.transform(at);
    at.setToTranslation(0.0f,150.0f);
    g2d.transform(at);

    g2d.setColor(Color.green);
    g2d.fill(path);

    //Now, let's use some of the Java font and text support.
    //Note that you need to be sure you have the same fonts I
    //use in the example (Times New Roman True Type) if you
    //execute this example code.
    Font exFont = new Font("TimesRoman",Font.PLAIN,40);

    //Un-comment the following diagnostic println's if you
    //want to see what font was returned.  This can be useful
    //when you have limited font supports on your system and
    //are not sure which font the Java runtime may have
    //substituted for your requested font.
    //System.out.println(exFont.getFamily());
    //System.out.println(exFont.isPlain());
    //System.out.println(exFont.getSize());

    g2d.setFont(exFont);
    g2d.setColor(Color.black);
    g2d.drawString("Hello Camp",0.0f,0.0f);
  }
}
