//==============================================================================
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//==============================================================================

//==============================================================================
// File: cVector2f.hpp
// Project: Shooting Star
// Author: Jarmo Hekkanen <jarski@2ndpoint.fi>
// Copyrights (c) 2003 2ndPoint ry (www.2ndpoint.fi)
//------------------------------------------------------------------------------
// Revision history
//==============================================================================
#ifndef cVector2f_hpp
#define cVector2f_hpp
//==============================================================================
// Includes
#include <math.h>
//------------------------------------------------------------------------------
// Namespaces
using namespace std;
namespace ShootingStar {
//------------------------------------------------------------------------------
// Forward declarations
//==============================================================================
	
//==============================================================================
//! 2D vector
//------------------------------------------------------------------------------
class cVector2f
{
	// Constructor & Destructor
	public:
		//! Constructor
		cVector2f (void) {};
		//! Constructor
		cVector2f (float x, float y):
		mX (x),
		mY (y)
		{
		};
		//! Destructor
		~cVector2f (void) {};
	
	// Public methods
	public:
		//! Return lenght of the  vector
		float GetLenght (void) const { return sqrt (mX * mX + mY * mY); };

		//! Return normalized version of this vector
		cVector2f Normalize (void) const
		{
			float lenght = GetLenght ();
			return cVector2f (mX / lenght, mY / lenght);
		};

		//! Normalize this vector
		void NormalizeSelf (void) { *this = this->Normalize (); };
		
		//! Calculate dot product of two vectors
		float DotProduct (const cVector2f &b) const
		{
			return mX * b.mX + mY * b.mY;
		};
		
		//! Project this vector onto vector b
		void Project (const cVector2f &b)
		{
			*this = b * (DotProduct (b) / b.DotProduct (b));
		};
		
		//! Return projectio of this vector onto vector b
		cVector2f GetProjection (const cVector2f &b) const
		{
			cVector2f projection = *this;
			projection.Project (b);
			return projection;
		};
		
		//! Return the angle between two vectors
		float GetAngle (const cVector2f &b)
		{
			return acos (DotProduct (b) / (GetLenght () * b.GetLenght ()));
		};
		
		//! Return normal vector
		cVector2f GetNormal (void)
		{
			return cVector2f (-mY, mX);
		};
		
	// Const operators
	public:
		//! Addition of two vectors
		cVector2f operator+ (const cVector2f &rs) const
		{ 
			return cVector2f (mX + rs.mX, mY + rs.mY);
		};
		//! Subtraction of two vectors
		cVector2f operator- (const cVector2f &rs) const
		{ 
			return cVector2f (mX - rs.mX, mY - rs.mY);
		};
		//! Multiply with scalar
		cVector2f operator* (float rs) const 
		{ 
			return cVector2f (mX * rs, mY * rs);
		};
		//! Divine with scalar
		cVector2f operator/ (float rs) const
		{ 
			return cVector2f (mX / rs, mY / rs);
		};
		//! Compare two vectors
		bool operator== (const cVector2f &rs) const
		{
			return mX == rs.mX && mY == rs.mY;
		}
	
	// Nonconst operators
	public:
		//! Add vector to this vector
		const cVector2f &operator+= (const cVector2f &rs)
		{ 
			mX += rs.mX;
			mY += rs.mY;
			return *this;
		};
		//! Subtract vector from this vector
		const cVector2f &operator-= (const cVector2f &rs)
		{ 
			mX -= rs.mX;
			mY -= rs.mY;
			return *this;
		};
		//! Multiply this vector with scalar
		const cVector2f &operator*= (float rs)
		{ 
			mX *= rs;
			mY *= rs;
			return *this;
		};
		//! Divine this vector with scalar
		const cVector2f &operator/= (float rs)
		{ 
			mX /= rs;
			mY /= rs;
			return *this;
		};
		
	// Member variables
	public:
		float mX;	//!< X component
		float mY;	//!< Y component
};
//==============================================================================

//==============================================================================
}		// End of the ShootingStar namespace
#endif // cVector2f_hpp
//------------------------------------------------------------------------------
// EOF
//==============================================================================
