/*
 * Blob/metaball/marching cube code
 * Written by neon of nocturnal (adapted for the demolib, optimized
 * and almost totally rewritten) -- used with permission.
 */
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>

#ifdef WIN32
#include <windows.h>
#endif
#include <GL/gl.h>

#include "demolib_prefs.h"
#include "opengl/blob.h"
#include "opengl/tritable.h"
#include "opengl/extensions.h"
#include "math/array.h"
#include "exception.h"

#if DEMOLIB_BLOBS

#define MAX_POLYS 20000

const float D = 500.0;
const float THRESHOLD = 100.0;

void Blob::ResetThresholdPoints()
{
        memset(Vertex3DList, 0, sizeof(Vertex3DList));
}

void Blob::ResetGrid()
{
	for (int i = 0; i < GP_X * GP_Y * GP_Z; i++) {
		GridPoints[i].value = 0.0f;
		GridPoints[i].nx = 0.0f;
		GridPoints[i].ny = 0.0f;
		GridPoints[i].nz = 0.0f;
		GridPoints[i].inside = 0;
		GridPoints[i].valid = 0;
	}
}

void Blob::move_from_point(int index, int i, int j, int k, float x, float y, float z)
{
        int numInside = 0;
        //int index = i*GP_Y*GP_Z + j*GP_Z + k;
        
        /* if all points in the cube are precalculated, we just skip back */
        if (GridPoints[index].valid &&
            GridPoints[index + 1].valid &&
            GridPoints[index + GP_Z].valid &&
            GridPoints[index + GP_Z + 1].valid &&
            GridPoints[index + GP_Z*GP_Y].valid &&
            GridPoints[index + GP_Z*GP_Y + 1].valid &&
            GridPoints[index + GP_Z*GP_Y + GP_Z].valid &&
            GridPoints[index + GP_Z*GP_Y + GP_Z + 1].valid) {
                return;
        }

        /* then make sure all the points in the cube are valid */
        numInside += calcPoint(index, x, y, z);
        numInside += calcPoint(index + 1, x, y, z + GP_STEPZ);
        numInside += calcPoint(index + GP_Z, x, y + GP_STEPY, z);
        numInside += calcPoint(index + GP_Z + 1, x, y + GP_STEPY, z + GP_STEPZ);
        numInside += calcPoint(index + GP_Z*GP_Y, x + GP_STEPX, y, z);
        numInside += calcPoint(index + GP_Z*GP_Y + 1, x + GP_STEPX, y, z + GP_STEPZ);
        numInside += calcPoint(index + GP_Z*GP_Y + GP_Z, x + GP_STEPX, y + GP_STEPY, z);
        numInside += calcPoint(index + GP_Z*GP_Y + GP_Z + 1, x + GP_STEPX, y + GP_STEPY, z + GP_STEPZ);

        /* if completely outside, don't calculate it */
        if (numInside == 0) {
                return;
        }
        /* if completely inside, move further out */
        if (numInside == 8) {
                if (k > 0) {
                        move_from_point(index - 1, i, j, k - 1, x, y, z - GP_STEPZ);
                }
                return;
        }

        /*
         * otherwise, calculate all the points around this one. including diagonal
         * points, that would be 26 cubes, and I don't want to write all that code,
         * so I just go for the 6 nearby cubes -- I'm lazy, and I guess the code
         * will perform better this way anyhow ;-)
         */
        if (i > 0) {
                move_from_point(index - GP_Z*GP_Y, i-1, j, k, x - GP_STEPX, y, z);
        }
        if (i < GP_X-2) {
                move_from_point(index + GP_Z*GP_Y, i+1, j, k, x + GP_STEPX, y, z);
        }
        if (j > 0) {
                move_from_point(index - GP_Z, i, j-1, k, x, y - GP_STEPY, z);
        }
        if (j < GP_Y-2) {
                move_from_point(index + GP_Z, i, j+1, k, x, y + GP_STEPY, z);
        }
        if (k > 0) {
                move_from_point(index - 1, i, j, k-1, x, y, z - GP_STEPZ);
        }
        if (k < GP_Z-2) {
                move_from_point(index + 1, i, j, k+1, x, y, z + GP_STEPZ);
        }
}

int Blob::calcPoint(int index, float x, float y, float z)
{
        if (!GridPoints[index].valid) {
                for (int l = 0; l < this->num_mps; l++) {
                        MovingPoint *mp = &MovingPoints[l];
        
                        const float nx = x - mp->x;
                        const float ny = y - mp->y;
                        const float nz = z - mp->z;

                        float distance = nx*nx + ny*ny + nz*nz;
                        const float div = 1.0f / (distance + 1.0f);

#if !defined(i386)
                        distance = sqrt(distance);
#else   /* defined(i386) */
#if defined(__GNUC__)
                        asm (
                                "subl   $0x3f800000, %1\n"
                                "shrl   $1, %1\n"
                                "addl   $0x3f800000, %1\n" :
                        "=r" (distance) : "0" (distance));
#else   /* !defined(__GNUC__) */
                        _asm {
                                mov             eax,distance
                                sub             eax,0x3F800000
                                shr             eax,1
                                add             eax,0x3F800000
                                mov             distance,eax
                        }
#endif  /* !defined(__GNUC__) */
#endif  /* defined(i386) */

                        /* could move this point to later, but I don't think it's worth it */
                        GridPoints[index].nx += nx * div;
                        GridPoints[index].ny += ny * div;
                        GridPoints[index].nz += nz * div;

                        const float div2 = 1.0f / (distance + 1.0f);
                        GridPoints[index].value += mp->value * div2;
                }

                GridPoints[index].inside = (GridPoints[index].value >= THRESHOLD);
                GridPoints[index].valid = 1;
        }

        if (GridPoints[index].inside) {
                return 1;
        } else {
                return 0;
        }
}

void Blob::CalcGrid()
{
        for (int i = 0; i < this->num_mps; i++) {
                MovingPoint *mp = &MovingPoints[i];

                /* find the closest grid point */
                int j = (int)((mp->x - GP_MINX) / GP_STEPX);
                int k = (int)((mp->y - GP_MINY) / GP_STEPY);
                int l = (int)((mp->z - GP_MINZ) / GP_STEPZ);
                int index = j*GP_Y*GP_Z + k*GP_Z + l;

                const float x = (float)(j) * GP_STEPX + GP_MINX;
                const float y = (float)(k) * GP_STEPY + GP_MINY;
                const float z = (float)(l) * GP_STEPZ + GP_MINZ;

                if (j < 0 || k < 0 || l < 0 ||
                    j >= GP_X || k >= GP_Y || l >= GP_Z) {
                        throw new FatalException("Out of bounds in metaball code!");
                        /* uh-oh! out of bounds... */
                        continue;
                }

                /* now start recursively from this point on */
                this->move_from_point(index, j, k, l, x, y, z);
        }
}

/* needs optimization (SSE/3DNow?) */
inline void Blob::CalcThresholdPoint(GridPoint *gp1, GridPoint *gp2,
                                     Vertex3D *tp,
                                     float gx, float gy,
                                     float gz,
                                     enum axis ax)
{
	const float gp1v = gp1->value, gp2v = gp2->value;

	if ((gp1v < THRESHOLD && gp2v < THRESHOLD) ||
	    (gp1v > THRESHOLD && gp2v > THRESHOLD)) {
		return;
	}

	/*
	 * interpolate linearly between the grid points, based on their
	 * energy -- not perfect, but an OK approximation
	 */
	const float t = fabs((gp1v - THRESHOLD) / (gp2v - gp1v));

	tp->x = gx;
	tp->y = gy;
	tp->z = gz;

	tp->nx = (gp2->nx * t) + (gp1->nx * (1.0f - t));
	tp->ny = (gp2->ny * t) + (gp1->ny * (1.0f - t));
	tp->nz = (gp2->nz * t) + (gp1->nz * (1.0f - t));

	switch (ax) {
		case xaxis:  tp->x += t * (float)GP_STEPX;  break;
		case yaxis:  tp->y += t * (float)GP_STEPY;  break;
		case zaxis:  tp->z += t * (float)GP_STEPZ;  break;
	}
}

void Blob::CalcThresholdPoints()
{
        int i, j, k;
        float gx, gy, gz;
        int index = 0;

	this->max_vert = 0;
	
        for (i = 0, gx = GP_MINX; i < GP_X-1; i++, gx += GP_STEPX, index += GP_Z) {
                for (j = 0, gy = GP_MINY; j < GP_Y-1; j++, gy += GP_STEPY, index++) {
                        int index2 = (j*(GP_Z-1) + i*(GP_Y-1)*(GP_Z-1)) * 3;
                        for (k = 0, gz = GP_MINZ; k < GP_Z-1; k++, gz += GP_STEPZ, index++) {
                                if (i < GP_X-1) {
                                        CalcThresholdPoint(&GridPoints[index],&GridPoints[index+GP_Z*GP_Y],&Vertex3DList[index2++],gx,gy,gz,xaxis);
                                }
                                if (j < GP_Y-1) {
                                        CalcThresholdPoint(&GridPoints[index],&GridPoints[index+GP_Z],&Vertex3DList[index2++],gx,gy,gz,yaxis);
                                }
                                if (k < GP_Z-1) {
                                        CalcThresholdPoint(&GridPoints[index],&GridPoints[index+1],&Vertex3DList[index2++],gx,gy,gz,zaxis);
                                }
                        }
			if (index2 > this->max_vert)
				this->max_vert = index2;
                }
        }
}

void Blob::set_moving_point(int point, float x,float y,float z, float value)
{
        MovingPoints[point].x = x;
        MovingPoints[point].y = y;
        MovingPoints[point].z = z;
        MovingPoints[point].value = value;
}

void Blob::MakePolys(bool draw_halo, float halo_length, int stencil_mask)
{
        int i, j, k, a, mask;
        int EdgeList[12];
        int index = 0;

	Array<int> indices;

        for (i = 0; i < (GP_X-1)*(GP_Z-1)*(GP_Y-1)*3; i += (GP_X-1)*(GP_Z-1)*3) {
                for (j = 0; j < (GP_Y-1)*(GP_Z-1)*3; j += (GP_Z-1)*3) {
                        for (k = 0; k < (GP_Z-1)*3; k += 3) {
                                GridPoint *gps[8];
                                gps[0] = &GridPoints[index + 1];
                                gps[1] = &GridPoints[index + GP_Y*GP_Z + 1];
                                gps[2] = &GridPoints[index + GP_Y*GP_Z];
                                gps[3] = &GridPoints[index];
                                gps[4] = &GridPoints[index + GP_Z + 1];
                                gps[5] = &GridPoints[index + GP_Y*GP_Z + GP_Z + 1];
                                gps[6] = &GridPoints[index + GP_Y*GP_Z + GP_Z];
                                gps[7] = &GridPoints[index + GP_Z];
                                index++;

                                mask = 0;
                                if (gps[0]->inside) mask |= 0x01;
                                if (gps[1]->inside) mask |= 0x02;
                                if (gps[2]->inside) mask |= 0x04;
                                if (gps[3]->inside) mask |= 0x08;
                                if (gps[4]->inside) mask |= 0x10;
                                if (gps[5]->inside) mask |= 0x20;
                                if (gps[6]->inside) mask |= 0x40;
                                if (gps[7]->inside) mask |= 0x80;

                                EdgeList[0] = i                       + j              + k+3 + xaxis;
                                EdgeList[1] = i + (GP_Z-1)*(GP_Y-1)*3 + j              + k   + zaxis;
                                EdgeList[2] = i                       + j              + k   + xaxis;
                                EdgeList[3] = i                       + j              + k   + zaxis;

                                EdgeList[4] = i                       + j + (GP_Z-1)*3 + k+3 + xaxis;
                                EdgeList[5] = i + (GP_Z-1)*(GP_Y-1)*3 + j + (GP_Z-1)*3 + k   + zaxis;
                                EdgeList[6] = i                       + j + (GP_Z-1)*3 + k   + xaxis;
                                EdgeList[7] = i                       + j + (GP_Z-1)*3 + k   + zaxis;

                                EdgeList[8] = i                       + j              + k+3 + yaxis;
                                EdgeList[9] = i + (GP_Z-1)*(GP_Y-1)*3 + j              + k+3 + yaxis;
                                EdgeList[10]= i + (GP_Z-1)*(GP_Y-1)*3 + j              + k   + yaxis;
                                EdgeList[11]= i                       + j              + k   + yaxis;

                                for (a = 0; TriTable[mask][a] != -1; a+=3) {
                                        const int i1 = EdgeList[TriTable[mask][a]];
                                        const int i2 = EdgeList[TriTable[mask][a+1]];
                                        const int i3 = EdgeList[TriTable[mask][a+2]];

					indices.add_end(i1);
					indices.add_end(i2);
					indices.add_end(i3);
                                }
                        }
                        index++;
              }
              index += GP_Z;
        }

	if (stencil_mask != 0x0) {
		glEnable(GL_STENCIL_TEST);
		glStencilFunc(GL_ALWAYS, stencil_mask, stencil_mask);
		glStencilMask(stencil_mask);
		glStencilOp(GL_KEEP, GL_KEEP, GL_REPLACE);
	}
	glInterleavedArrays(GL_N3F_V3F, 0, Vertex3DList);
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_NORMAL_ARRAY);

	if (this->has_compiled_vertex_array) {
		(*this->glLockArraysEXT)(0, this->max_vert);
	}	
	
	if (this->has_draw_range_elements) {
		(*this->glDrawRangeElementsEXT)(GL_TRIANGLES, 0, this->max_vert,
			indices.num_elems(), GL_UNSIGNED_INT, indices.get_array());
	} else {
		glDrawElements(GL_TRIANGLES, indices.num_elems(),
			       GL_UNSIGNED_INT, indices.get_array());
	}
	if (this->has_compiled_vertex_array) {
		(*this->glUnlockArraysEXT)();
	}

	if (draw_halo) {
		glDisable(GL_TEXTURE_2D);
		glDisable(GL_BLEND);
		glDisable(GL_LIGHTING);
		glEnable(GL_DEPTH_TEST);
		glEnable(GL_STENCIL_TEST);
		glStencilFunc(GL_EQUAL, 0x0, stencil_mask);
		glStencilMask(stencil_mask);
		glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);

		glColor4f(0.0f, 0.0f, 0.0f, 1.0f);
		
		/* okay, this isn't very pretty ;-) */
		const float hl = halo_length;
		for (int i = 0; i < this->max_vert; i++) {
			Vertex3DList[i].x += Vertex3DList[i].nx * hl;
			Vertex3DList[i].y += Vertex3DList[i].ny * hl;
			Vertex3DList[i].z += Vertex3DList[i].nz * hl;
		}
		
		if (this->has_compiled_vertex_array) {
			(*this->glLockArraysEXT)(0, this->max_vert);
		}	
	
		if (this->has_draw_range_elements) {
			(*this->glDrawRangeElementsEXT)(GL_TRIANGLES, 0, this->max_vert,
				indices.num_elems(), GL_UNSIGNED_INT, indices.get_array());
		} else {
			glDrawElements(GL_TRIANGLES, indices.num_elems(),
				       GL_UNSIGNED_INT, indices.get_array());
		}

		if (this->has_compiled_vertex_array) {
			(*this->glUnlockArraysEXT)();
		}
	}
	glDisable(GL_STENCIL_TEST);
}

Blob::Blob(int num_mps)
{
        this->num_mps = num_mps;

        GridPoints = new GridPoint[GP_X*GP_Y*GP_Z];
        Vertex3DList = new Vertex3D[MAX_VERTEX];
        MovingPoints = new MovingPoint[this->num_mps];

        ResetGrid();

	if (GLExtensions::has_ext("GL_EXT_draw_range_elements")) {
		this->has_draw_range_elements = true;
		this->glDrawRangeElementsEXT = (PFNGLDRAWRANGEELEMENTSEXTPROC)
			GLExtensions::func_ptr("glDrawRangeElementsEXT");
	} else {
		this->has_draw_range_elements = false;
	}
	
	if (GLExtensions::has_ext("GL_EXT_compiled_vertex_array")) {
		this->has_compiled_vertex_array = true;
		this->glLockArraysEXT = (PFNGLLOCKARRAYSEXTPROC)
			GLExtensions::func_ptr("glLockArraysEXT");
		this->glUnlockArraysEXT = (PFNGLUNLOCKARRAYSEXTPROC)
			GLExtensions::func_ptr("glUnlockArraysEXT");
	} else {
		this->has_compiled_vertex_array = false;
	}
}

Blob::~Blob()
{
        delete[] GridPoints;
        delete[] Vertex3DList;
        delete[] MovingPoints;
}

void Blob::draw_object()
{
	this->draw_object(false, 0.0f, 0);
}

void Blob::draw_object(bool draw_halo, float halo_length, int stencil_mask)
{
        ResetGrid();
        ResetThresholdPoints();
        CalcGrid();
        CalcThresholdPoints();
        MakePolys(draw_halo, halo_length, stencil_mask);
}
#endif 
