///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <ThCore.h>
#include <ThMemory.h>
#include <ThServer.h>

#include "ThServer.h"

#include <cstring>

///////////////////////////////////////////////////////////////////////////////////////////////////

ThSingleton<IThServer> Server;

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThMessage constructors -------------------------------------------------------------------------

ThMessage::ThMessage(unsigned int message, unsigned int toID, unsigned int fromID):
	m_message(message),
	m_toID(toID),
	m_fromID(fromID)
{
}

// ThMessage attributes ---------------------------------------------------------------------------

unsigned int ThMessage::getMessage(void) const
{
	return m_message;
}

void ThMessage::setMessage(unsigned int message)
{
	m_message = message;
}

unsigned int ThMessage::getToID(void) const
{
	return m_toID;
}

void ThMessage::setToID(unsigned int toID)
{
	m_toID = toID;
}

unsigned int ThMessage::getFromID(void) const
{
	return m_fromID;
}

void ThMessage::setFromID(unsigned int fromID)
{
	m_fromID = fromID;
}

const void* ThMessage::getData(void) const
{
	return m_data;
}

void ThMessage::setData(const void* data)
{
	m_data = data;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThMessageItem constructors ---------------------------------------------------------------------

ThMessageItem::ThMessageItem(unsigned int message, unsigned int toID, unsigned int fromID):
	ThMessage(message, toID, fromID)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThServerObject attributes ----------------------------------------------------------------------

unsigned int ThServerObject::getID(void) const
{
	return m_id;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThServerObject constructors --------------------------------------------------------------------

ThServerObject::ThServerObject(unsigned int id)
{
	if (id == THID_INVALID)
		m_id = Server->createObjectID();
	else
		m_id = id;
}

ThServerObject::~ThServerObject(void)
{
}

// ThServerObject methods -------------------------------------------------------------------------

bool ThServerObject::recieve(ThMessage* message)
{
	return true;
}

// ThServerObject methods -------------------------------------------------------------------------

void ThServerObject::sendMessage(unsigned int message, unsigned int toID, const void* data)
{
	THASSERT(Server != NULL, "Cannot send messages until the server object is created.");

	Server->sendMessage(message, toID, m_id, data);
}

void ThServerObject::postMessage(unsigned int message, unsigned int toID, const void* data, unsigned int size)
{
	THASSERT(Server != NULL, "Cannot post messages until the server object is created.");

	Server->postMessage(message, toID, m_id, data, size);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThServer constructors -------------------------------------------------------------------------

IThServer::~IThServer(void)
{
}

// IThServer static methods -----------------------------------------------------------------------

bool IThServer::create(void)
{								
	if (Server)
		return true;

	ThPtr<ThServer> server = new ThServer();

	if (!server->open())
		return false;

	Server.attach(server.detach());
	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThServer constructors --------------------------------------------------------------------------

ThServer::ThServer(void)
{
	m_nextID = THID_CUSTOM_BASE;
}

ThServer::~ThServer(void)
{
	close();
}

// ThServer methods -------------------------------------------------------------------------------

bool ThServer::open(void)
{
	close();

	return true;
}

void ThServer::close(void)
{
	m_objects.release();
}

// ThServer interface methods ---------------------------------------------------------------------

void ThServer::sendMessage(unsigned int message, unsigned int toID, unsigned int fromID, const void* data)
{
	ThMessage messageObject(message, toID, fromID);
	messageObject.setData(data);

	sendMessage(&messageObject);
}

void ThServer::postMessage(unsigned int message, unsigned int toID, unsigned int fromID, const void* data, unsigned int size)
{
	ThMessageItem* messageItem = new ThMessageItem(message, toID, fromID);
	
	if (size)
	{
		THASSERT(data != NULL, "Cannot pass NULL data parameter with non-zero size.");

		messageItem->m_data.allocate(size);
		messageItem->setData(messageItem->m_data);

		memcpy(messageItem->m_data, data, size);
	}

	m_messages.attachLast(messageItem);
}

// ThServer interface object methods --------------------------------------------------------------

unsigned int ThServer::createObjectID(void)
{
	return m_nextID++;
}

bool ThServer::registerObject(ThServerObject* object)
{
	if (findObject(object->getID()))
		return false;

	m_objects.attachFirst(object);
	return true;
}

ThServerObject* ThServer::findObject(unsigned int id)
{
	for (ThIterator<ThServerObject> object(m_objects);  object;  object.next())
	{
		if (object->getID() == id)
			return object;
	}

	return NULL;
}

// ThServer methods -------------------------------------------------------------------------------

void ThServer::sendMessage(ThMessage* message)
{
	switch (message->getToID())
	{
		case THID_ANNOUNCE:
		{
			// send to all except sender

			if (message->getFromID() != THID_SERVER)
				recieve(message);

			for (ThIterator<ThServerObject> object(m_objects);  object; )
			{
				if (object->getID() == message->getFromID())
					object.next();
				else
				{
					if (object->recieve(message))
						object.next();
					else
						object.release();
				}
			}

			break;
		}

		case THID_SERVER:
		{
			// send to server

			recieve(message);
			break;
		}

		default:
		{
			// send to reciever

			if (ThServerObject* object = findObject(message->getToID()))
			{
				if (!object->recieve(message))
					delete object;
			}

			break;
		}
	}
}

void ThServer::recieve(ThMessage* message)
{
	switch (message->getMessage())
	{
		case THMSG_UPDATE:
		{
			for (ThIterator<ThMessageItem> message(m_messages);  message;  message.release())
				sendMessage(message);

			break;
		}
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////
