///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <ThCore.h>
#include <ThGeometry.h>
#include <ThMemory.h>
#include <ThDisplay.h>
#include <ThSystem.h>
#include <ThEffect.h>
#include <ThParticle.h>

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThParticleEmitter methods ----------------------------------------------------------------------

void ThParticleEmitter::generateDefaults(void)
{
	m_position.set(0.f, 0.f, 0.f);
	m_speed.set(1.f, 1.f);
	m_vector.set(0.f, 1.f, 0.f);
	m_angle.set(0.f, 0.f);
	m_accel.set(0.f, -1.f, 0.f);
	m_color.set(ThVector4(1.f, 1.f, 1.f, 1.f), ThVector4(1.f, 1.f, 1.f, 1.f));
	m_size.set(ThVector2(1.f, 1.f), ThVector2(1.f, 1.f));
	m_time.set(1.f, 1.f);
	m_ratio = 10.f;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThParticleSystem constructors ------------------------------------------------------------------

ThParticleSystem::ThParticleSystem(void)
{
	m_started = false;
	m_paused  = false;
	m_counter = 0.f;
}

ThParticleSystem::~ThParticleSystem(void)
{
	release();
}

// ThParticleSystem methods -----------------------------------------------------------------------

void ThParticleSystem::create(unsigned int count)
{
	release();

	m_emitter = new ThParticleEmitter();
	m_emitter->generateDefaults();

	m_particles.allocate(count);

	for (unsigned int i = 0;  i < count;  i++)
		m_passive.attachFirst(m_particles + i);
}

void ThParticleSystem::release(void)
{
	m_started = false;
	m_paused  = false;
	m_counter = 0.f;

	m_emitter.release();

	m_particles.release();
}

void ThParticleSystem::start(void)
{
	stop();

	m_started = true;
}

void ThParticleSystem::stop(void)
{
	m_started = false;
	m_paused  = false;
	m_counter = 0.f;

	while (ThParticle* particle = m_active.getFirst())
		m_passive.attachFirst(particle);
}

void ThParticleSystem::pause(void)
{
	if (!m_started)
		return;

	m_paused = true;
}

void ThParticleSystem::resume(void)
{
	if (!m_started)
		return;

	m_paused = false;
}

// ThParticleSystem callbacks ---------------------------------------------------------------------

bool ThParticleSystem::update(float deltaTime)
{
	if (!ThEffectObject::update(deltaTime))
		return false;

	if (!m_started || m_paused)
		return true;

	// emit x new particles per second
	{
		m_counter += deltaTime * m_emitter->m_ratio;

		unsigned int count = (unsigned int) m_counter;
		if (count > m_passive.getCount())
			count = m_passive.getCount();

		m_counter -= count;

		const float timeSlice = deltaTime / (float) count;

		for (unsigned int i = 0;  i < count;  i++)
		{
			ThParticle* particle = m_passive.getFirst();

			createParticle(particle);

			particle->m_timeLeft -= timeSlice * (float) i;

			m_active.attachFirst(particle);
		}
	}

	for (ThIterator<ThParticle> particle(m_active);  particle;  particle.next())
	{
		if (!updateParticle(particle, deltaTime))
		{
			ThParticle* next = particle->getNext();

			m_passive.attachFirst(particle);

			particle = next;
		}
	}

	return true;
}

bool ThParticleSystem::render(void)
{
	if (!ThEffectObject::render())
		return false;

	if (!m_started)
		return true;

	for (ThIterator<ThParticle> particle(m_active);  particle;  particle.next())
		renderParticle(particle);

	return true;
}

// ThParticleSystem attributes --------------------------------------------------------------------

bool ThParticleSystem::isStarted(void) const
{
	return m_started;
}

bool ThParticleSystem::isPaused(void) const
{
	return m_paused;
}

unsigned int ThParticleSystem::getParticleCount(void) const
{
	return m_particles.getCount();
}

ThParticle* ThParticleSystem::getFirstParticle(void)
{
	return m_particles;
}

ThParticleEmitter* ThParticleSystem::getEmitter(void)
{
	return m_emitter;
}

// ThParticleSystem callbacks ---------------------------------------------------------------------

void ThParticleSystem::createParticle(ThParticle* particle)
{
	if (m_emitter)
	{
		particle->m_position = m_emitter->m_position;

		{
			particle->m_speed = m_emitter->m_vector;

			ThVector3 normal;

			if (particle->m_speed.x || particle->m_speed.z)
				normal = particle->m_speed.crossProduct(ThVector3(0.f, 1.f, 0.f));
			else
				normal = particle->m_speed.crossProduct(ThVector3(1.f, 0.f, 0.f));

			ThMatrix3 rotation;

			rotation.setVector(particle->m_speed, random() * 2.f * (float) M_PI);
			rotation.rotateVector(normal);

			particle->m_speed += normal;
			particle->m_speed.scaleTo(m_emitter->m_speed.generate());
		}

		particle->m_color    = m_emitter->m_color.generate();
		particle->m_size     = m_emitter->m_size.generate();
		particle->m_lifeTime = m_emitter->m_time.generate();
		particle->m_timeLeft = particle->m_lifeTime;
	}
}

bool ThParticleSystem::updateParticle(ThParticle* particle, float deltaTime)
{
	particle->m_position += particle->m_speed * deltaTime + m_emitter->m_accel * deltaTime * deltaTime;
	particle->m_speed    += m_emitter->m_accel * deltaTime;

	particle->m_color.w -= deltaTime * 1.f / particle->m_lifeTime;

	particle->m_timeLeft -= deltaTime;
	if (particle->m_timeLeft <= 0.f)
		return false;

	return true;
}

void ThParticleSystem::renderParticle(ThParticle* particle)
{
	glColor4fv(particle->m_color);

	const ThVector2 offset(particle->m_size.x * 0.5f, particle->m_size.y * 0.5f);

	glBegin(GL_QUADS);

	glTexCoord2f(0.f, 0.f);
	glVertex3f(particle->m_position.x - offset.x, particle->m_position.y - offset.y, particle->m_position.z);

	glTexCoord2f(0.f, 1.f);
	glVertex3f(particle->m_position.x - offset.x, particle->m_position.y + offset.y, particle->m_position.z);

	glTexCoord2f(1.f, 1.f);
	glVertex3f(particle->m_position.x + offset.x, particle->m_position.y + offset.y, particle->m_position.z);

	glTexCoord2f(1.f, 0.f);
	glVertex3f(particle->m_position.x + offset.x, particle->m_position.y - offset.y, particle->m_position.z);

	glEnd();
}

///////////////////////////////////////////////////////////////////////////////////////////////////
