///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <ThCore.h>
#include <ThMemory.h>
#include <ThString.h>
#include <ThWindow.h>
#include <ThError.h>
#include <ThStream.h>
#include <ThServer.h>
#include <ThMusic.h>

#include "bass.h"

#include "ThMusic.h"

///////////////////////////////////////////////////////////////////////////////////////////////////

#define THMUSICWINDOWCLASSNAME "Theresa Music Message Window"
#define THMUSICSYNCMESSAGENAME "Theresa Music Sync Message"

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThMusic constructors --------------------------------------------------------------------------

IThMusic::~IThMusic(void)
{
}

// IThMusic static methods ------------------------------------------------------------------------

IThMusic* IThMusic::createInstance(IThInputStream* source)
{
	ThPtr<ThMusic> music = new ThMusic();

	if (!music->open(source))
		return false;

	return music.detach();
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThMusicWindow constructors ---------------------------------------------------------------------

ThMusicWindow::ThMusicWindow(UINT message):
	m_message(message)
{
}

// ThMusicWindow methods --------------------------------------------------------------------------

bool ThMusicWindow::create(void)
{
	WNDCLASSEX wcl;

	ZeroMemory(&wcl, sizeof(wcl));
	wcl.cbSize = sizeof(wcl);
	wcl.lpszClassName = THMUSICWINDOWCLASSNAME;

	return ThWindow::create(wcl, 0, 0);
}

// ThMusicWindow callbacks ------------------------------------------------------------------------

LRESULT ThMusicWindow::message(UINT message, WPARAM wParam, LPARAM lParam)
{
	if (message == m_message)
	{
		unsigned char data = LOBYTE(wParam);

		Server->sendMessage(THMSG_MUSIC_SYNC, THID_ANNOUNCE, THID_MUSIC, &data);
		return 0;
	}

	return ThWindow::message(message, wParam, lParam);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThMusic constructors ---------------------------------------------------------------------------

ThMusic::ThMusic(void):
	ThServerObject(THID_MUSIC)
{
	m_started = false;
	m_paused  = false;
	m_music   = NULL;
	m_stream  = NULL;
}

ThMusic::~ThMusic(void)
{
	close();
}

// ThMusic methods --------------------------------------------------------------------------------

bool ThMusic::open(IThInputStream* source)
{
	close();

	UINT message;

	// register message
	{
		message = RegisterWindowMessage(THMUSICSYNCMESSAGENAME);
		if (!message)
		{
			Error->display("Music", "Unable to register music sync message.");
			return false;
		}
	}

	// create message window
	{
		m_window = new ThMusicWindow(message);

		if (!m_window->create())
		{
			Error->display("Music", "Unable to create system message window.");
			return false;
		}
	}

	if (!BASS_Init(-1, 44100, BASS_DEVICE_LEAVEVOL, m_window->getHandle()))
	{
		if (!Error->request("Music", "Unable to initialize BASS."))
			return false;

		if (!BASS_Init(-2, 44100, BASS_DEVICE_LEAVEVOL, m_window->getHandle()))
			return false;
	}

	// load music file
	{
		m_buffer.allocate(source->getSize());

		if (!source->readItems(m_buffer.getData(), m_buffer.getCount()))
			return Error->request("Music", "Unable to read data from music file.");

		m_music = BASS_MusicLoad(TRUE, m_buffer, 0, m_buffer.getSize(), BASS_MUSIC_RAMP | BASS_MUSIC_FT2MOD);

		if (m_music)
		{
			if (!BASS_ChannelSetSync(m_music, BASS_SYNC_MESSAGE | BASS_SYNC_MUSICFX, 1, (SYNCPROC*) message, NULL))
				return Error->request("Music", "Unable to set music channel synchronizer.");
		}
		else
		{
			m_stream = BASS_StreamCreateFile(TRUE, m_buffer, 0, m_buffer.getSize(), 0);
			if (!m_stream)
				return Error->request("Music", "Unable to load music file as any known format.");
		}
	}

	return true;
}

void ThMusic::close(void)
{
	stop();

	m_started = false;
	m_paused  = false;

	if (m_stream)
	{
		BASS_StreamFree(m_stream);
		m_stream = NULL;
	}

	if (m_music)
	{
		BASS_MusicFree(m_music);
		m_music = NULL;
	}

	m_buffer.release();

	BASS_Free();

	m_window.release();
}

// ThMusic interface methods ----------------------------------------------------------------------

void ThMusic::start(void)
{
	stop();

	BASS_Start();

	if (m_music)
	{
		BASS_MusicPreBuf(m_music);
		BASS_MusicPlay(m_music);
	}

	if (m_stream)
	{
		BASS_StreamPreBuf(m_stream);
		BASS_StreamPlay(m_stream, TRUE, 0);
	}

	m_started = true;
}

void ThMusic::stop(void)
{
	if (m_started)
	{
		BASS_Stop();

		m_started = false;
	}
}

void ThMusic::pause(void)
{
	if (m_paused)
		return;

	if (m_music)
		BASS_ChannelPause(m_music);

	if (m_stream)
		BASS_ChannelPause(m_stream);

	m_paused = true;
}

void ThMusic::resume(void)
{
	if (!m_paused)
		return;

	if (m_music)
		BASS_ChannelResume(m_music);

	if (m_stream)
		BASS_ChannelResume(m_stream);

	m_paused = false;
}

// ThMusic interface attributes -------------------------------------------------------------------

bool ThMusic::isStarted(void) const
{
	return m_started;
}

bool ThMusic::isPaused(void) const
{
	return m_paused;
}

float ThMusic::getTime(void) const
{
	HCHANNEL channel;

	if (m_music)
		channel = m_music;
	else
		channel = m_stream;

	const unsigned int position = BASS_ChannelGetPosition(channel);
	if (position == 0xffffffff)
		return 0.f;

	const float time = BASS_ChannelBytes2Seconds(channel, position);
	if (time < 0.f)
		return 0.f;

	return time;
}

bool ThMusic::setTime(float time)
{
	HCHANNEL channel;

	if (m_music)
		channel = m_music;
	else
		channel = m_stream;

	const unsigned int position = BASS_ChannelSeconds2Bytes(channel, time);
	if (position == 0xffffffff)
		return false;

	if (!BASS_ChannelSetPosition(channel, position))
		return false;

	return true;
}

// ThMusic callbacks ------------------------------------------------------------------------------

bool ThMusic::recieve(ThMessage* message)
{
	return ThServerObject::recieve(message);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
