///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <ThCore.h>
#include <ThMemory.h>
#include <ThString.h>
#include <ThDialog.h>
#include <ThError.h>

#include <windows.h>
#include <windowsx.h>

#include "ThError.h"

#include <cstdio>
#include <cstdarg>

#include "resource.h"

///////////////////////////////////////////////////////////////////////////////////////////////////

ThSingleton<IThError> Error;

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThError constructors --------------------------------------------------------------------------

IThError::~IThError(void)
{
}

// IThError static methods ------------------------------------------------------------------------

bool IThError::create(void)
{
	if (Error)
		return true;

	ThPtr<ThError> error = new ThError();

	if (!error->open())
		return false;

	Error.attach(error.detach());
	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef THERESA_SMALL_BUILD
///////////////////////////////////////////////////////////////////////////////////////////////////

// ThErrorDialog constructors ---------------------------------------------------------------------

ThErrorDialog::ThErrorDialog(const char* message, bool request):
	ThDialog(IDD_ERROR_DIALOG)
{
	m_message = message;
	m_request = request;
}

// ThErrorDialog callbacks ------------------------------------------------------------------------

BOOL ThErrorDialog::initialize(HWND control)
{
	Edit_SetText(getControl(IDC_ERROR_EDITMESSAGE), m_message);

	if (m_request)
	{
		Button_SetText(getControl(IDOK), "&Continue");
		Button_SetText(getControl(IDCANCEL), "&Abort");
	}
	else
	{
		Button_SetText(getControl(IDOK), "&OK");
		ShowWindow(getControl(IDCANCEL), SW_HIDE);
	}

	return ThDialog::initialize(control);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
#endif /*THERESA_SMALL_BUILD*/
///////////////////////////////////////////////////////////////////////////////////////////////////

// ThError constructors ---------------------------------------------------------------------------

ThError::ThError(void)
{
	m_file = INVALID_HANDLE_VALUE;
}

ThError::~ThError(void)
{
	close();
}

// ThError methods --------------------------------------------------------------------------------

bool ThError::open(void)
{
	close();

	// create log file
	{
		char tempPath[MAX_PATH];

		if (!GetTempPath(sizeof(tempPath), tempPath))
			return false;

		char filePath[MAX_PATH];

		if (!GetTempFileName(tempPath, "The", 0, filePath))
			return false;

		m_file = CreateFile(filePath, GENERIC_WRITE, FILE_SHARE_READ, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_WRITE_THROUGH, NULL);
		if (m_file == INVALID_HANDLE_VALUE)
			return false;
	}

	return true;
}

void ThError::close(void)
{
	if (m_file != INVALID_HANDLE_VALUE)
	{
		CloseHandle(m_file);
		m_file = INVALID_HANDLE_VALUE;
	}
}

// ThError interface methods ----------------------------------------------------------------------

void ThError::write(const char* name, const char* format, ...)
{
	char buffer[THERESA_BUFFER_SIZE];

	THSAFEFORMAT(buffer, sizeof(buffer), format);

	char message[THERESA_BUFFER_SIZE];

	ThString::formatS(message, sizeof(message), "Module: %s%s%s%s", name, THERESA_NEWLINE, buffer, THERESA_NEWLINE);

#ifdef THERESA_DEBUG_BUILD
	OutputDebugString(message);
#endif

	DWORD bytesWritten;

	WriteFile(m_file, message, ThString::length(message), &bytesWritten, NULL);
}

void ThError::display(const char* name, const char* format, ...)
{
	char buffer[THERESA_BUFFER_SIZE];

	THSAFEFORMAT(buffer, sizeof(buffer), format);

	write(name, buffer);

	char message[THERESA_BUFFER_SIZE];

	ThString::formatS(message, sizeof(message), "A runtime error has occurred in module %s.%sFor more detailed information, see below.%s%s%s", name, THERESA_NEWLINE, THERESA_NEWLINE, THERESA_NEWLINE, buffer);

#ifndef THERESA_SMALL_BUILD

	ThErrorDialog dialog(message, false);

	dialog.display(GetActiveWindow());

#endif /*THERESA_SMALL_BUILD*/
}

bool ThError::request(const char* name, const char* format, ...)
{
	char buffer[THERESA_BUFFER_SIZE];

	THSAFEFORMAT(buffer, sizeof(buffer), format);

	write(name, buffer);

	char message[THERESA_BUFFER_SIZE];

	ThString::formatS(message, sizeof(message), "A runtime error has occurred in module %s.%sFor more detailed information, see below.%s%s%s%s%sIf you wish to attempt resuming the operation, click Continue. Or click Abort to cancel the operation.", name, THERESA_NEWLINE, THERESA_NEWLINE, THERESA_NEWLINE, buffer, THERESA_NEWLINE, THERESA_NEWLINE);

#ifndef THERESA_SMALL_BUILD

	ThErrorDialog dialog(message, true);

	const int result = dialog.display(GetActiveWindow());

	if (result != IDOK)
		return false;

#endif /*THERESA_SMALL_BUILD*/

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
