// This program mixes WAV files together. The mixing process is controlled
// by a mix script (*.MIX), each line containing a name of the
// mixable file, mixing level in dB, and the position of the mixable
// compared to the beginning of the output file. The position is
// given as a sample ordinal number.

// The actual mixing work is done in the mix () subroutine. Its input
// and output may be a free mixture of mono and stereo.

// The mixing is done with floating-point numbers, and the result is
// normalized by subroutine norm () before conversion to integers.
// The mixing is done in ping-pong fashion between two temporary
// mixing files mix0.tmp and mix1.tmp.

#include <stdio.h>
#include <io.h>
#include <math.h>



#define SRATE 44100L;



char mn [2] [9] = {"mix0.tmp", "mix1.tmp"};
long ms = 0L;   // Mix file selector



void writehdr (FILE *f, long bc, long sr, long sof) {

  unsigned long  d, sl;
  unsigned short w, ss;

  ss = sof + 1;
  sl = 2L * (unsigned long) ss;

                 fwrite ("RIFF", 1, 4, f);
  d =  bc + 36L; fwrite (&d,     4, 1, f);       /* Bytes + header  */
                 fwrite ("WAVE", 1, 4, f);
                 fwrite ("fmt ", 1, 4, f);
  d =       16L; fwrite (&d,     4, 1, f);
  w =         1; fwrite (&w,     2, 1, f);       /* WAV format      */
  w =        ss; fwrite (&w,     2, 1, f);       /* Stereo          */
  d =        sr; fwrite (&d,     4, 1, f);       /* Sample rate     */
  d =   sl * sr; fwrite (&d,     4, 1, f);       /* Avg BPS         */
  w =    2 * ss; fwrite (&w,     2, 1, f);       /* Block align     */
  w =        16; fwrite (&w,     2, 1, f);       /* Bits per sample */
                 fwrite ("data", 1, 4, f);
  d =        bc; fwrite (&d,     4, 1, f); }     /* Bytes to go     */



long mix (char *fn, double ampl, long org, long sof) {

  // Argument ampl is the linear level of the mixable, org is the
  // sample number of the mixable origin, and sof is the stereo
  // output mode flag.

  double vl,         // Left channel sample
         vr,         // Right channel sample
         tl,         // Left channel mixable sample
         tr;         // Right channel mixable sample
  FILE   *mi,        // Temporary mix input file
         *mo,        // Temporary mix output file
         *mt;        // Mixable file
  long   i,
         mtlen,      // Length of the mixable in samples
         sif;        // Stereo input mode flag
  short  t;          // Mixable sample
  char   hdr [44];   // WAV header buffer

  sof = (sof != 0L);

  mi = fopen (mn [ms], "rb");
  if (mi == NULL) {
    printf ("Cannot open mix read file\n");
    return (1); }

  mo = fopen (mn [1 - ms], "wb");
  if (mo == NULL) {
    printf ("Cannot open mix write file\n");
    return (2); }

  mt = fopen (fn, "rb");
  if (mt == NULL) {
    printf ("Cannot open mixable file\n");
    return (3); }
  fread (hdr, 44, 1, mt);
  memcpy (&t, hdr + 22L, sizeof (short));
  sif = (long) t - 1L;
  memcpy (&mtlen, hdr + 40L, sizeof (long));
  mtlen /= sizeof (short);
  if (sif) mtlen /= 2L;

  ms = 1 - ms;

  for (i = 0 ; i < org ; i++) {
    if (fread (&vl, sizeof (double), 1, mi) != 1) vl = 0.0;
    if (sof) {if (fread (&vr, sizeof (double), 1, mi) != 1) vr = 0.0;}
    fwrite (&vl, sizeof (double), 1, mo);
    if (sof) fwrite (&vr, sizeof (double), 1, mo); }

  for (i = 0 ; i < mtlen ; i++) {
    if (fread (&vl, sizeof (double), 1, mi) != 1) vl = 0.0;
    if (sof) {if (fread (&vr, sizeof (double), 1, mi) != 1) vr = 0.0;}
    fread (&t, sizeof (short),  1, mt); tl = (double) t;
    if (sif) {fread (&t, sizeof (short),  1, mt); tr = (double) t;}
    if (sif && ! sof) tl = (tl + tr) / 2.0;
    if (! sif && sof) tr = tl;
    vl = vl + ampl * tl;
    vr = vr + ampl * tr;
    fwrite (&vl, sizeof (double), 1, mo);
    if (sof) fwrite (&vr, sizeof (double), 1, mo); }

  while (1) {
    if (fread (&vl, sizeof (double), 1, mi) != 1) break;
    if (sof) {if (fread (&vr, sizeof (double), 1, mi) != 1) break;}
    fwrite (&vl, sizeof (double), 1, mo);
    if (sof) fwrite (&vr, sizeof (double), 1, mo); }

  fclose (mt);
  fclose (mo);
  fclose (mi);

  return (0); }




// Finds the maximum level in the mix.

double getmax (void) {
  double m, v;
  FILE  *mi;
  mi = fopen (mn [ms], "rb");
  if (mi == NULL) {
    printf ("Cannot open mix read file\n");
    return (-1.0); }
  m = 0.0;
  while (fread (&v, sizeof (double), 1, mi) == 1) {
    if (m < fabs (v)) m = fabs (v); }
  fclose (mi);
  return (m); }



// Normalizes the final mix output.

long norm (char *fn, double m, long sr, long sof) {
  double v;
  FILE  *mi, *mo;
  long  bc;
  short i;
  sof = (sof != 0L);
  mi = fopen (mn [ms], "rb");
  if (mi == NULL) {
    printf ("Cannot open mix read file\n");
    return (1); }
  mo = fopen (fn, "wb");
  if (mo == NULL) {
    printf ("Cannot open output file\n");
    return (2); }
  bc = filelength (fileno (mi)) * sizeof (short) / sizeof (double);
  writehdr (mo, bc, sr, sof);
  while (fread (&v, sizeof (double), 1, mi) == 1) {
    if (m == 0.0) v = 0.0; else v = 32767.0 * v / m;
    i = (short) v;
    fwrite (&i, sizeof (short), 1, mo); }
  fclose (mo);
  fclose (mi);
  return (0); }



int main (int argc, char *argv []) {

  double m,            // Maximum amplitude of the mix
         level;        // Level of the mixable in dB
  FILE   *f;
  long   org,          // Time position of the mixable (sample index)
         sr,           // Sample rate
         stereo;       // Stereo output flag
  char   fn [256],     // The name of the mixable file
         line [256];   // Mix script line buffer

  if (argc < 4) {
    printf ("Use: mix <outfile> <mixscript> <stereoflag> [samplerate]\n");
    return (1); }

  if (argc >= 5) sscanf (argv [4], "%ld", &sr); else sr = SRATE;

  sscanf (argv [3], "%ld", &stereo);

  f = fopen (mn [ms],     "wb"); fclose (f);
  f = fopen (mn [1 - ms], "wb"); fclose (f);

  f = fopen (argv [2], "r");
  if (f == NULL) {
    printf ("Cannot open mix script '%s'\n", argv [2]);
    return (2); }

  while (1) {
    fgets (line, 250, f);
    if (feof (f)) break;
    sscanf (line, "%s %lf %ld\n", fn, &level, &org);
    mix (fn, pow (10.0, level / 20.0), org, stereo); }

  fclose (f);

  m = getmax ();
  if (m == -1.0) return (3);
  printf ("\n\nNormalizing from %lf\n", m);
  if (norm (argv [1], m, sr, stereo) != 0) return (4);

  remove (mn [ms]);
  remove (mn [1 - ms]);

  return (0); }
