/* mp3.c
 *
 * Copyright 2001-2002 Vesa Halttunen (Vesuri/dA JoRMaS)
 *
 * This file is part of JRm-core.
 *
 * JRm-core is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * JRm-core is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JRm-core; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include <SDL.h>
#ifdef WINDOWS
#include <Windows.h>
#else
#include <sys/time.h>
#endif
#include "mpglib/mpg123.h"
#include "mpglib/mpglib.h"
#include "mp3.h"

static struct mpstr mp;
static FILE *file;
static char *inbuf=NULL;
static int inbufsize, decoded, ret=MP3_NEED_MORE, starttimereset=0;
extern int quit;
#ifdef WINDOWS
extern DWORD starttc;
#else
extern struct timeval starttv;
extern struct timezone tz;
#endif

int openmp3(char *filename, int outbufsize, Uint8 *decodebuf, int skipms) {
  /* Open MP3 file */
  file=fopen(filename, "r");

  if(file) {
    int decode=skipms*1764/10+outbufsize;

    /* Initialize MP3 recoder */
    InitMP3(&mp);

    /* Read the input file */
    fseek(file, 0, SEEK_END);
    inbufsize=ftell(file);
    inbuf=(char *)calloc(inbufsize, 1);
    fseek(file, 0, SEEK_SET);
    fread(inbuf, 1, inbufsize, file);

    decoded=outbufsize;
    while(decode>0) {
      memmove(decodebuf+(decoded-outbufsize), decodebuf, outbufsize);
      decoded-=outbufsize;

      while(decoded<outbufsize) {
	int add=0;
	if(ret!=MP3_OK)
	  ret=decodeMP3(&mp, inbuf, inbufsize, decodebuf+decoded, 32768, &add);
	else
	  ret=decodeMP3(&mp, NULL, 0, decodebuf+decoded, 32768, &add);
	
	if(ret==MP3_OK)
	  decoded+=add;
	decode-=add;
      }
    }

    return 0;
  } else
    return -1;
}

/* Copies decoded data to the SDL buffer */
void playmp3(void *userdata, Uint8 *outbuf, int outbufsize) {
  Uint8 *decodebuf=(Uint8 *)userdata;

  if(starttimereset<1) {
    /* Get the accurate time when exactly two buffers are played - the
     * first one is empty */
#ifdef WINDOWS
    starttc=GetTickCount()+outbufsize*10/441/4;
#else
    gettimeofday(&starttv, &tz);
    
    starttv.tv_usec+=outbufsize*10000/441/4;
    if(starttv.tv_usec>999999) {
      starttv.tv_sec++;
      starttv.tv_usec-=1000000;
    }
#endif
    starttimereset++;
  }

  memcpy(outbuf, decodebuf, outbufsize);
  memmove(decodebuf, decodebuf+outbufsize, decoded-outbufsize);
  decoded-=outbufsize;

  while(decoded<outbufsize) {
    int add=0;

    if(ret!=MP3_OK)
      ret=decodeMP3(&mp, inbuf, inbufsize, decodebuf+decoded, 32768, &add);
    else
      ret=decodeMP3(&mp, NULL, 0, decodebuf+decoded, 32768, &add);
  
    if(ret==MP3_OK)
      decoded+=add;
  }
}

void closemp3() {
  /* Shutdown MP3 recoder */
  ExitMP3(&mp);

  if(file) {
    fclose(file);
    file=0;
  }
  if(inbuf!=NULL) {
    free(inbuf);
    inbuf=NULL;
  }
}
