/* copy.c
 *
 * Copyright 2001-2002 Vesa Halttunen (Vesuri/dA JoRMaS)
 *
 * This file is part of JRm-core.
 *
 * JRm-core is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * JRm-core is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JRm-core; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include "effect.h"
#include "copy.h"

#define getalpha(data) ((((data)&0xff000000)>>24)+1)
#define copynormal(data1, data2, alpha) (0xff000000|((((((data1)&0xff0000)*(alpha))>>8)&0xff0000)+(((((data2)&0xff0000)*(257-(alpha)))>>8)&0xff0000))|((((((data1)&0xff00)*(alpha))>>8)&0xff00)+(((((data2)&0xff00)*(257-(alpha)))>>8)&0xff00))|((((((data1)&0xff)*(alpha))>>8)&0xff)+(((((data2)&0xff)*(257-(alpha)))>>8)&0xff)))
#define copyaddpass1(data1, data2, alpha) (((data2)&0xfefefe)+((((((data1)&0xff0000)*alpha)>>8)&0xfe0000)|(((((data1)&0xff00)*alpha)>>8)&0xfe00)|(((((data1)&0xff)*alpha)>>8)&0xfe)))
#define copyaddpass2(temp) ((((temp)|((((((temp)&0x01010100)|0x80808000)-0x01010100)^0xffffffff)>>7))&0x00ffffff)|0xff000000)
#define copyoverwrite(data, alpha) (0xff000000|(((((data)&0xff0000)*(alpha))>>8)&0xff0000)|(((((data)&0xff00)*(alpha))>>8)&0xff00)|(((((data)&0xff)*(alpha))>>8)&0xff))

void copy_run(void *d, long time) {
  copy_data *data=(copy_data *)d;
  int sw, sh, dw, dh, alpha;

  /* Make sure the destination is not outside the buffer */
  if(data->destx>=data->dest->width)
    return;
  if(data->desty>=data->dest->height)
    return;

  sw=data->sourcewidth;
  sh=data->sourceheight;
  dw=data->destwidth;
  dh=data->destheight;
  alpha=data->alpha+1;
  if(alpha<1)
    alpha=1;
  else if(alpha>256)
    alpha=256;

  if(sw==dw && sh==dh) {
    /* Normal copy */
    int x, y, sdx, ddx;
    unsigned long *source, *dest, t1, t2, a;

    /* Make sure the area to be copied fits inside the destination buffer */
    if((data->destx+dw)>data->dest->width) {
      sw=data->dest->width-data->destx;
      dw=data->dest->width-data->destx;
    }
    if((data->desty+dh)>data->dest->height) {
      sh=data->dest->height-data->desty;
      dh=data->dest->height-data->desty;
    }

    /* Get local copies of the parameters */
    source=data->source->data+((data->sourcey)*data->source->width+data->sourcex);
    dest=data->dest->data+((data->desty)*data->dest->width+data->destx);
    sdx=data->source->width-sw;
    ddx=data->dest->width-dw;
    
    switch(data->mode) {
    case COPY_MODE_NORMAL:
      /* Copy the image over the destination */
      switch(alpha) {
      case 1:
	/* If alpha is zero do absolutely nothing */
	break;
      case 256:
	/* If given alpha is full use the alpha of the source image */
	for(y=0; y<sh; y++) {
	  for(x=0; x<sw; x++) {
	    t1=*source++;
	    t2=*dest;
	    a=getalpha(t1);
	    *dest++=copynormal(t1, t2, a);
	  }
	  source+=sdx;
	  dest+=ddx;
	}
	break;
      default:
	/* Otherwise the alpha of the source image must be multiplied */
	for(y=0; y<sh; y++) {
	  for(x=0; x<sw; x++) {
	    t1=*source++;
	    t2=*dest;
	    a=(getalpha(t1)*alpha)>>8;
	    *dest++=copynormal(t1, t2, a);
	  }
	  source+=sdx;
	  dest+=ddx;
	}
	break;
      }
      break;
    case COPY_MODE_ADD:
      /* Copy the image over the destination */
      switch(alpha) {
      case 1:
	/* If alpha is zero do absolutely nothing */
	break;
      case 256:
	/* If given alpha is full use the alpha of the source image */
	for(y=0; y<sh; y++) {
	  for(x=0; x<sw; x++) {
	    t1=*source++;
	    a=getalpha(t1);
	    t1=copyaddpass1(t1, *dest, a);
	    *dest++=copyaddpass2(t1);
	  }
	  source+=sdx;
	  dest+=ddx;
	}
	break;
      default:
	/* Otherwise the alpha of the source image must be multiplied */
	for(y=0; y<sh; y++) {
	  for(x=0; x<sw; x++) {
	    t1=*source++;
	    a=(getalpha(t1)*alpha)>>8;
	    t1=copyaddpass1(t1, *dest, a);
	    *dest++=copyaddpass2(t1);
	  }
	  source+=sdx;
	  dest+=ddx;
	}
	break;
      }
      break;
    case COPY_MODE_OVERWRITE:
      /* We're copying the image and not taking care of what's underneath */
      switch(alpha) {
      case 1:
	/* If alpha is zero we're just clearing the image */
	for(y=0; y<sh; y++) {
	  for(x=0; x<sw; x++)
	    *dest++=0;
	  dest+=ddx;
	}
	break;
      case 256:
	/* If we're doing full alpha just copy the image */
	for(y=0; y<sh; y++) {
	  for(x=0; x<sw; x++)
	    *dest++=*source++;
	  source+=sdx;
	  dest+=ddx;
	}
	break;
      default:
	/* Otherwise we have to do the alpha thong */
	for(y=0; y<sh; y++) {
	  for(x=0; x<sw; x++) {
	    t1=*source++;
	    a=(getalpha(t1)*alpha)>>8;
	    *dest++=copyoverwrite(t1, a);
	  }
	  source+=sdx;
	  dest+=ddx;
	}
	break;
      }
      break;
    }
  } else {
    /* Scale the image */
    unsigned long *source, *dest;
    double sx, sy, sdx, sdy;
    int swidth=data->source->width, dwidth=data->dest->width;
    int x, y, ddx;

    /* Make sure the area to be copied fits inside the destination buffer */
    if((data->destx+dw)>data->dest->width) {
      dw=data->dest->width-data->destx;
      sw*=((double)dw/(double)data->destwidth);
    }
    if((data->desty+dh)>data->dest->height) {
      dh=data->dest->height-data->desty;
      sh*=((double)dh/(double)data->destheight);
    }

    /* Get local copies of the parameters */
    source=data->source->data;
    dest=data->dest->data;

    /* Where to move? */
    sdx=(double)sw/(double)dw;
    sdy=(double)sh/(double)dh;
    ddx=dwidth-dw;

    if(alpha>1) {
      dest+=data->desty*dwidth+data->destx;
      if(data->interpolate) {
	sy=data->sourcey;
	for(y=data->desty; y<data->desty+dh; y++) {
	  int wy2=(int)(sy*256)%256, wy1=256-wy2;
	  int y1=(int)sy, y2=y1+1;
	  sx=data->sourcex;
	  for(x=0; x<dw; x++) {
	    int wx2=(int)(sx*256)%256, wx1=256-wx2;
	    int x1=(int)sx, x2=x1+1;
	    unsigned long t1=source[y1*swidth+x1], t2=source[y1*swidth+x2];
	    unsigned long t3=source[y2*swidth+x1], t4=source[y2*swidth+x2];
	    unsigned long t5=((((((t1&0xff000000)>>8)*wx1)&0xff000000)+
			       ((((t2&0xff000000)>>8)*wx2)&0xff000000))|
			      (((((t1&0xff0000)*wx1)>>8)&0xff0000)+
			       ((((t2&0xff0000)*wx2)>>8)&0xff0000))|
			      (((((t1&0xff00)*wx1)>>8)&0xff00)+
			       ((((t2&0xff00)*wx2)>>8)&0xff00))|
			      (((((t1&0xff)*wx1)>>8)&0xff)+
			       ((((t2&0xff)*wx2)>>8)&0xff)));
	    unsigned long t6=((((((t3&0xff000000)>>8)*wx1)&0xff000000)+
			       ((((t4&0xff000000)>>8)*wx2)&0xff000000))|
			      (((((t3&0xff0000)*wx1)>>8)&0xff0000)+
			       ((((t4&0xff0000)*wx2)>>8)&0xff0000))|
			      (((((t3&0xff00)*wx1)>>8)&0xff00)+
			       ((((t4&0xff00)*wx2)>>8)&0xff00))|
			      (((((t3&0xff)*wx1)>>8)&0xff)+
			       ((((t4&0xff)*wx2)>>8)&0xff)));
	    *dest++=((((((t5&0xff000000)>>8)*wy1)&0xff000000)+
		      ((((t6&0xff000000)>>8)*wy2)&0xff000000))|
		     (((((t5&0xff0000)*wy1)>>8)&0xff0000)+
		      ((((t6&0xff0000)*wy2)>>8)&0xff0000))|
		     (((((t5&0xff00)*wy1)>>8)&0xff00)+
		      ((((t6&0xff00)*wy2)>>8)&0xff00))|
		     (((((t5&0xff)*wy1)>>8)&0xff)+
		      ((((t6&0xff)*wy2)>>8)&0xff)));
	    sx+=sdx;
	  }
	  sy+=sdy;
	  dest+=ddx;
	}
      } else {
	sy=data->sourcey;
	for(y=data->desty; y<data->desty+dh; y++) {
	  sx=data->sourcex;
	  for(x=data->destx; x<data->destx+dw; x++) {
	    *dest++=source[(int)sy*swidth+(int)sx];
	    sx+=sdx;
	  }
	  sy+=sdy;
	  dest+=ddx;
	}
      }    
    }
  }
}

void copy_free(effect *effect) {
  free(effect->data);
}

void copy_new(effect *effect) {
  effect->run=copy_run;
  effect->new=copy_new;
  effect->parameters=NULL;
  effect->freee=copy_free;
  effect->data=(copy_data *)calloc(sizeof(copy_data), 1);
}
