/* crap_video.c - simple support for libggi or svgalib
   Copyright (C) 2000 Tijs van Bakel.
   Tijs van Bakel <smoke@casema.net>, 
 
 This file is part of crapstation, a collection of demobuilder tools.

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include "config.h"
#include "crap_video.h"
#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>

#ifdef USE_SVGALIB
#include <vga.h>
#include <vgakeyboard.h>
#include <sys/io.h>
#include <sys/perm.h>
#endif /* USE_SVGALIB */

#ifdef USE_LIBGGI
#include <ggi/ggi.h>
#include <ggi/misc.h>

ggi_visual_t _crap_video_visual;
sint32 _crap_video_rp_x, _crap_video_rp_y; /* ray position goals for waitretrace */
int _crap_video_use_dbuf = 0;
int _crap_video_have_dbuf = 0;
const ggi_directbuffer *_crap_video_dbuf;
int _crap_video_wait_vr = 0;

#endif /* USE_LIBGGI */

Image* _crap_video_root_image;
uint8* _crap_video_buffer;
uint8* _crap_video_dbuf_ptr;

struct timeval _crap_video_start, _crap_video_end, _crap_video_mark;
long _crap_video_frametime;  // in usec;

int _crap_video_width, _crap_video_height, _crap_video_bpp;

int _last_key_press;

uint8 _palette[256*3];
int _update_palette;

#ifdef USE_SVGALIB
int
crap_svgalib_init( uint32 width, uint32 height )
{
  _last_key_press = 0;
  _update_palette = 1;
  memset ( _palette, 0, 256*3 );

  iopl(3);

  vga_init();
  vga_setlinearaddressing();
  vga_setmode(5);
  
  _crap_video_dbuf_ptr = vga_getgraphmem();
  
  _crap_video_width = 320;
  _crap_video_height = 200;
  _crap_video_bpp = 8;
  
  memset ( _crap_video_dbuf_ptr, 0, _crap_video_width *_crap_video_height );
  
  _crap_video_buffer = malloc( 320*200 );

  _crap_video_root_image = (Image*) malloc ( sizeof(Image) );
  _crap_video_root_image->width = crap_video_get_width();
  _crap_video_root_image->height = crap_video_get_height();
  _crap_video_root_image->stride = crap_video_get_width();
  _crap_video_root_image->buffer = _crap_video_buffer;
  
  _crap_video_frametime = 1000000/70; // 70Hz   -- warp-tmt@dds.nl
  gettimeofday(&_crap_video_start,NULL);
  gettimeofday(&_crap_video_mark,NULL);

  return 0;
}
#endif /* USE_SVGALIB */

#ifdef USE_LIBGGI
int
crap_ggi_init( uint32 width, uint32 height )
{
  if (ggiInit () != 0)
    return -1;
  if ((_crap_video_visual = ggiOpen (NULL)) == NULL)
    return -1;
  
  _crap_video_width = width;
  _crap_video_height = height;
  _crap_video_bpp = GT_8BIT;
  
  if (ggiSetGraphMode ( _crap_video_visual,
		        _crap_video_width, _crap_video_height,
		        GGI_AUTO, GGI_AUTO,
                        _crap_video_bpp ) < 0)
    {
      printf("Couldn't set 8bpp mode, trying palemu target..\n");
      if ( ggiClose ( _crap_video_visual ) )
	{
	  printf("could not close visual.\n");
	  return -1;
	}
      if ( ( _crap_video_visual = ggiOpen ( "display-palemu" , NULL ) ) == NULL)
	{
	  printf("could not open palemu visual.\n");
	  return -1;
	}
      if (ggiSetGraphMode ( _crap_video_visual,
			    _crap_video_width, _crap_video_height,
			    _crap_video_width, _crap_video_height,
			    _crap_video_bpp ) < 0)
	{
	  printf("could not set requested graphics mode in palemu visual.\n");
	  return -1;
	}
    }
       
  ggiSetFlags (_crap_video_visual, GGIFLAG_ASYNC);

  if (ggiMiscInit () != 0)
    return -1;
  
  if (ggiMiscAttach (_crap_video_visual) < 0)
    return -1;

  _crap_video_have_dbuf = 1;
  
  if (!(_crap_video_dbuf = ggiDBGetBuffer (_crap_video_visual, 0)))
    _crap_video_have_dbuf = 0;
  
  if ( _crap_video_have_dbuf )
    if (!( _crap_video_dbuf->type & GGI_DB_SIMPLE_PLB))
      _crap_video_have_dbuf = 0;
  
  if ( _crap_video_have_dbuf )
    {
      _crap_video_buffer = (uint8*) malloc ( _crap_video_width * _crap_video_height );
      _crap_video_dbuf_ptr = (uint8*) _crap_video_dbuf->write;
      memset ( _crap_video_buffer, 0, _crap_video_width * _crap_video_height );
    }
  else
    {
      _crap_video_buffer = (uint8*) malloc ( _crap_video_width * _crap_video_height );
      memset ( _crap_video_buffer, 0, _crap_video_width * _crap_video_height );

      ggiPutBox ( _crap_video_visual,
		  0, 0, crap_video_get_width(), crap_video_get_height(), _crap_video_buffer );
    }
  
  _crap_video_rp_x = GGI_RP_DONTCARE;
  _crap_video_rp_y = GGI_RP_SYNC;

  if ( ggiWaitRayPos ( _crap_video_visual, &_crap_video_rp_x, &_crap_video_rp_y ) != -1 )
    _crap_video_wait_vr = 1;

  _crap_video_frametime = 1000000/70; // 70Hz   -- warp-tmt@dds.nl
  gettimeofday(&_crap_video_start,NULL);
  gettimeofday(&_crap_video_mark,NULL);

  _crap_video_root_image = (Image*) malloc ( sizeof(Image) );
  _crap_video_root_image->width = crap_video_get_width();
  _crap_video_root_image->height = crap_video_get_height();
  _crap_video_root_image->stride = crap_video_get_width();
  _crap_video_root_image->buffer = _crap_video_buffer;
  
  return 0;
}
#endif /* USE_LIBGGI */

int
crap_video_init( uint32 width, uint32 height )
{
#ifdef USE_SVGALIB
  return crap_svgalib_init ( width, height );
#endif
#ifdef USE_LIBGGI
  return crap_ggi_init ( width, height );
#endif /* USE_LIBGGI */
  return -1;
}

Image*
crap_video_root_image()
{
  return _crap_video_root_image;
}

void
crap_video_setcolor ( uint8 color, uint8 r, uint8 g, uint8 b )
{
  _palette[color*3] = r;
  _palette[color*3+1] = g;
  _palette[color*3+2] = b;
  
  _update_palette = 1;
}

#ifdef USE_LIBGGI
/* timeval_subtract is taken verbatim from the GNU libc documention -- warp-tmt@dds.nl */
int
crap_video_timeval_subtract (struct timeval *result, struct timeval *x, struct timeval *y)
{
  int nsec;
  
  // Perform the carry for the later subtraction by updating Y. 
  if (x->tv_usec < y->tv_usec)
    {
      nsec = (y->tv_usec - x->tv_usec) / 1000000 + 1;
      y->tv_usec -= 1000000 * nsec;
      y->tv_sec += nsec;
    }
  if (x->tv_usec - y->tv_usec > 1000000)
    {
      nsec = (y->tv_usec - x->tv_usec) / 1000000;
      y->tv_usec += 1000000 * nsec;
      y->tv_sec -= nsec;
    }
  
  // Compute the time remaining to wait. `tv_usec' is certainly positive. 
  result->tv_sec = x->tv_sec - y->tv_sec;
  result->tv_usec = x->tv_usec - y->tv_usec;
  
  // Return 1 if result is negative. 
  return x->tv_sec < y->tv_sec;
}

void 
crap_video_wait(void)
{
  struct timeval _crap_video_t;
  int td = 0;
  
  while ( td < _crap_video_frametime )
    {
      gettimeofday ( &_crap_video_t, NULL );
      crap_video_timeval_subtract ( &_crap_video_end, &_crap_video_t, &_crap_video_mark );
      td = _crap_video_end.tv_usec;
    }
  gettimeofday ( &_crap_video_mark, NULL );
}
#endif /* USE_LIBGGI */

void
crap_video_waitretrace()
{
#ifdef USE_SVGALIB
  vga_waitretrace();
#endif /* USE_SVGALIB */

#ifdef USE_LIBGGI
  if(_crap_video_wait_vr) 
    ggiWaitRayPos ( _crap_video_visual, &_crap_video_rp_x, &_crap_video_rp_y);
  else 
    crap_video_wait(); 

  ggiFlush ( _crap_video_visual );
#endif /* USE_LIBGGI */
}

int
crap_video_keypress()
{
#ifdef USE_SVGALIB
  return ( _last_key_press = vga_getkey() );
#endif /* USE_SVGALIB */
#ifdef USE_LIBGGI
  return ( ggiKbhit( _crap_video_visual ) );
#endif /* USE_LIBGGI */
  return 0;
}

int
crap_video_getkey()
{
#ifdef USE_SVGALIB
  if ( _last_key_press == 0 )
    _last_key_press = vga_getkey();
  
  if ( _last_key_press == 27 )
    {
      _last_key_press = 0;
      return CRAP_KEY_ESCAPE;
    }

  _last_key_press = 0;
  return CRAP_KEY_NONE;
#endif /* USE_SVGALIB */
  
#ifdef USE_LIBGGI
  return ( ggiGetc ( _crap_video_visual ) == 27 ) ? CRAP_KEY_ESCAPE : CRAP_KEY_NONE;
#endif /* USE_LIBGGI */

  return 0;
}

int
crap_video_get_width()
{
  return _crap_video_width;
}

int
crap_video_get_height()
{
  return _crap_video_height;
}

void
crap_video_blit ( Image* image )
{
#ifdef USE_LIBGGI
  ggi_color pal[768];
#endif /* USE_LIBGGI */
  
  uint32* d;
  uint32* s;
  int i;
  int count = 16000;

  if ( _update_palette )
    {
      _update_palette = 0;

#ifdef USE_SVGALIB
      outb( 0, 0x3c8 );
      for ( i = 0; i < 768; i++ )
	outb ( _palette[i], 0x3c9 );
#endif /* USE_SVGALIB */
      
#ifdef USE_LIBGGI
      for ( i = 0; i < 256; i++ )
	{
	  pal[i].r = (_palette[i*3]) << 10;
	  pal[i].g = (_palette[i*3+1]) << 10;
	  pal[i].b = (_palette[i*3+2]) << 10;
	}
      ggiSetPalette ( _crap_video_visual, 0, 256, pal );
#endif /* USE_LIBGGI */
    }

#ifdef USE_LIBGGI
  if ( !_crap_video_have_dbuf )
    {
      ggiPutBox ( _crap_video_visual,
		  0, 0, crap_video_get_width(), crap_video_get_height(),
		  image->buffer );
    }
  else
    {
      d = (uint32*) _crap_video_dbuf_ptr;
      s = (uint32*) image->buffer;
      
      while (count--)
	*d++ = *s++;
    }

  return;
#endif /* USE_LIBGGI */

#ifdef USE_SVGALIB 
  d = (uint32*) _crap_video_dbuf_ptr;
  s = (uint32*) image->buffer;
  
  while (count--)
    *d++ = *s++;
#endif
}

void
crap_video_done()
{
#ifdef USE_SVGALIB
  vga_setmode(0);
#endif /* USE_SVGALIB */
  
#ifdef USE_LIBGGI
  ggiClose ( _crap_video_visual );
#endif /* USE_LIBGGI */
}

/* palette: RGB,RGB,RGB,.. */
void
crap_video_setcolors ( uint8 color, int length, uint8* palette )
{
  int i;

  for ( i = 0; i < length; i++ )
    {
        uint8 r,g,b;
        r = *palette++;
        g = *palette++;
        b = *palette++;
	crap_video_setcolor( i+color, r,g,b );
    }
}

uint8 * crap_video_getcurrentpalette()
{
   uint8 * colors = (uint8*) malloc( 768 );
   memcpy( colors, _palette, 768 );
   return colors;
}

/* palette: RGB,RGB,RGB,.. */
void crap_video_fadecolors ( uint8 color, int length, uint8* palette,float f )
{
  int i;

  for ( i = 0; i < length; i++ )
    {
        uint8 r,g,b;
        r = *palette++;
        g = *palette++;
        b = *palette++;
	crap_video_setcolor( i, r*f,g*f,b*f );
    }
}
