//
// Java Producer class for OpenPTC 1.0 Java Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licenced under the GNU LGPL
// See http://www.gnu.org/copyleft/lgpl.html for details
//

// package name
package ptc.java;

// import classes
import ptc.Area;
import ptc.Error;
import ptc.Format;
import ptc.Palette;
import java.util.Vector;
import java.util.Enumeration;
import java.awt.image.ImageConsumer;
import java.awt.image.ImageProducer;



public class Producer implements ImageProducer 
{
    public Producer(int width,int height,Format format)
    {
        // setup data
        this.width  = width;
        this.height = height;
        this.format = format.copy();

        // create color model
        model = new Model(format);
    
        // create consumer list
        consumers = new Vector();
    }

    public synchronized void addConsumer(ImageConsumer ic) 
    {
        // check if already registered
     if (consumers.contains(ic)) return;

        // register image consumer
        consumers.addElement(ic);

        // clear properties
        ic.setProperties(null);

        // set image dimensions
        ic.setDimensions(width,height);

        // set image color model
        ic.setColorModel(model.model());
        
        // set image consumer hints
        ic.setHints(ImageConsumer.TOPDOWNLEFTRIGHT | ImageConsumer.SINGLEPASS | ImageConsumer.SINGLEFRAME | ImageConsumer.COMPLETESCANLINES);
    }

    public synchronized boolean isConsumer(ImageConsumer ic) 
    {
        // check if consumer is registered
        return consumers.contains(ic);
    }

    public synchronized void removeConsumer(ImageConsumer ic) 
    {
        // remove image consumer
        consumers.removeElement(ic);
    }

    public synchronized void startProduction(ImageConsumer ic) 
    {
        // add consumer
        addConsumer(ic);
    }

    public synchronized void requestTopDownLeftRightResend(ImageConsumer ic) 
    {
        // ignore resend request
    }

    public void load(Object pixels,int width,int height,int pitch,Format format,Palette palette) throws Error
    {
        // load pixels to image consumers
        load(pixels,width,height,pitch,format,palette,new Area(0,0,width,height),new Area(0,0,width,height));
    }

    public void load(Object pixels,int width,int height,int pitch,Format format,Palette palette,Area source,Area destination) throws Error
    {
        // check pitch is a multiple of bytes per pixel
        if (pitch%format.bytes()!=0) throw new Error("invalid pitch");

        // calculate stride from pitch
        final int stride = pitch / format.bytes();

        // setup the default hints mask
        int hints = ImageConsumer.TOPDOWNLEFTRIGHT | ImageConsumer.SINGLEPASS | ImageConsumer.SINGLEFRAME;

        // check for complete scanlines hint
        if (destination.left()==0 && destination.width()==this.width) hints |= ImageConsumer.COMPLETESCANLINES;

        // color model
        Model model;

        // setup color model object
        if (this.format.indexed() && format.indexed()) model = this.model;
        else
        {
            // check if indexed
            if (format.indexed()) model = new Model(format,palette);
            else model = new Model(format);
        }
  
        // iteratate through all consumers
        Enumeration enum = consumers.elements();
        while (enum.hasMoreElements()) 
        {   
            // get image consumer object from list
            ImageConsumer ic = (ImageConsumer) enum.nextElement();

            // set image hints
            ic.setHints(hints);
            
            // check pixels type
            if (pixels instanceof int[])
            {
                // copy integer pixel data to image
                ic.setPixels(destination.left(),destination.top(),destination.width(),destination.height(),model.model(),(int[])pixels,width*source.top()+source.left(),stride);
            }
            else if (pixels instanceof byte[])
            {
                // copy char pixel data to image
                ic.setPixels(destination.left(),destination.top(),destination.width(),destination.height(),model.model(),(byte[])pixels,width*source.top()+source.left(),stride);
            }
            else
            {
                // unsupported bits per pixel
                throw new Error(format.bits() + " bits per pixel is not supported");
            }

            // notify image consumer that the frame is done
            ic.imageComplete(ImageConsumer.SINGLEFRAMEDONE);
        }
    }

    public void save(Object pixels,int width,int height,int pitch,Format format,Palette palette) throws Error
    {
        // error message
        throw new Error("producer save not implemented");
    }

    public void save(Object pixels,int width,int height,int pitch,Format format,Palette palette,Area source,Area destination) throws Error
    {
        // error message
        throw new Error("producer save area not implemented");
    }

    public void palette(Palette palette)
    {
        // create color model
        model = new Model(format,palette);

        // iteratate through all consumers
        Enumeration enum = consumers.elements();
        while (enum.hasMoreElements()) 
        {   
            // get image consumer object from list
            ImageConsumer ic = (ImageConsumer) enum.nextElement();

            // set image consumer model
            ic.setColorModel(model.model());
        }
    }

    // producer data
    private int width;
    private int height;
    private Model model;
    private Format format;
    private Vector consumers;
}
