#include "rotozoom.h"

#include "lookup.h"
#include <iostream>
#include <cmath>

using namespace std;

/*
	some math rules:
	
		r = sqrt(x^2 + y^2)
		x = cos(a) * r
		y = sin(a) * r
		
		cos(a - x) = cos a * cos x + sin a * sin x
		sin(a - x) = sin a * cos x - cos a * sin x

*/


/** 
 *	Initialize surfaces so we dont have to meddle with after calculations anymore
 */
void Rotozoom::initializeSurfaces(SDL_Surface *pic, SDL_Surface *nmap) {

	if (pic->w != nmap->w || pic->h != nmap->h) {
		cerr << "Incorrect surfaces" << endl;
		return;
	}
	
	SDL_LockSurface(pic);
	SDL_LockSurface(nmap);
	
	Uint32 *picPixels = (Uint32 *) pic->pixels;
	Uint32 *nmapPixels = (Uint32 *) nmap->pixels;
	
	// initialize arrays
	d_picture = new color[d_picWidth * d_picWidth];
	d_normalMap = new vector3[d_picWidth * d_picWidth];

	if (d_picture == 0 || d_normalMap == 0) {
		cerr << "Unable to allocate memory properly" << endl;
		return;
	}

	size_t idx = 0;

	// iterate through normal map and picture surfaces
	// and transform the information into useful stuff
	for (int y = 0; y < d_picWidth; ++y) {
		for (int x = 0; x < d_picWidth; ++x) {
		
			// 32-bit color value
			Uint32 colorVal = *(picPixels + (x + y * (pic->pitch >> 2)));
			Uint32 vectorVal = *(nmapPixels + (x + y * (nmap->pitch >> 2)));

			Uint8 
				r, g, b,
				bx, by, bz;
				
			d_gfx.convert(pic->format, colorVal, r, g, b);
			d_gfx.convert(nmap->format, vectorVal, bx, by, bz);
			
			// initialize picture information			
			d_picture[idx] = d_gfx.color(r, g, b);
			
			// initialize normal map
			d_normalMap[idx] = 
				vector3(
						-1.0f + (1.0 / 128.0f) * bx, 
						-1.0f + (1.0 / 128.0f) * bz, 	// switch components because 
						-1.0f + (1.0 / 128.0f) * by		// of our perspective
					);
					
			d_normalMap[idx].y *= -1;
	
			++idx;
		}
	}

	SDL_UnlockSurface(pic);
	SDL_UnlockSurface(nmap);
}


/**
 *	Draw the rotozoom
 */
void Rotozoom::draw() {

	SDL_Surface *primary = d_gfx.surface();

	// setup surfaces and pointers
	SDL_LockSurface(primary);
	
	Uint32 *pxl = 0; 
	Uint32 *pxlBase = (Uint32 *) primary->pixels;
	
	vector3 normal(0, -1, 0);
	vector3 position(0, 0, 0);	

	float 
		u = 0, 
		v = 0;

	float 
		min_cos = lu_cos(d_angle),
		min_sin = lu_sin(d_angle);

	float 
		plus_cos = lu_cos(-d_angle),
		plus_sin = lu_sin(-d_angle);
	
	// for each pixel on screen
	for (size_t py = 0; py < d_gfx.height(); ++py) {
		
		pxl = pxlBase + (py * (primary->pitch >> 2));
	
		// iterate every x
		for (size_t px = 0; px < d_gfx.width(); ++px) {

			int 
				x = px - d_gfx.width() / 2,
				y = py - d_gfx.height() / 2;

			if (x == 0) x++;
			if (y == 0) y++;

			float radius = sqrt(x * x + y * y);
			
			float 
				cos_a = x / radius,
				sin_a = y / radius;
		
			float
				new_cos = cos_a * min_cos + sin_a * min_sin,
				new_sin = sin_a * min_cos - cos_a * min_sin;
		
			u = (new_cos * radius) / d_distance;
			v = (new_sin * radius) / d_distance;
			
			// wrap to within picture
			u = wrap(u, d_picWidth); 
			v = wrap(v, d_picWidth);

			// texture lookup			
			union color tColor =
				d_picture[	
						static_cast<int>(floor(u)) + 
						static_cast<int>(floor(v)) * d_picWidth
					];
					
			// normal lookup
			vector3 &N =
				d_normalMap[	
						static_cast<int>(floor(u)) + 
						static_cast<int>(floor(v)) * d_picWidth
					];
					
			vector3 rotN(N.x, N.y, N.z);
			rotN.x = N.x * plus_cos + N.z * plus_sin;
			rotN.z = N.z * plus_cos - N.x * plus_sin;

			// light color at position
			position.x = px;
			position.y = -5.0 * N.y;
			position.z = py;			
			union color lColor = d_lightModel.colorAt(d_gfx, position, rotN);
			
			// do multiply blend
			union color mulColor;
			mulColor.r = (lColor.r * tColor.r) >> 8;
			mulColor.g = (lColor.b * tColor.g) >> 8;
			mulColor.b = (lColor.g * tColor.b) >> 8;

			// copy pixel from picture to surface
			*(pxlBase + (py * (primary->pitch >> 2) + px)) = mulColor.rgba;
		}
	}	

	// release surfaces
	SDL_UnlockSurface(primary);

}


