/*
	modulo.cpp

	This program calculates (a modulo n), where (a) is a positive decimal number
	entered by the user with up to NUMBER_OF_DIGITS digits and without a decimal
	comma, and (n) is an integer. Since the result is stored as unsigned int
	(which I assume to be 16 bit, so the maximum value is 65536), the maximum
	possible NUMBER_OF_DIGITS is about 7200. If you think it's necessary for your
	purposes, you can easily adapt the code so that it works with unsigned long,
	allowing to use even more digits.

	(a) is entered as a string, which is the reason why its NUMBER_OF_DIGITS can
	(in theory) be infinite.

	If (n == 9), (a modulo n) is calculated by computing the sum of the digits
	of the input, storing it in an unsigned int and then performing the standard
	operator '%'. This is possible because the sum of the digits of any number
	that can be divided by 9 can itself be divided by 9, so the results for
	modulo 9 are the same.

	For any other (n), the program simply uses the obvious formula:
	(a modulo n) == (a) - (n) * (a / n), where '/' stands for integer division.

	If SPEED is defined, this program will use the function modulo_batch instead
	of modulo and divide, which will slightly improve the speed of
	execution.
	
	If SPEED is not defined and DEBUG is defined, the built-in '/' and '%'
	operators of C will be used as well so that you can compare the results.
	However, this limits the size of the string since we need the function atol
	of stdlib.h for conversion. (atol is a variant of atoi with the difference
	being that it converts the input string to a long instead of an int.) Yet
	there is no need to reduce NUMBER_OF_DIGITS as long as you simply don't enter
	too high numbers on the prompt.

	In contrast to the functions of the standard library, which only work with
	numbers that fit to the range of unsigned long, my algorithm is supposed
	to work with larger numbers as well. For example, try the number
	4294967296. 4294967296 = 2^32; therefore it occupies 33 bits, where only
	bit 32 is set, while the lower bits are all zero. But since the largest
	standard C data type, unsigned long, has only 32 bits, assigning the value
	4294967296 to an unsigned long will have the same effect as assigning zero.
	Therefore the functions of the standard library will produce utterly wrong
	results. By contrast, my functions do return the correct results.

	By the way, my function divide1 might be an interesting alternative to atol.
	Unlike atol, divide1 only works with positive numbers and doesn't accept
	spaces and tabs in the source string, so it's probably quite a bit faster than
	atol.

	This program was written by Claus D. Volko a.k.a. Adok/Hugi.

	History:
	* 1.00 (June 14, 2003): first version, working only with (1 <= n <= 9); the
	  algorithm is based on the assumption that for any (n <= 9), (a modulo n) is
	  equal to (9 - n) * (a / 9) + (a modulo 9).
	* 1.10 (June 15, 2003): modulo and modulo_batch now accept any (n) as the
	  parameter. However, for any (n > 9) they use divide1, a function similar to
	  atol, in order to convert the input string to an unsigned long; afterwards 
	  they perform a standard '%'. For this reason, my functions are still only
	  really useful if (1 <= n <= 9) because if this condition is met, you can use
	  them with very big numbers (a). However, for any (n > 9), the maximum value
	  for (a) is the maximum value of unsigned long, i.e. 2^32 - 1 = 4294967295.
	* 1.15 (June 15, 2003): I've created a function that allows to substract an
	  unsigned int from a string-encoded integer so it's no longer necessary to
	  convert the input string using divide1; divide1 is only used to convert the
	  result of substraction to unsigned int. This vastly expands the range of (a)
	  in which (a modulo n) can be performed for any (n > 9). For this reason, my
	  functions are now also superior to standard '%' for any (n > 9).
	* 1.20 (June 16th, 2003): Since the assumption for (1 <= n <= 9) which v 1.00
	  was based on has proven not always to produce the correct results, it has
	  been dropped. Now for all (n) except (n == 9), the same algorithm is used,
	  that is, the one that was introduced in v 1.15 for any (n > 9). So there is
	  no reason to differenciate between (1 <= n <= 9) and (n > 9) any more.

	In theory, this version ought to provide correct results for *any* input
	value. Let me explain why:

	The core code line of the modulo function is:

	output = divide1 (substract (outstring, input, output, n * (divide (input, n))));

	The divide function returns an unsigned long value. If input divided by n
	returned a value that didn't fit an unsigned long, it would therefore only
	return the last 9 to 10 pre-decimal-comma digits (to be more precise: the
	operation "modulo 2^32" would be performed). (n) multiplied with this number
	is exactly the same as multiplying (n) with the correct result of division
	and then "cutting" the result of multiplication so that it fits the range of
	unsigned long.

	Then the difference between input and the result of multiplication is
	computed. By means of divide1, it is stored in an unsigned long. So if it
	didn't fit an unsigned long, it would once again be "cut".

	The reason why we can simply "cut" the results of our operations is that it
	makes no difference. Let's take a look at the formula:
	
	(a modulo n) == (a) - (n) * (a / n)

	Since (a modulo n) < (n), this is the same as:

	(a - n * (a / n)) modulo (x)

	for any (x >= n). This is equal to:

	(a - (n * (a / n)) modulo (x)) modulo (x)

	This is exactly how this version of modulo and modulo_batch works. x is
	the largest possible unsigned long number + 1, i.e. 2^32. Since (n) is an
	unsigned int with the largest possible number being 2^16 - 1, the
	condition (x => n) is always met. It would even be met if (n) were an
	unsigned long itself.

	The latest version of this program is (probably) available at my homepage:
	http://www.hugi.de/adok/
*/


// #define SPEED
// #define DEBUG


#define NUMBER_OF_DIGITS 100
#define TEST_MAX 22


#include <stdio.h>
#include <stdlib.h>


unsigned long divide (unsigned char *input, unsigned int divisor)
{
	unsigned int output	= 0,
	    	     temp	= 0,
	    	     temp1;

	while (*input >= '0' && *input <= '9')
	{
		output *= 10;
		temp += (unsigned int) *input - '0';
		output += temp1 = temp / divisor;		
		temp -= temp1 * divisor;
		temp *= 10;
		input ++;

		/*
		   The code lines
		   
			temp -= temp1 * divisor;
			temp *= 10;
		   
		   could also be written as:
			temp = (temp % divisor) * 10;
			
		   Then temp1 would become obsolete.
		   But using an extra temporary variable temp1 instead of
		   performing an additional modulo is probably faster.
		*/
	}

	return output;
}


unsigned long divide1 (unsigned char *input)
{
	unsigned int output	= 0;

	while (*input >= '0' && *input <= '9')
	{
		output *= 10;
		output += (unsigned int) *input - '0';
		input ++;
	}
	
	return output;
}


unsigned int modulo9 (unsigned char *input)
{
	unsigned int output	= 0;

	while (*input >= '0' && *input <= '9')
	{
		output += (unsigned int) *input - '0';
		input ++;
	}
	
	return output % 9;
}


unsigned char *substract (unsigned char *output, unsigned char *input,
			  unsigned int len, unsigned int n)
{
	signed int temp;
	unsigned int carry	= 0;

	output [len] = '\0';

	do
	{
		len --;

		temp = (signed int) input [len] - '0' - n % 10 - carry;
		if (temp < 0)
		{
			temp += 10;
			carry = 1;
		}
		else
		{
			carry = 0;
		}
		output [len] = (unsigned char) temp + '0';
		n /= 10;		
	} while (len);

	return output;
}


#ifdef SPEED

void modulo_batch (unsigned char *input, unsigned int *inout_n)
{
	unsigned long output;

	unsigned int  result_modulo9	= modulo9 (input),
		      i			= 0,
		      len		= 0,
		      temp;

	unsigned char *outstring;

	while (input [len] >= '0' && input [len] <= '9')
	{
		len ++;
	}

	outstring = new char [len + 1];

	while (inout_n [i])
	{
		if (inout_n [i] == 9)
		{
			inout_n [i] = result_modulo9;
		}
		else
		{
			temp = inout_n [i];
			inout_n [i] = divide1 (substract (outstring, input, len,
				       inout_n [i] * (divide (input, inout_n [i]))));
 			inout_n [i] %= temp;
 		}
		i ++;
	}

	delete outstring;
}


void main ()
{
	unsigned char input [NUMBER_OF_DIGITS];

	unsigned int i,
		     n [TEST_MAX + 1];

	do
	{
		for (i = 1; i <= TEST_MAX; i ++)
		{
			n [i - 1] = i;
		}
		n [TEST_MAX] = 0;

		printf ("Enter the number (0 for quit): ");
		scanf ("%s", input);

		modulo_batch (input, n);

		for (i = 1; i <= TEST_MAX; i++)
		{
			printf ("%s modulo %d equals: %u\n", 
				input, i, n [i - 1]);
		}
		printf ("\n");
	} while (input [0] != '0');
}


#else


unsigned int modulo (unsigned char *input, unsigned int n)
{
	unsigned long output;
	unsigned char *outstring;

	if (n == 9)
	{
		output = modulo9 (input);
	}
	else
	{
  		output = 0;
  		
		while (input [output] >= '0' && input [output] <= '9')
		{
			output ++;
		}

		outstring = new char [output + 1];
		output = divide1 (substract (outstring, input, output, n * (divide (input, n))));
		output %= n;
		delete outstring;
	}

	return (output);
}


void main ()
{
	unsigned char input [NUMBER_OF_DIGITS];
	unsigned int i;

	do
	{
		printf ("Enter the number (0 for quit): ");
		scanf ("%s", input);

		for (i = 1; i <= TEST_MAX; i++)
		{
  #ifdef DEBUG			
			printf ("%s modulo %d equals: %u "
			        "(%lu modulo %d equals: %u)\n", 
				input,  i,  modulo (input, i),
				(unsigned) atol (input),  i,  (unsigned) atol (input) % i);
  #else
			printf ("%s modulo %d equals: %u\n", 
				input, i, modulo (input, i));
  #endif				
		}
		printf ("\n");
	} while (input [0] != '0');
}

#endif
