// Dynamic Huffman decompressor
//
// Fabian Giesen 24.08.98

#include <stdio.h>
#include <string.h>

#define OUTBUFLEN 1024

typedef unsigned long ulong;

typedef struct hnode {
          int  count, orgval, index;
          struct hnode *child[2], *parent;
        } huffnode;

ulong     count[257], totcount;
huffnode *hufftree, *huffroot;
int       hufflen, symsused;
int       bitbuf=0, bitsfull=0;
char      outbuf[OUTBUFLEN];
int       outcp=0;

void read_statistics(FILE *from)
{
  fread(&count, 4, 256, from);

  count[256]=1;
};

void create_huffman_tree()
{
  int       i, j, huffind, mincount, *allowed;
  huffnode *min1, *min2;

  for (i=0, symsused=0; i<257; i++) if (count[i]) symsused++;

  hufflen=(symsused<<1)-1; huffind=0;
  hufftree=new huffnode[hufflen];
  allowed=new int[hufflen];

  for (i=0; i<257; i++)
  {
    if (count[i])
    {
      memset(&hufftree[huffind], 0, sizeof(huffnode));

      allowed[huffind]=1;
      hufftree[huffind].orgval=i;
      hufftree[huffind].index=huffind;
      hufftree[huffind++].count=count[i];
    };
  };

  for (i=huffind; i<hufflen; i++)
  {
    mincount=0x7FFFFFFF; min1=NULL; min2=NULL;

    for (j=0; j<huffind; j++)
    {
      if ((hufftree[j].count<=mincount) && (allowed[j]))
      {
        min1=&hufftree[j]; mincount=hufftree[j].count;
      };
    };

    allowed[min1->index]=0;

    mincount=0x7FFFFFFF;

    for (j=0; j<huffind; j++)
    {
      if ((hufftree[j].count<=mincount) && (allowed[j]))
      {
        min2=&hufftree[j]; mincount=hufftree[j].count;
      };
    };

    allowed[min2->index]=0;

    min1->parent=min2->parent=&hufftree[huffind];

    hufftree[huffind].count=min1->count+min2->count;
    hufftree[huffind].orgval=-1;
    hufftree[huffind].index=huffind;
    hufftree[huffind].child[0]=min1;
    hufftree[huffind].child[1]=min2;
    hufftree[huffind].parent=NULL;
    allowed[huffind]=1;

    huffind++;
  };

  huffroot=&hufftree[huffind-1];
};

void writeb(char b, FILE *out)
{
  outbuf[outcp++]=b;

  if (outcp==OUTBUFLEN)
  {
    fwrite(&outbuf, 1, outcp, out);
    outcp=0;
  };
};

void flushbuf(FILE *out)
{
  if (outcp)
  {
    fwrite(&outbuf, 1, outcp, out);
    outcp=0;
  };
};

int getbits(int bits, FILE *in)
{
  while (bitsfull<24)
  {
    bitbuf+=fgetc(in)<<bitsfull;
    bitsfull+=8;
  };

  return bitbuf & ((1<<bits)-1);
};

void advancebits(int bits, FILE *in)
{
  while (bitsfull<24)
  {
    bitbuf+=fgetc(in)<<bitsfull;
    bitsfull+=8;
  };

  bitbuf>>=bits; bitsfull-=bits;

  while (bitsfull<24)
  {
    bitbuf+=fgetc(in)<<bitsfull;
    bitsfull+=8;
  };
};

int getbit_adv(FILE *in)
{
  char b;

  b=getbits(1, in);
  advancebits(1, in);

  return b;
};

void do_decompress(FILE *in, FILE *out)
{
  int       sym;
  huffnode *pos;

  pos=huffroot; sym=-1; bitbuf=bitsfull=0;

  while (sym!=256)
  {
    pos=pos->child[getbit_adv(in)];

    if (pos->orgval>=0)
    {
      sym=pos->orgval;

      if (sym<256) writeb(sym, out);

      pos=huffroot;
    };
  };

  flushbuf(out);
};

void decompress_huffman(char *name1, char *name2)
{
  FILE *inf, *outf;

  inf=fopen(name1, "rb");

  if (inf)
  {
    printf("Opened file \"%s\".\n", name1);

    printf("Reading statistics...\n");
    read_statistics(inf);

    printf("Creating huffman tree\n");
    create_huffman_tree();

    outf=fopen(name2, "wb");

    if (outf)
    {
      printf("Decompressing...");
      do_decompress(inf, outf);
      printf("Done!\n");

      fclose(outf);
    };

    fclose(inf);
  };
};

int main(int argc, char *argv[])
{
  setbuf(stdout, NULL);

  printf("Huffman decoder (c) 1998 F. Giesen\n\n");

  if (argc!=3)
  {
    printf("Syntax: HUFFD <INFILE.EXT> <OUTFILE.EXT>\n");
    return 1;
  };

  decompress_huffman(argv[1], argv[2]);

  return 0;
};
