/****************************************************************************
 * fbwtdec.cpp                         the "final" combined BWT/MTF decoder *
 ****************************************************************************
 * made by Fabian Giesen aka RYG/Chrome Design                              *
 * for my BWT article in hugi #13                                           *
 *                                                                          *
 * please credit me if you use this code (or a derived one) in one of your  *
 * productions. thanks.                                                     *
 ****************************************************************************
 * WARNING: This code assumes that char is unsigned and memcmp compares     *
 *          unsigned chars! If that isn't right for your platform, you have *
 *          to write an own memcmp function and change all chars in this    *
 *          source to unsigned ones!                                        *
 ****************************************************************************/

// This code works I bit different than the one I discussed by now. Look at
// FBWTENC.CPP for details.

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <iostream.h>

typedef unsigned long ulong;           // useful!

ulong *T;                              // the transformation vector

ulong startpos[257];                   // "virtual F array" stuff
ulong counts[257];

char order[256];                       // the MTF order table

// the "virtual F array" generator

void generateStartingPositions(char *data, ulong length, ulong eobp)
{
  ulong i, sum;                        // counters.

  for (i=0; i<257; i++) counts[i]=0;   // build the counts array

  for (i=0; i<length; i++)             // from input data
  {
    if (i!=eobp)                       // if we're not at our "end of buffer"
      counts[data[i]]++;               // character simply process byte
    else
      counts[256]++;                   // else increment our "eob" counter
  };

  sum=0;

  for (i=0; i<257; i++)                // then calculate our "starting
  {                                    // positions". read the article if you
    startpos[i]=sum;                   // want to know how it's actually done
    sum+=counts[i];
  };

  // this changed a bit because of end of buffer character handling.
};

// the transformation vector generator

void buildTransformationVector(char *L, ulong L_len, ulong eobp)
{
  ulong i;

  if (T) delete[] T;                   // setup T
  T=new ulong[L_len];

  generateStartingPositions(L, L_len, eobp); // create "virtual F array"

  for (i=0; i<L_len; i++)              // build the transformation vector.
  {
    int ind;                           // I don't need the counts array.
                                       // This way it also works and is
    if (i!=eobp)                       // faster, too.
      ind=L[i];
    else                               // ohh and please notice special eob
      ind=256;                         // handling in here

    T[startpos[ind]++]=i;
  };
};

// the MTF initialization function

void initMTF()
{
  for (int i=0; i<256; i++) order[i]=i;
};

// the MTF "get" function

char getMTF(char *&from)
{
  int  i;                              // standard counter
  char code;                           // our code

  code=order[*from];                   // get output byte from order table

  for (i=*from; i>0; i--) order[i]=order[i-1]; // shift order table
  order[0]=code;

  from++;                              // increment our "from" pointer

  return code;                         // then return our code
};

// get MTF encoded DWord (little endian)

ulong getDWordMTF(char *&from)
{
  ulong i;

  i =getMTF(from);
  i|=getMTF(from) << 8;
  i|=getMTF(from) << 16;
  i|=getMTF(from) << 24;

  return i;
};

// directly decode a MTF encoded DWord

ulong unMTFDWord(ulong dword)
{
  char *p;

  p=(char *) &dword;
  return getDWordMTF(p);
};

// decode a BWT encoded block

void decodeBWT(char *indata, char *outdata, ulong length)
{
  char  *L;                            // our L array
  char  *p;                            // universal pointer
  ulong  i, j;                         // counters
  ulong  primind;                      // primary index
  ulong  eobind;                       // end of buffer char index

  L=new char[length];                  // allocate the L array
  p=L;                                 // let our pointer point to it

  for (i=0; i<length; i++)             // decode our L array (it's MTF
    *p++=getMTF(indata);               // encoded, remember?)

  primind=getDWordMTF(indata);         // get some status info from end
  eobind=getDWordMTF(indata);          // of block

  // create our transformation vector
  buildTransformationVector(L, length, eobind);

  p=outdata;                           // set pointer to output data
  i=primind;                           // and index to primary index

  for (j=0; j<length-1; j++)           // then decode
  {
    *p++=L[i];                         // output L[index]
    i=T[i];                            // then update index
  };

  delete[] L;                          // last step: delete our L array
};

int main(int argc, char *argv[])
{
  FILE  *in, *out;                     // input/output files
  char  *inbuffer, *outbuffer;         // input/output buffer
  ulong  L_len;                        // L array length
  int    block;                        // simple block counter
  int    r;                            // read counter

  cerr << "The \"Final\" BWT/MTF Decoder" << endl;

  if (argc!=3)                         // if wrong parameter count, exit
  {
    cerr << endl << "Use: FBWTDEC <Input filename> <Output filename>" << endl;
    return 1;
  };

  in=fopen(argv[1], "rb");             // open input file

  if (!in)                             // if open failed, exit
  {
    cerr << endl << "cannot open input file!" << endl;
    return 1;
  };

  out=fopen(argv[2], "wb");            // open output file

  if (!out)                            // if open failed, exit
  {
    cerr << endl << "cannot open output file!" << endl;
    return 1;
  };

  initMTF();                           // initialize MTF decoder

  block=1;                             // block init value

  do                                   // the decoding loop (easy)
  {
    r=fread(&L_len, 1, 4, in);         // read L array length
    L_len=unMTFDWord(L_len);           // decode it (its MTF encoded)

    if (r==4)                          // only if succesful continue
    {
      cerr << "Decoding block " << block++ << "..." << endl;

      inbuffer=new char[L_len+8];      // allocate memory for input buffer

      fread(inbuffer, 1, L_len+8, in); // then read L array+status info

      outbuffer=new char[L_len-1];     // allocate output buffer (minus EOB
                                       // character)

      // then decode it

      decodeBWT(inbuffer, outbuffer, L_len);

      fwrite(outbuffer, 1, L_len-1, out); // write output buffer

      delete[] inbuffer;               // free temp. buffer memory
      delete[] outbuffer;
    };
  } while (r==4);

  fclose(in);                          // close our files
  fclose(out);

  return 0;                            // that's all, folks
};
