;-----------------------------------------------------------------------------
; This is the world smallest text file viewer that allows
; smoothly scrolling upto 204 lines of text.
;
; Usage: TEXTVIEW <textfile>
;
; Keys while viewing: arrow up and down, page up and down, home and end.
; Esc key exits the program.
;
; TEXTVIEW has been written by Chris Dragan.
; This and other free sources you can find at: http://ams.ampr.org/cdragan/
;
; To compile source of TEXTVIEW use Netwide Assembler:
;  nasm -o textview.com textview.asm
; Netwide Assembler's home page is: http://www.cryogen.com/Nasm/
;-----------------------------------------------------------------------------

%define CH_SPACE	' '
%define CH_CR		13
%define CH_TAB		9

%define TEXT_COLOR	17h	; light white (7) on blue (1)
%define LINE_WIDTH	80 * 2	; every of 80 fields consists of char and attr
%define NUM_LINES	204	; about 32768 (vid mem size) / 160 (line wid)
%define BUFFER_SIZE	4000h	; allows upto 16KB of input file

%define CRTC_PORT	0463h	; address in BIOS data area
%define CRTC_SCANLINE	08h
%define CRTC_ROWHI	0Ch
%define CRTC_ROWLO	0Dh

%define KEY_AR_DN	80
%define KEY_PG_DN	81
%define KEY_HOME	71
%define KEY_END 	79

org 100h
section .text

		; Find beginning of first cmdline argument
			mov	si, 0081h
_cmdline:		mov	dx, si
			lodsb
			cmp	al, CH_SPACE
			jz	_cmdline
			cmp	al, CH_CR
			jnz	_filename

		; Display info message and quit
			mov	dx, InfoMessage
			jmp	_message_and_quit

		; Append zero sign to filename
_filename:		lodsb
			cmp	al, CH_SPACE
			jz	_append_zero
			cmp	al, CH_CR
			jnz	_filename
_append_zero:		mov	[si-1], byte 0

		; Open the file
			mov	ax, 3D90h
			int	21h
			mov	dx, ErrorMessage
			jc	near _message_and_quit
			mov	bx, ax

		; Load the file
			mov	ah, 3Fh
			mov	cx, BUFFER_SIZE
			push	dx
			mov	dx, TextBuffer
			mov	si, dx
			int	21h
			pop	dx
			jc	near _message_and_quit

		; Get CRTC port
			xor	di, di
			push	di
			pop	es
			push	word [es:CRTC_PORT]

		; Clear video memory
			push	word 0B800h
			pop	es
			push	ax
			mov	ax, (TEXT_COLOR << 8) + CH_SPACE
		    rep stosw

		; Draw the file in the video memory
			pop	cx
			xor	bx, bx
			xor	di, di

_draw_next_byte:	lodsb			; Get a byte
			cmp	al, CH_CR
			jz	_draw_enter
			cmp	al, CH_TAB
			jz	_draw_tab

			mov	[es:bx+di], al	; Draw a byte
			inc	di
			inc	di
			jmp	short _check_line_overflow

_draw_tab:		and	di, byte -16	; Draw a tab (8-char align)
			add	di, byte 16

_check_line_overflow:	cmp	di, LINE_WIDTH	; Check if line isn't overflown
			jc	_decrement_byte_count
_find_enter:		dec	cx
			jz	_file_drawn
			lodsb
			cmp	al, CH_CR
			jnz	_find_enter

_draw_enter:		inc	si		; Skip assumed LF
			dec	cx
			jz	_file_drawn

			xor	di, di		; Move to next line
			cmp	bx, (NUM_LINES-1) * LINE_WIDTH
			jnc	_file_drawn
			add	bx, LINE_WIDTH

_decrement_byte_count:	loop	_draw_next_byte ; Enumerate read characters

		; Calculate last line address
_file_drawn:		mov	ax, (LINE_WIDTH >> 4)
			xchg	ax, bx
			cwd
			div	bx
			sub	ax, 24*16
			jnc	_many_lines
			xchg	ax, dx
_many_lines:		xchg	ax, si
			xor	di, di
			pop	dx
			add	dx, byte 6

		; Hide cursor
			mov	ah, 01h
			mov	ch, 20h
			int	10h

_key_loop:	; ----- keyboard input loop -----

		; Wait for horizontal retrace
_short_wait:		in	al, dx
			test	al, 01h
			jnz	_short_wait

		; Update scanline position
			push	dx
			sub	dx, byte 6
			mov	ax, di
			and	al, 0Fh
			mov	ah, al
			mov	al, CRTC_SCANLINE
			out	dx, ax

		; Update row position
			mov	ax, di
			and	al, 0F0h
			lea	eax, [eax+eax*4]
			mov	bh, al
			mov	al, CRTC_ROWHI
			mov	bl, CRTC_ROWLO
			out	dx, ax
			xchg	ax, bx
			out	dx, ax
			pop	dx

		; Ignore keyboard buffer
_get_key:		mov	ah, 01h
			int	16h
			jz	_buf_ignored
			mov	ah, 00h
			int	16h

		; Get key code
_buf_ignored:		xor	ax, ax
			in	al, 60h
			dec	ax	; al == Esc ?
			jz	_exit
			cmp	al, KEY_HOME-1
			jz	_home
			cmp	al, KEY_END-1
			jz	_end
			sub	al, 71
			jc	_get_key
			test	al, ~(KEY_AR_DN-72)
			jz	_arrow
			dec	ax
			test	al, ~(KEY_PG_DN-73)
			jnz	_get_key

		; Execute page key
			add	cl, 2
			jnc	_get_key
			or	ax, ax
			jz	_move_up
			jmp	short _move_down

		; Execute arrow key
_arrow: 		xchg	ax, bx
_long_wait:		in	al, dx
			test	al, 08h
			jz	_long_wait
			or	bx, bx
			jz	_move_up

		; Move screen
_move_down:		inc	di
			cmp	si, di
			jnc	_key_loop
_move_up:		or	di, di
			jz	_get_key
			dec	di
			jmp	short _key_loop

		; Execute home or end key
_home:			xor	di, di
			jmp	short _key_loop
_end:			mov	di, si
			jmp	short _key_loop

		; Exiting... - Reset screen mode
_exit:			mov	al, 3
			int	10h

		; Display ending message and quit
			mov	dx, EndingMessage
_message_and_quit:	mov	ah, 09h
			int	21h
			mov	dx, EOL
			int	21h
			ret

InfoMessage		db 'Usage: TEXTVIEW <textfile>$'
EndingMessage		db 'Thank you for using Chris Dragan', 27h, 's text'
			db ' file viewer$'
ErrorMessage		db 'Error accessing file$'
EOL			db 13, 10, 13, 10, '$'

section .bss
TextBuffer		resb BUFFER_SIZE
