// =========================================================================
//  pre2com v1.15 -- Hugi Compo #7 entry by Jibz '99
// =========================================================================
//
// Takes a preprocessed image data file and an optimized palette. Packs the
// image data by finding the optimal path through the encoding tree, and
// packs the palette data using delta-compression. Saves both the packed
// image and the packed palette to the final ENTRY.COM, prepending the
// depacker.
//

#include <string.h>
#include <stdio.h>
#include <io.h>
#include <dos.h>
#include <conio.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <errno.h>

#define MAXMATCHLEN 32767
#define MAXMATCHPOS 32767

// Possible errors
enum { OPEN_ERR = 1, BIG_ERR,   READ_ERR,   TMP_ERR,
       ABORT_ERR,    MEM_ERR,   WRITE_ERR,  SMALL_ERR };

// Make sure we have more than enough mem to work with ;)
#define BUFFERSIZE (65536)

// Program name and version
const char *versionstr = "pre2com v1.15";

enum { MATCH_TOKEN, LITERAL_TOKEN };

typedef struct {
        int type;
        int pos;
        int len;
        int totallength;
} TOKEN;

TOKEN tokenbuffer[BUFFERSIZE];

unsigned char inbuffer[BUFFERSIZE];
unsigned char outbuffer[BUFFERSIZE];
unsigned char palbuffer[2048];
unsigned char tagbuffer[BUFFERSIZE];

// The depacker
unsigned char depacker[] = {
   0xB0, 0x13, 0xCD, 0x10, 0xBE, 0x98, 0x01, 0xBF, 0x00, 0x24,
   0xBD, 0x98, 0x01, 0xA4, 0xE8, 0x6F, 0x00, 0x73, 0xFA, 0xE8,
   0x74, 0x00, 0x91, 0x48, 0x48, 0xB9, 0x06, 0x00, 0xE8, 0x61,
   0x00, 0x11, 0xC0, 0xE2, 0xF9, 0xF7, 0xD8, 0x74, 0x0B, 0x93,
   0xE8, 0x5F, 0x00, 0x8A, 0x01, 0xAA, 0xE2, 0xFB, 0xEB, 0xDC,
   0xBA, 0xC8, 0x03, 0xEE, 0x42, 0xB4, 0xA0, 0x8E, 0xC0, 0x31,
   0xDB, 0xBF, 0x00, 0x03, 0xB4, 0x00, 0xE8, 0x3B, 0x00, 0x10,
   0xE4, 0xE8, 0x36, 0x00, 0x10, 0xE4, 0x74, 0x0F, 0x7B, 0x06,
   0xE8, 0x37, 0x00, 0x41, 0x88, 0xCC, 0xE8, 0x27, 0x00, 0x73,
   0x02, 0xF6, 0xDC, 0x00, 0xE0, 0xEE, 0x86, 0xC3, 0x86, 0xC7,
   0x4F, 0x75, 0xD9, 0xBD, 0x98, 0x01, 0xE8, 0x13, 0x00, 0x73,
   0x01, 0xAC, 0xAA, 0x81, 0xFF, 0x00, 0xFA, 0x72, 0xF3, 0x31,
   0xC0, 0xCD, 0x16, 0xB8, 0x03, 0x00, 0xCD, 0x10, 0xD0, 0x6E,
   0x00, 0x75, 0x04, 0x45, 0xD0, 0x5E, 0x00, 0xC3, 0xB9, 0x01,
   0x00, 0xE8, 0xF0, 0xFF, 0x11, 0xC9, 0xE8, 0xEB, 0xFF, 0x72,
   0xF6, 0xC3
};
const unsigned int depacker_size = 152;

const unsigned int init_si_offs     = 0x05;
const unsigned int init_di_offs     = 0x08;
const unsigned int pal_entries_offs = 0x3e;
const unsigned int init_bp_offs     = 0x68;

// Structure to hold matches
struct MATCH {
   int pos;
   int len;
} match, nextmatch, literalmatch, testmatch;

// Global variables used
int infile = -1, outfile = -1;
unsigned int inpos, outpos, palpos, tokenpos;
unsigned int tagpos, bitcount;

// =========================================================================
//  COMPRESSION STUFF
// =========================================================================

inline int log2x(int n)
{
   if (n < 2) return (100);
   if (n < 4) return (2);
   if (n < 8) return (4);
   if (n < 16) return (6);
   if (n < 32) return (8);
   if (n < 64) return (10);
   if (n < 128) return (12);
   if (n < 256) return (14);
   if (n < 512) return (16);
   if (n < 1024) return (18);
   if (n < 2048) return (20);
   if (n < 4096) return (22);
   if (n < 8192) return (24);
   if (n < 16384) return (26);
   if (n < 32768) return (28);
   if (n < 65536) return (30);
   return (32);
}

static void advancetagbyte(int bit)
{
   // Check tagpos and then decrement
   if (!bitcount--)
   {
      bitcount = 7;
      tagpos++;
   }

   // Shift in bit
   if (bit)
   {
      tagbuffer[tagpos] = (tagbuffer[tagpos] >> 1) + 0x80;
   } else {
      tagbuffer[tagpos] = tagbuffer[tagpos] >> 1;
   }
}

// Output Gamma2-code for val in range [2..?] ...
static void outputGAMMA(unsigned int val)
{
   int invertlen = 0;
   unsigned int invert;

   do {
      invert = (invert << 1) + (val & 0x0001);
      invertlen++;
   } while ((val >>= 1) > 1);

   while (--invertlen)
   {
      advancetagbyte(invert & 0x0001);
      advancetagbyte(1);
      invert >>= 1;
   }
   advancetagbyte(invert & 0x0001);
   advancetagbyte(0);
}

int getLITERALlength(unsigned char lit)
{
   return (9);
}

void outputLITERALtoken(unsigned char lit)
{
   // Insert the literal token
   tokenbuffer[tokenpos].type = LITERAL_TOKEN;
   tokenpos++;
}

void outputLITERAL(unsigned char lit)
{
   // 0 indicates a literal
   advancetagbyte(0);

   // Output the literal
   outbuffer[outpos++] = lit;
}

int getCODEPAIRlength(unsigned int pos, unsigned int len)
{
   int length = 1;

      length += log2x((pos >> 6) + 2);
      length += 6;

   length += log2x(len);

   return (length);
}

void outputCODEPAIRtoken(unsigned int pos, unsigned int len)
{
   // Insert the match token
   tokenbuffer[tokenpos].type = MATCH_TOKEN;
   tokenbuffer[tokenpos].pos = pos;
   tokenbuffer[tokenpos].len = len;
   tokenpos++;
}

void outputCODEPAIR(unsigned int pos, unsigned int len)
{
   // 1 indicates a match
   advancetagbyte(1);

   outputGAMMA((pos >> 6) + 2);
   advancetagbyte(pos & 0x0020);
   advancetagbyte(pos & 0x0010);
   advancetagbyte(pos & 0x0008);
   advancetagbyte(pos & 0x0004);
   advancetagbyte(pos & 0x0002);
   advancetagbyte(pos & 0x0001);

   outputGAMMA(len);
}

void outputEOD()
{
   // A match ..
   advancetagbyte(1);

   // .. with position 0 indicates EOD
   outputGAMMA(2);
   advancetagbyte(0);
   advancetagbyte(0);
   advancetagbyte(0);
   advancetagbyte(0);
   advancetagbyte(0);
   advancetagbyte(0);
}

int getPALDISTlength(unsigned int oldval, unsigned int newval)
{
   unsigned int dist = abs(oldval - newval);

   switch (dist)
   {
      case 0  : return (2); break;
      case 1  :
      case 2  : return (3); break;
      default : return (3 + log2x(dist - 1)); break;
   }
}

void outputPALDIST(unsigned int oldval, unsigned int newval)
{
   unsigned int dist = abs(oldval - newval);

   if (dist == 0)
   {
      advancetagbyte(0);
      advancetagbyte(0);
   } else {
      if (dist < 3)
      {
         if (dist == 1)
         {
            advancetagbyte(0);
            advancetagbyte(1);
         } else {
            advancetagbyte(1);
            advancetagbyte(0);
         }
      } else {
         advancetagbyte(1);
         advancetagbyte(1);

         outputGAMMA(dist - 1);
      }

      if (oldval < newval) advancetagbyte(1); else advancetagbyte(0);
   }
}

void findmatch(MATCH *thematch, int lookpos, int lookback, int lookforward)
{
   int matchlen, matchpos;

   thematch->pos = 0; thematch->len = 0;

   if (lookforward > MAXMATCHLEN) lookforward = MAXMATCHLEN;
   if (lookback > MAXMATCHPOS) lookback = MAXMATCHPOS;

   for (matchpos = lookback; matchpos > 0; matchpos--)
   {
      matchlen = 0;

      // Find next possible match
      while ((matchpos > 1) && (inbuffer[lookpos - matchpos] != inbuffer[lookpos])) matchpos--;

      // Scan it
      while ((inbuffer[lookpos - matchpos + matchlen] == inbuffer[lookpos + matchlen]) && (matchlen < lookforward))
      {
         matchlen++;
      }
      // Check it
      if (matchlen >= thematch->len)
      {
         thematch->len = matchlen;
         thematch->pos = matchpos;
      }
   }
}

void findnearestmatch(MATCH *thematch, int lookpos, int lookback, int lookforward)
{
   int matchlen, matchpos;

   thematch->pos = 0; thematch->len = 0;

   if (lookforward > MAXMATCHLEN) lookforward = MAXMATCHLEN;
   if (lookback > MAXMATCHPOS) lookback = MAXMATCHPOS;

   for (matchpos = 1; ((matchpos <= lookback) && (thematch->len < lookforward)); matchpos++)
   {
      matchlen = 0;

      // Find next possible match
      while ((matchpos < lookback) && (inbuffer[lookpos - matchpos] != inbuffer[lookpos])) matchpos++;

      // Scan it
      while ((inbuffer[lookpos - matchpos + matchlen] == inbuffer[lookpos + matchlen]) && (matchlen < lookforward))
      {
         matchlen++;
      }
      // Check it
      if (matchlen > thematch->len)
      {
         thematch->len = matchlen;
         thematch->pos = matchpos;
      }
   }
}

// =========================================================================
//  MISC. FUNCTIONS
// =========================================================================

void cursoff();
#pragma aux cursoff = \
        "mov   ah, 3"    \
        "xor   ebx, ebx" \
        "int   10h"      \
        "or    ch, 20h"  \
        "mov   ah, 1"    \
        "int   10h"      \
        modify exact [eax ebx ecx edx];

void curson();
#pragma aux curson = \
        "mov   ah, 3"    \
        "xor   ebx, ebx" \
        "int   10h"      \
        "and   ch, 1fh"  \
        "mov   ah, 1"    \
        "int   10h"      \
        modify exact [eax ebx ecx edx];

// Some I/O error handler :)
void myerror(int n)
{
   // Print out error information
   printf("\n\n\007ERR: ");
   switch (n)
   {
      case OPEN_ERR  : printf("Unable to open input-file!\n"); break;
      case BIG_ERR   : printf("File is not a COM-file or too big!\n"); break;
      case READ_ERR  : printf("Unable to read from input-file!\n"); break;
      case TMP_ERR   : printf("Unable to create 'APACKTMP.$$$'!\n"); break;
      case ABORT_ERR : printf("Aborted by user!\n"); break;
      case MEM_ERR   : printf("Not enough memory!\n"); break;
      case WRITE_ERR : printf("Unable to write to output-file!\n"); break;
      case SMALL_ERR : printf("File is too small to be packed!\n"); break;
      default        : printf("An unknown error occured!\n");
   }

   // Free memory and close files
   if (infile != -1) close(infile);
   if (outfile != -1) close(outfile);

   // Turn cursor on
   curson();

   // Exit with errorlevel = error no.
   exit(n);
}

// =========================================================================
//  MAIN
// =========================================================================

int main(int argc, char *argv[])
{
   unsigned char ch;

   unsigned int counter = 0;
   unsigned char *rotator = "-/|\\";
   int rotatorpos = 0;

   unsigned int infilesize, outfilesize;

   unsigned int literalcount = 0;

   unsigned int i, j, tmp;

   // Write name and copyright notice
   printf("---------------------------------------\\/--------------------------------------\n");
   printf("%-20s                            Hugi Compo #7 entry by Jibz '99\n", versionstr);
   printf("---------------------------------------/\\--------------------------------------\n\n");

   // Write syntax if not correct number of parameters
   if (argc != 2)
   {
      printf("Syntax:   PRE2COM <input file>\n\n");
      exit(1);
   }

   // Turn cursor off
   cursoff();

   // Open raw data file
   printf("- Opening files\n");
   if ((infile = open(argv[1], O_RDONLY | O_BINARY)) == -1) myerror(OPEN_ERR);
   lseek(infile, 0, SEEK_SET);

   // Get file size and check it
   infilesize = filelength(infile);
   if (infilesize > 28160) myerror(BIG_ERR);
   if (infilesize < 64) myerror(SMALL_ERR);

   // Read raw data
   lseek(infile, 0, SEEK_SET);
   read(infile, inbuffer, infilesize);

   // Open palette file
   if (infile != -1) close(infile);
   if ((infile = open("ENTRY.PAL", O_RDONLY | O_BINARY)) == -1) myerror(OPEN_ERR);
   lseek(infile, 0, SEEK_SET);

   // Get file size
   palpos = filelength(infile);

   // Read palette data
   lseek(infile, 0, SEEK_SET);
   read(infile, palbuffer, palpos);

// -------------------------------------------------------------------------
//  FIND LONGEST MATCHES
// -------------------------------------------------------------------------

   tokenbuffer[0].type = LITERAL_TOKEN;
   tokenpos = 1;

   // Find the longest match at each position in the input
   for (inpos = 1; inpos < infilesize; inpos++)
   {
      // Show progress and check for ESC every now and then..
      if ((counter++ & 0x003f) == 0)
      {
         printf("\r%c Finding longest matches ....... %3d%%", rotator[rotatorpos], inpos * 100 / infilesize);
         rotatorpos = (rotatorpos + 1) & 0x0003;

         // Check for ESC-hit
         if (kbhit())
         {
            ch = getch();
            if (ch == 0) ch = getch();
            if (ch == 27) myerror(ABORT_ERR);
         }
      }

      // Find best match at current position
      findmatch(&match, inpos, inpos, infilesize - inpos);

      // Insert the appropriate token
      if (match.len >= 2)
      {
         outputCODEPAIRtoken(match.pos, match.len);
      } else {
         outputLITERALtoken(inbuffer[inpos]);
      }
   }
   printf("\r- Finding longest matches ....... Done\n");

// -------------------------------------------------------------------------
//  FIND SHORTEST PATH
// -------------------------------------------------------------------------

   // Set the length of the last token (a literal)
   tokenbuffer[infilesize].totallength = 0;
   tokenbuffer[infilesize - 1].totallength = getLITERALlength(inbuffer[infilesize - 1]);

   // Run backwards through the tokens, trying to find shortest path
   for (inpos = infilesize - 2; inpos > 0; inpos--)
   {
      // Show progress and check for ESC every now and then..
      if ((counter++ & 0x003f) == 0)
      {
         printf("\r%c Finding shortest path ......... %3d%%", rotator[rotatorpos], (infilesize - inpos) * 100 / infilesize);
         rotatorpos = (rotatorpos + 1) & 0x0003;

         // Check for ESC-hit
         if (kbhit())
         {
            ch = getch();
            if (ch == 0) ch = getch();
            if (ch == 27) myerror(ABORT_ERR);
         }
      }

      // Check bitlength of encoding possibilities and make choice
      if (tokenbuffer[inpos].type == MATCH_TOKEN)
      {
         match.pos = tokenbuffer[inpos].pos;
         match.len = tokenbuffer[inpos].len;

         i = 2;
         do {
               // Find nearest match with current length
               findnearestmatch(&testmatch, inpos, inpos, i);

               if ((testmatch.pos > 0) && (testmatch.len >= i))
               {
                  if ((getCODEPAIRlength(testmatch.pos, i) +
                       tokenbuffer[inpos + i].totallength) <
                      (getCODEPAIRlength(match.pos, match.len) +
                       tokenbuffer[inpos + match.len].totallength)
                     )
                  {
                     match.pos = testmatch.pos;
                     match.len = i;
                  }
               } else {
                  if ((getCODEPAIRlength(match.pos, i) +
                       tokenbuffer[inpos + i].totallength) <
                      (getCODEPAIRlength(match.pos, match.len) +
                       tokenbuffer[inpos + match.len].totallength)
                     )
                  {
                     match.len = i;
                  }
               }

            i++;
         } while (i < tokenbuffer[inpos].len);

         if ((getCODEPAIRlength(match.pos, match.len) +
              tokenbuffer[inpos + match.len].totallength) <
             (getLITERALlength(inbuffer[inpos]) + tokenbuffer[inpos + 1].totallength)
            )
         {
            tokenbuffer[inpos].pos = match.pos;
            tokenbuffer[inpos].len = match.len;
            tokenbuffer[inpos].totallength = getCODEPAIRlength(tokenbuffer[inpos].pos, match.len) +
                                             tokenbuffer[inpos + match.len].totallength;
         } else {
            tokenbuffer[inpos].type = LITERAL_TOKEN;
            tokenbuffer[inpos].totallength = tokenbuffer[inpos + 1].totallength + getLITERALlength(inbuffer[inpos]);
         }
      } else {
         tokenbuffer[inpos].totallength = tokenbuffer[inpos + 1].totallength + getLITERALlength(inbuffer[inpos]);
      }
   }
   printf("\r- Finding shortest path ......... Done\n");

// -------------------------------------------------------------------------
//  OUTPUT ENCODING
// -------------------------------------------------------------------------

   tagpos = 0;
   tagbuffer[tagpos++] = 1;
   bitcount = 8;
   inpos = 0;
   outpos = 0;

   // The first byte is sent verbatim
   outbuffer[outpos++] = inbuffer[inpos++];

   // Run through tokens and output encoding
   for (inpos = 1; inpos < infilesize; )
   {
      // Show progress and check for ESC every now and then..
      if ((counter++ & 0x003f) == 0)
      {
         printf("\r%c Encoding ...................... %3d%%", rotator[rotatorpos], inpos * 100 / infilesize);
         rotatorpos = (rotatorpos + 1) & 0x0003;

         // Check for ESC-hit
         if (kbhit())
         {
            ch = getch();
            if (ch == 0) ch = getch();
            if (ch == 27) myerror(ABORT_ERR);
         }
      }

      if (tokenbuffer[inpos].type == MATCH_TOKEN)
      {
         outputCODEPAIR(tokenbuffer[inpos].pos, tokenbuffer[inpos].len);
         inpos += tokenbuffer[inpos].len;
      } else {
         outputLITERAL(inbuffer[inpos]);
         inpos++;
      }
   }

   // Output EOD-marker
   outputEOD();

   // Print out result
   printf("\r- Encoding ...................... Done (%u bytes -> %u bytes)\n", infilesize, outpos + tagpos);

// -------------------------------------------------------------------------
//  COMPRESS PALETTE DATA
// -------------------------------------------------------------------------

   printf("- Packing palette ............... ");

   int rgb_bitlength = 0;
   int rval = 0, gval = 0, bval = 0;

   // Delta encode linear distances between palette entries
   for (i = 0; i < palpos/3; i++)
   {
      // Output Gamma2 encoding of difference between last and current
      rgb_bitlength += getPALDISTlength(palbuffer[i*3], rval);
      outputPALDIST(palbuffer[i*3], rval);
      rval = palbuffer[i*3];

      rgb_bitlength += getPALDISTlength(palbuffer[i*3+1], gval);
      outputPALDIST(palbuffer[i*3+1], gval);
      gval = palbuffer[i*3+1];

      rgb_bitlength += getPALDISTlength(palbuffer[i*3+2], bval);
      outputPALDIST(palbuffer[i*3+2], bval);
      bval = palbuffer[i*3+2];
   }
   printf("Done (%u bytes -> %u bytes)\n", palpos, (rgb_bitlength + 7) / 8);

   // Shift last tagbits into position
   tagbuffer[tagpos] = tagbuffer[tagpos] >> bitcount;
   tagpos++;

// -------------------------------------------------------------------------
//  FINALIZE AND WRITE TO FILE
// -------------------------------------------------------------------------

   // Set stuff in the depacker
   tmp = 0x100 + depacker_size + tagpos;
   depacker[init_si_offs    ] = tmp & 0x00ff;
   depacker[init_si_offs + 1] = (tmp >> 8) & 0x00ff;

   tmp = 0x100 + depacker_size + tagpos + outpos;
   depacker[init_di_offs    ] = tmp & 0x00ff;
   depacker[init_di_offs + 1] = (tmp >> 8) & 0x00ff;

   depacker[pal_entries_offs    ] = palpos & 0x00ff;
   depacker[pal_entries_offs + 1] = (palpos >> 8) & 0x00ff;

   tmp = 0x100 + depacker_size + tagpos + outpos + infilesize - 8001;
   depacker[init_bp_offs    ] = tmp & 0x00ff;
   depacker[init_bp_offs + 1] = (tmp >> 8) & 0x00ff;

   // Open ENTRY.COM for writing
   if ((outfile = open("ENTRY.COM", O_WRONLY | O_CREAT | O_BINARY | O_TRUNC, S_IREAD | S_IWRITE)) == -1) myerror(WRITE_ERR);
   lseek(outfile, 0, SEEK_SET);

   // Write depacker
   lseek(outfile, 0, SEEK_SET);
   write(outfile, depacker, depacker_size);
   printf("- Depacker added ................ %u bytes\n", depacker_size);

   // Write tag data
   write(outfile, tagbuffer, tagpos);

   // Write raw data
   write(outfile, outbuffer, outpos);

   outfilesize = filelength(outfile);

   // Close in and out files
   printf("- Closing files\n");
   if (infile != -1) close(infile);
   if (outfile != -1) close(outfile);

   // Print out results
   printf("\nDone ... compression ratio: %d%% ", 100 - ((outfilesize * 100) / 64768));
   printf("(64768 bytes -> %lu bytes)\n", outfilesize);

   // Turn cursor back on
   curson();

   // We're happy :)
   return (0);
}
// =========================================================================
//  END
// =========================================================================
