/*
 * Packer for Hugi compo #7
 * by INT-E
 *
 * usage:
 *   pack > hugi.inc
 *
 * tested with djgpp.
 *
 * data is compressed by first performing a BWT transform, followed by a
 * MTF coding step. after this, runs of zeros are RLE-coded and finally
 * the whole thing is run through an arithmetic coder (a very slow but
 * precise one which is easy to implement).
 *
 * credits:
 *   ryg/chrome design
 *     - for his article about the Burrows-Wheeler-Transform in Hugi #13
 *   Julian R. Seward
 *     - for bzip2, and for writing readable code and for the basics of the
 *       RLE coder used here
 *   Michael Burrows and David J. Wheeler
 *     - for the block sorting transform and a few additional ideas, like
 *       using the move to front coder
 *   adok/hugi
 *     - for Hugi and these nice compos...
 */
#include <fcntl.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>

#define PREFIX          "hugi."
#define COM             "com"
#define PAK             "pak"
#define INC             "pack.inc"
#define UNPSIZE         unpsize
#define PACKSIZE        65536

#define Uchar           unsigned char
#define Ushrt           unsigned short

#ifndef O_BINARY
#define O_BINARY 0
#endif

int
#include "UNPACK.INC"

int packsize;

/* block sorter */
Uchar block[2*PACKSIZE];
int   ptr  [PACKSIZE];
int   first;

/* block sorter (horribly slow, but fast enough for this task) */
int compare(const void*a,const void*b) {
  int i;
  Uchar *aa=&block[*(int*)a];
  Uchar *bb=&block[*(int*)b];
  while(!(i=*aa++-*bb++));
  return i;
}

void sort() {
  int i;
  for (i=0;i<packsize;i++) ptr[i]=i;
  qsort(ptr,packsize,sizeof(int),compare);
}

/* mtf coder */
Uchar val  [256];
int   bp;

void initmtf() {
  int i;
  for (i=0;i<256;i++) val[i]=i;
  bp=0;
  first=ptr[0]-1;
}

int get_mtf_byte() {
  Uchar a,b=0,c,d;
  int p;
  if ((p=ptr[bp++])==0) {p=packsize;first=bp-1;}
  a=block[p-1];
  while (val[b]!=a) b++;
  d=0; // b>>4
  for (c=b;c-->d;) val[c+1]=val[c];
  val[d]=a;
  return b;
}

/* rle coder */
int   runl;
int   rlebuf[16];
int   rlep;
int   t;

void initrle() {
  runl=0;
  rlep=0;
  t=0;
  initmtf();
}

int get_rle_word() {
  int r=0;
  if (rlep==0) {
    if (bp==packsize && t==0) return -1;
    if (t==0) t=get_mtf_byte();
    rlebuf[rlep++]=t;
    t=0;
    while (bp<packsize && (t=get_mtf_byte())==0) r++;
    while (r>0) {
      r--;
      rlebuf[rlep++]=257-r%2;
      r/=2;
    }
  }
  return rlebuf[--rlep];
}

/* adaptive arithmetic coder */
int   model[258];
int   total;
Ushrt out[PACKSIZE];
int   outl;
int   tmp[PACKSIZE];

void initari() {
  int i;
  outl=0;
  total=255+128;
  model[0]=0;
  for (i=1;i<256;i++) model[i]=1;
  for (i=256;i<258;i++) model[i]=64;
  initrle();
}

Ushrt long_rem(Ushrt d) {
  int i,r=0;
  for (i=packsize-outl;i<packsize;i++) {
    r=((r<<16)+out[i]);out[i]=r/d;r%=d;
  }
  return r;
}

void long_mul(Ushrt m,Ushrt r) {
  int i;
  long t;
  for (i=packsize-1;i>=packsize-outl;i--) {
    t=((long)m)*out[i]+r;out[i]=t;r=t>>16;
  }
  if (r>0) {
    outl++;
    out[i]=r;
  }
}

void pack() {
  int t,p=0,r,q=5;
  initari();
  while ((t=get_rle_word())>0) {
    model[t]++;
    total++;
    tmp[p++]=t;
    if (q) q--,fprintf(stderr,"%d ",t);
  }
  fprintf(stderr,"pack(): phase 2: %d to do, one '.' is 200\n",p);
  while (p--) {
    if (p%200==0) fprintf(stderr,"."),fflush(stderr);
    t=tmp[p];
    model[t]--;
    total--;
    r=long_rem(model[t]);
    while (--t) r+=model[t];
    long_mul(total,r);
  }
  fprintf(stderr,"\n");
}

void init() {
  int f,i;
  f=open(PREFIX COM,O_RDONLY|O_BINARY);
  lseek(f,UNPSIZE,SEEK_SET);
  packsize=read(f,block,PACKSIZE);
  close(f);
  fprintf(stderr,"%d bytes read from "PREFIX COM"\n",packsize);
  for(i=0;i<packsize;i++) block[2*packsize-1-i]=block[i];
  memcpy(block,block+packsize,packsize);
}

void done() {
  int f;
  FILE *i;
  f=open(PREFIX PAK,O_WRONLY|O_BINARY|O_CREAT|O_TRUNC,S_IWUSR|S_IRUSR);
  write(f,out+packsize-outl,outl*2);
  close(f);
  i=fopen(INC,"w");
  fprintf(i,"; This file was generated automatically. Do not edit.\n"
            "\tLAST\tequ\t%d\n"
            "\tLEN\tequ\t%d\n"
            "\tBLOCKL\tequ\t%d\n",first,outl*2,packsize);
  fclose(i);
}

int main() {
  fprintf(stderr,"init()\n");
  init();
  fprintf(stderr,"sort()\n");
  sort();
  fprintf(stderr,"pack()\n");
  pack();
  fprintf(stderr,"done()\n");
  done();
  return 0;
}
