;*************************************************************************
; Program:       ENTRY.COM
; Created by:    Aleksa Todorovic aka Alexione, alexione@mail.ru
; Started:       31-8-2001
; Last modified: 13-10-2001
; Description:   Hugi Compo #15 competition entry
; Compile with:  nasm -fbin -o entry.com entry.asm
;*************************************************************************

picture         equ     03C8h
pixels          equ     (picture + 1078) ; = 07FEh

; Picture is stored in memory in .BMP format at SS:'picture'

%define bx_value        (function_table - 80h)

                org     100h

entry_point:

;*************************************************************************
initialization:
        dw      087FEh ; = 8000h + pixels
        dw      0A2D6h ; = A000h + (36 * 320 + 96) / 16

; FEh 87h D6h A2h       INC     BYTE [BX + 0A2D6h]
; dummy instruction, but carries values for BP and ES...

        les     bp, [si]

; ES:0 points to pixel (96,36)
; BP = 87FEh is used to access bitmap in memory at 'pixels' = 07FEh
;*************************************************************************



;*************************************************************************
load_input_bitmap:
        shr     si, 1 ; SI <- 0080h
        mov     al, 13h ; video mode

        call    set_video_mode_open_access_bitmap

        mov     bl, bx_value ; used to access 'function_table' and to
                             ; calculate access function for files
        mov     cl, 2
;*************************************************************************



;*************************************************************************
refresh_screen:
        mov     ax, 0811h

; AH = 08h / INT 21h - read from STDIN
; AL = 11h is null transformation code (just inverts Y - see end of file
;   for more details)
;*************************************************************************



;*************************************************************************
; see comments at end of file for description of used algorithm and
;   register usage
; this part of codes moves data in both directions memory <-> screen
; when it moves screen -> memory, it performs specific transformation,
;   and in the other direction, it just refreshes picture on screen
; direction is controlled using self-modifying code instruction after
;   processing loop

process_picture:
        ror     di, cl

; first time, DI = FFFEh, so DI <- BFFFh
; other times, DI = 7FFFh, so DI <- DFFFh

; in both cases, highest bit of DI is "eaten" in multiplication few lines
;   below, and, also, by highest bit of BP in memory accesses
; at closer look, there could be problematic situation in second case,
;   because, 14th bit of DI is 1 and it will "go" to the 15th (highest)
;   bit of BX, which means that BL will have value 128 + Y
; but, since, we first do "XCHG BH, BL", then operate on BL (i.e. Y), and
;   then do "AND BL, 127", Y coordinate will always be fixed, and there
;   will be no problems in displaying
; also, transformation/refresh will first be applied to region after
;   picture in memory, and, after that, to the picture, which guaranties
;   correct behaviour of the program

.loop:
        pusha

        imul    bx, di, byte 2
        shr     bl, 1 ; CF <- 0

; after this, BH = Y (+128), BL = X

.calculate_memory_coords:
; input for calculation:
;   CF          = 0 (first time) or bit-6/4/2 (mask)
;   bit-7 (AL)  = bit-7/5/3/1 (mask)

        xchg    bh, bl

        jnc     $ + 4   ; if code 10 or 11 then BL <- 127 - BL
        not     bl

        cbw             ; if code 01 or 11 then AH <- FFh = -1 else AH <- 0
        add     bl, ah  ; if code 01 or 11 then decrement BL
        and     bl, 127

        shl     al, cl  ; advances to next transformation code
        jnz     .calculate_memory_coords

        mov     ah, bh ; AL = 0
        shr     ax, cl
        xchg    si, ax ; SI <- BH * 64

        mov     al, [bp + di]
..@video_mem_access:
        mov     [es:bx + si], al ; 26h 88h 00h (refresh), PF = 1 on (*)
;       mov     al, [es:bx + si] ; 26h 8Ah 00h (transform), PF = 0 on (*)
        mov     [bp + di], al

.init_used_color:
        out     dx, al
        inc     dx
        shr     al, cl
        out     dx, al
        out     dx, al
        out     dx, al

        popa

        dec     di
        js      .loop

        xor     byte [byte bx - bx_value + ..@video_mem_access + 1], cl
; CL = 2 = 88h XOR 8Ah
        jp      refresh_screen ; (*)
;*************************************************************************



;*************************************************************************
wait_key:
        int     21h

        aad     ; AH <- 0, AL <- AL + AH (08) * 10 = AL + 50h
                ; ' ' + 50h = 70h, '0' + 50h = 80h, '1' + 50h = 81h, ...
        xlatb   ; AL <- transformation code (not used if SPACE was hit)
        js      process_picture ; SF from AAD
;*************************************************************************



;*************************************************************************
save_output_bitmap:
        mov     al, 3 ; AH = 0 from 'AAD' above

; set_video_mode_open_access_bitmap immediatly follows
;*************************************************************************



;*************************************************************************
set_video_mode_open_access_bitmap:
; * sets video mode AL
; * fixes bitmap filename which begins at of after [SI]
; * if CL = FFh then opens bitmap, else creates it
; * if BL = 00h then reads from bitmap, else writes to it

.set_video_mode:
        int     10h

        mov     ch, 3Ch

function_table:
        db      01h, 05h, 19h, 54h, 75h, 36h, 31h, 05h, 91h, 40h

; result of execution:
;   * AX = 3D00h (open file for read), or
;   * AH = 3Ch (create file)
; detailed explanation of function_table is given below

.set_creation_attribute:
        mov     cl, ' ' ; = 20h

; CL is used in parsing of filenames
; second time, CL is, also, attribute for 'create'

.fix_filename:
        inc     si
        cmp     [si], cl
        jbe     $ - 3

        mov     dx, si ; DH <- 0

        inc     si
        cmp     [si], cl
        ja      $ - 3

        mov     [si], dh ; DH = 0

.open_bitmap:
        int     21h

        xchg    ax, bx ; AX <- old BX value, BX <- value of handle
        add     ax, 3FFFh

; first time, AX = 0 (BX = 0 at begin), so AH <- 3Fh (read from file)
; second time, AX = bx_value = 00D5h, so AH <- 40h (write to file)

.access_bitmap:
        mov     dx, picture ; = 03C8h
        mov     cx, 1078 + 128 * 128
        int     21h

        retn

; first time, RETN returns to code
; second time, RETN quits program; when program quits, both opened files
;   are closed
;*************************************************************************



;*************************************************************************
; short description of bit-pairs:
;   00 : X -> X
;   01 : X -> X - 1
;   10 : X -> 127 - X
;   11 : X -> 127 - (X + 1)
; used bit-pairs are 87, 65, 43 and 21, where bit-8 is, by convention, 0
; bit-0 is not used in above manner, see below 'termination of loop ...'

;              YYxxYYxx.   key  transformation     X(x,y)  Y(x,y)

;       db      00000001b ; 0 - flip vertical        x     127-y
;       db      00000101b ; 1 - rotate left 180    127-x   127-y

; 01h 05h       ADD     [DI], AX
; has no special effect
; first time DI = -2, so WORD [-2] <- WORD [-2] + AX = 0 + AX = AX
; second time DI = 7FFFh - unimportant change

;       db      00011001b ; 2 - scroll down          x      y+1
;       db      01010100b ; 3 - rotate right 90      y     127-x
;       db      01110101b ; 4 - scroll left         x-1      y

; 19h 54h 75h   SBB     [SI+75h], DX
; has no special effect
; first time, SI = 80h, so SI+75h = F5h - no damage
; second time, 83h <= SI <= 8Eh, so F8h <= SI+75h <= 103h - first few
;   bytes of code could be rewritten, but they are already executed, so
;   it doesn't matter

;       db      00110110b ; 5 - flip horizontal    127-x     y
;       db      00110001b ; 6 - scroll right        x+1      y
;       db      00000101b ; 7 - rotate right 180   127-x   127-y

; 36h 31h 05h   XOR     [SS:DI], AX
; has no special effect
; first time DI = -2, so WORD [-2] <- WORD [-2] XOR AX = AX XOR AX = 0
; second time DI = 7FFFh - unimportant change

;       db      10010001b ; 8 - scroll up            x      y-1

; 91h           XCHG    CX, AX
; AH <- 3Ch

;       db      01000000b ; 9 - rotate left 90     127-y     x

; 40h           INC     AX
; first time, AL = FFh (CX = 00FFh at begin), so AX <- 3D00h
; second time, AL = 02h (CL set to 02h), so AH <- 3Ch
;*************************************************************************



;*************************************************************************
; data in function_table represent inverse functions X(x,y) and Y(x,y),
;   because transformations are applied from screen to memory (that is
;   inverse direction); additionaly, Y coordinate always has to be
;   inversed _before_ transformation (important for cases 2 and 8)

; termination of loop '.calculate_memory_coords' in 'process_picture':
;   * in cases 1, 2, 4, 6, 7, 8, transformed X depends on old X; since
;     bit-2 and bit-1 represent transformation of X, bit-0 = 1 will
;     prevent termination of loop until it goes to CF leaving AL = 0 when
;     further calculations are terminated with BL representing X', and BH
;     representing Y'
;   * in case 0, bit-4 of transformation code has role which bit-0 plays
;     in above situation
;   * in case 3, bit-2 of transformation code has role which bit-0 plays
;     in above situation; in this case, loop will be terminated with
;     Y' in BL, and X' in BH
;   * in case 9, bit-6 of transformation code has role which bit-0 plays
;     in above situation; in this case, loop will be terminated with
;     Y' in BL, and X' in BH
;   * in case 5, last transformation which is performed is 11 on X; in
;     this case, value of AL after transformation will be 80h (the other
;     bit has been "sent" to transformation algorithm through CF), so,
;     after shift left by 2, value of AL will be 0
;*************************************************************************



;*************************************************************************
; Transformation algorithm (written in Pascal):
;
; INPUT : mask = function_table[TransformationID];
;
;   for MemoryAddr := 127 * 128 + 127 downto 0 do
;   begin
;     ScreenX := MemoryAddr mod 128;
;     ScreenY := MemoryAddr div 128;
;
;     Carry := False;
;
;     repeat
;       Exchange (ScreenX, ScreenY);
;
;       if Carry then
;         ScreenX := 127 - ScreenX;
;
;       if Bit7 (mask) = 1 then
;         ScreenX := ScreenX - 1;
;
;       ScreenX := ScreenX and 127;
;
;       mask := mask shl 2 (* plus 'Carry' update *)
;     until mask = 0;
;
;     Memory[MemoryAddr] := Screen[ScreenY * 320 + ScreenX]
;   end;
;
; Registers:
;   AL = mask
;   DI = MemoryAddr
;   BH = ScreenY
;   BL = ScreenX
;*************************************************************************
