;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
;      example SPEW emulator 
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
;
; Assemble:
;   NBASM emu /o
;     ( http://www.cybertrails.com/~fys/newbasic.htm )
;
; Run:
;   EMU demo.spu
;
; This emulator does very little error checking.
;  I will leave it up to you to add error checking
;  if you so desire to improve or create your own
;  emulator from this code.
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

.model tiny
.code
           org  100h

; =-=-=-=-=- get filename from command-line -=-=-=-=-=
start:     mov  si,0081h
skipspcs:  lodsb
           cmp  al,20h
           je   short skipspcs

           lea  dx,[si-1]     ; [DS:DX] --> filename
findend:   lodsb
           cmp  al,20h
           ja   short findend
           mov  byte [si-1],00h

; =-=-=-=-=- load the 4Kb SPU image -=-=-=-=-=
           mov  ax,3D00h
           int  21h
           jnc  short FileOK

; =-=-=-=-=- Else, print error message and exit to DOS
FileEr:    mov  dx,offset FileErr
           mov  ah,09
           int  21h
           .exit

; =-=-=-=-=- Read from file
FileOK:    mov  bx,ax
           mov  cx,4096
           mov  dx,offset mem ; [DS:DX] --> 4096 buffer
           mov  ah,3Fh
           int  21h
           jc   short FileEr

; =-=-=-=-=- Close file
           mov  ah,3Eh
           int  21h

; =-=-=-=-=- fill mem[000]..mem[0FF] with 00h..FFh -=-=-=-=-=
           mov  cx,256
           xor  bx,bx
fillmem:   mov  mem[bx],bl
           inc  bx
           loop fillmem

; =-=-=-=-=- The Emulator -=-=-=-=-=
emulate:   call execute1      ; make it a call, so the RET at the end
           jmp  short emulate ;  of each 'procedure' will come back here

; =-=-=-=-=- Get next Instruction
execute1:  mov  bx,PC
           and  bx,0FFFh
           mov  dx,mem[bx]

; =-=-=-=-=- If high byte of instruction is 00h, then OSCALL
           cmp  dh,00h
           jz   short x_OSCALL ; 00xx ?

; =-=-=-=-=- If instruction is 1000h, then RETURN
           cmp  dx,1000h
           jz   short x_RETURN ; 1000 ?

; =-=-=-=-=- Else, get opcode from high nibble of instruction word
           mov  bx,00F0h
           and  bl,dh
           shr  bx,1          ; .186 would allow  'shr bx,3' !
           shr  bx,1
           shr  bx,1          ; (DX) = opcode word
           jmp  x_dtable[bx]

x_dtable   dw  offset x_JP    ; 0xxx
           dw  offset x_GOSUB ; 1xxx
           dw  offset x_PUSHB ; 2xxx
           dw  offset x_POPB  ; 3xxx
           dw  offset x_LDA   ; 4xxx
           dw  offset x_STA   ; 5xxx
           dw  offset x_RDI   ; 6xxx
           dw  offset x_WRI   ; 7xxx
           dw  offset x_RDSYS ; 8xxx
           dw  offset x_ADDW  ; 9xxx
           dw  offset x_JPcc  ; Axxx
           dw  offset x_ADCA  ; Bxxx
           dw  offset x_SBBA  ; Cxxx
           dw  offset x_ORA   ; Dxxx
           dw  offset x_ANDA  ; Exxx
           dw  offset x_XORA  ; Fxxx

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; 00xx    OSCALL xx    Operating System CALL
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_OSCALL   proc near
           add  word PC,2     ; PC + 2
           mov  ah,dl         ; ah = xx
           mov  al,A          ; al = A
           mov  dl,A          ; dl = A
           call extflags      ; flags <--<-- STATUS
           int  21h           ; int 21h  
           mov  A,al          ; A = al
           call packstatus    ; STATUS <--<-- flags
           ret
x_OSCALL   endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; 0xxx    JP xxx       Jump to new PC address (ie. JMP)
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_JP       proc near
           and  dx,0FFFh
           mov  PC,dx         ; PC = xxx
           ret
x_JP       endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; 1000    RETURN       Return from sub-routine (ie. RET)
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_RETURN   proc near
           mov  bx,STK
           and  bx,0FFFh
           mov  ax,mem[bx]
           mov  PC,ax         ; PC = mem[STK]
           add  word STK,2    ; STK + 2
           ret
x_RETURN   endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; 1xxx    GOSUB xxx    Goto sub-routine (ie. a CALL)
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_GOSUB    proc near
           add  word PC,2     ; PC + 2
           sub  word STK,2    ; STK - 2
           mov  bx,0FFFh
           and  bx,STK
           mov  ax,PC
           mov  mem[bx],ax    ; mem[STK] = PC
           and  dx,0FFFh
           mov  PC,dx         ; PC = xxx
           ret
x_GOSUB    endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; 2xxx    PUSHB xxx    Push memory byte onto the stack
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_PUSHB    proc near
           add  word PC,2     ; PC +2
           dec  word STK      ; STK - 1
           mov  bx,0FFFh
           and  bx,dx
           mov  al,mem[bx]    ; temp = mem[xxx]
           mov  bx,0FFFh
           and  bx,STK
           mov  mem[bx],al    ; mem[STK] = temp
           ret
x_PUSHB    endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; 3xxx    POPB xxx     Pop memory byte from the stack
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_POPB     proc near
           add  word PC,2     ; PC + 2
           mov  bx,0FFFh
           and  bx,STK
           mov  al,mem[bx]    ; temp = mem[STK]
           mov  bx,0FFFh
           and  bx,dx
           mov  mem[bx],al    ; mem[xxx] = temp
           inc  word STK      ; STK + 1
           ret
x_POPB     endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; 4xxx    LDA [xxx]    Load A (accumulator) from memory
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_LDA      proc near
           add  word PC,2     ; PC + 2
           mov  bx,0FFFh
           and  bx,dx
           mov  al,mem[bx]
           mov  A,al          ; A = mem[xxx]
           ret
x_LDA      endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; 5xxx    STA [xxx]    Store A (accumulator) in memory
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_STA      proc near
           add  word PC,2     ; PC + 2
           mov  bx,0FFFh
           and  bx,dx
           mov  al,A
           mov  mem[bx],al    ; mem[xxx] = A
           ret
x_STA      endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; 6xxx    RDI [(xxx)]  Read indirect byte from memory pointer
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_RDI      proc near
           add  word PC,2     ; PC + 2
           mov  bx,0FFFh
           and  bx,dx
           mov  bx,mem[bx]    ; temp = mem[xxx]
           and  bx,0FFFh
           mov  al,mem[bx]
           mov  A,al          ; A = mem[temp]
           ret
x_RDI      endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; 7xxx    WRI [(xxx)]  Write indirect byte from memory-pointer
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_WRI      proc near
           add  word PC,2     ; PC + 2
           mov  bx,0FFFh
           and  bx,dx
           mov  bx,mem[bx]
           and  bx,0FFFh      ; temp = mem[xxx]
           mov  al,A
           mov  mem[bx],al    ; mem[temp] = A
           ret
x_WRI      endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; 8xxx    RDSYS [0000:0xxx] Read a system byte from segment zero.
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_RDSYS    proc near
           add  word PC,2     ; PC + 2
           mov  bx,0FFFh
           and  bx,dx
           push es
           sub  ax,ax
           mov  es,ax
           mov  al,es:[bx]
           pop  es
           mov  A,al          ; A = SYS RAM[0000:0xxx]
           ret
x_RDSYS    endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; 9xxx    ADDW [xxx],A  Add sign-extended A register to word location
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_ADDW     proc near
           add  word PC,2     ; PC + 2
           mov  bx,0FFFh
           and  bx,dx
           mov  al,A
           cbw
           add  mem[bx],ax    ; WORD mem[xxx] + A
           ret
x_ADDW     endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Acpp    JPcc    +pp  Conditional jump instruction (ie. Jcc)
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_JPcc     proc near
           add  word PC,2     ; PC + 2
           and  dh,0Fh        ;
           xor  dh,71h        ; opcode = 70h for jcc and 1 for jNcc
           mov  Here,dh       ; self modify code
           jmp  short ($+2)   ; *clear queue* (for older processors)
           call extflags      ; flags <--<-- STATUS
Here:      db   71h,07        ; if NOT jcc then --\ (07 is length until ret)
x_goJP:    mov  al,dl         ;                   |  (2 bytes)
           cbw                ;                   |  (1 byte)
           add  PC,ax         ; PC = +pp          |  (4 bytes)
           ret                ; <-----------------/  ( = 7 bytes)
x_JPcc     endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Bxxx    ADCA [xxx]   Add byte to the accumulator with Carry
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_ADCA     proc near
           add  word PC,2     ; PC + 2
           mov  bx,0FFFh
           and  bx,dx
           call extflags
           mov  al,mem[bx]
           adc  A,al          ; A + mem[xxx] + CF
           call packstatus
           ret
x_ADCA     endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Cxxx    SBBA [xxx]   Subtract byte to the accumulator with Borrow
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_SBBA     proc near
           add  word PC,2     ; PC + 2
           mov  bx,0FFFh
           and  bx,dx
           call extflags
           mov  al,mem[bx]
           sbb  A,al          ; A - mem[xxx] - CF
           call packstatus
           ret 
x_SBBA     endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Dxxx    ORA [xxx]    Logical OR byte to the accumulator
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_ORA      proc near
           add  word PC,2     ; PC + 2
           mov  bx,0FFFh
           and  bx,dx
           call extflags
           mov  al,mem[bx]
           or   A,al          ; A - mem[xxx] - CF
           call packstatus
           ret 
x_ORA      endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Exxx    ANDA [xxx]   Logical AND byte to the accumulator
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_ANDA     proc near
           add  word PC,2     ; PC + 2
           mov  bx,0FFFh
           and  bx,dx
           call extflags
           mov  al,mem[bx]
           and  A,al          ; A - mem[xxx] - CF
           call packstatus
           ret 
x_ANDA     endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Fxxx    XORA [xxx]   Logical XOR byte to the accumulator
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
x_XORA     proc near
           add  word PC,2     ; PC + 2
           mov  bx,0FFFh
           and  bx,dx
           call extflags
           mov  al,mem[bx]
           xor  A,al          ; A - mem[xxx] - CF
           call packstatus
           ret 
x_XORA     endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; pack SPEW "STATUS <--<-- flags"
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
packstatus proc near
           push  ax
           pushf
           pop  ax
           and  ax,0000100011010101b
           or   al,ah
           mov  STATUS,al
           pop  ax
           ret
packstatus endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; extract 80x86 "flags <--<-- STATUS"
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
extflags   proc near uses ax dx
           mov  al,STATUS
           mov  ah,al
           and  ax,0000100011010101b
           pushf
           pop  dx
           and  dx,1111011100101010b
           or   dx,ax
           push dx
           popf
           ret
extflags   endp

;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
;  Data used by EMU
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
FileErr    db  13,10,'Error with file...',13,10,36


;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
;  the 4096 byte SPEW cpu memory image
;=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
mem        dup 256,?
           dup (0F00h-256),?
A          dup 1,?              ; mem[F00]
STATUS     dup 1,?              ; mem[F01]
STK        dup 2,?              ; mem[F02]..mem[F03]
PC         dup 2,?              ; mem[F04]..mem[F05]
           dup (1000h-0F06h),?

.end start
