/*
   -===========================================================-
   | HUGI Compo #11 -- Example Maze Solver (depth-first)       |
   | by Joergen Ibsen / Jibz <jibz@hotmail.com>                |
   -===========================================================-
*/

/*
   The basic idea is to enumerate all path-fields of the maze
   using a depth-first recursive function. This will guarantee
   that a path is found if there exists one.

     Since we only call recursively on a field if we are able to
   improve on it's count, the count of each field will always be
   the length of the shortest path we have found so far that
   leads from the beginning to that field (in this code the
   length plus 1).

     So when the algorithm is done (ie. it has worked through
   all possible routes in the maze) the end point will have
   count equal to the length of the shortest path. Now we start
   from the end point and work our way back to the begin point.
   This is done because we can not know which turn to take when
   we get to a crossroad if we start from the begin point.

     I did a little optimization by making the recursive
   function return if there is no way it can get a shorter path
   than the best so far.
*/

#include <stdlib.h>
#include <stdio.h>
#include <io.h>
#include <sys/stat.h>
#include <fcntl.h>

/* dimensions of the maze (and thus the MINO file) */
#define MAZELINE (100)
#define MAZESIZE (MAZELINE*MAZELINE)

/* buffers */
unsigned char MINO[MAZESIZE];
unsigned int  maze[MAZESIZE];

int begin_pos = -1, end_pos = -1;
int best_length = MAZESIZE + 1;

long recursive_calls = 0;

/* -------------------------------------------------------------
   | helper functions                                          |
   ------------------------------------------------------------- */

/* compute minimal distance between point a and b */
int dist(int a, int b)
{
   int ax = a % MAZELINE;
   int ay = a / MAZELINE;
   int bx = b % MAZELINE;
   int by = b / MAZELINE;

   return(abs(bx - ax) + abs(by - ay));
}

/* -------------------------------------------------------------
   | the actual maze solving functions                         |
   ------------------------------------------------------------- */

/* enumerate fields to find shortest path */
void enumerate(int pos, int count)
{
   /* count number of recursive calls used */
   recursive_calls++;

   /* enumerate current field */
   maze[pos] = count++;

   /* stop if we are at the end point */
   if (MINO[pos] == 3)
   {
      if (count < best_length) best_length = count;
      return;
   }

   /* stop if there is no way this can result in a shorter path */
   if (count + dist(pos, end_pos) > best_length) return;

   /* recursively enumerate all the sorrounding fields that we
      can improve */
   if (maze[pos + 1] > count) enumerate(pos + 1, count);
   if (maze[pos + MAZELINE] > count) enumerate(pos + MAZELINE, count);
   if (maze[pos - 1] > count) enumerate(pos - 1, count);
   if (maze[pos - MAZELINE] > count) enumerate(pos - MAZELINE, count);
}

/* trace path from end to begin and put in MINO */
void trace_shortest(int pos)
{
   /* get count in end position */
   int count = maze[pos];

   do {
      count--;

      /* find the neighbouring field with value one lower than
         the current -- if there are multiple solutions, any will
         do, as the value is the distance to the begin field */
      if (maze[pos + 1] == count) pos = pos + 1;
      else if (maze[pos + MAZELINE] == count) pos = pos + MAZELINE;
      else if (maze[pos - 1] == count) pos = pos - 1;
      else if (maze[pos - MAZELINE] == count) pos = pos - MAZELINE;

      /* mark field in MINO */
      MINO[pos] = 0x90;

   } while (count > 2);
}

/* -------------------------------------------------------------
   | main                                                      |
   ------------------------------------------------------------- */

int main()
{
   int minofile = -1, tourfile = -1;
   int i;

   /* read MINO file */
   if ((minofile = open("MINO", O_RDONLY | O_BINARY)) == -1)
   {
      printf("\n\nERROR: unable to open 'MINO' file!\n");
      return(-1);
   }

   lseek(minofile, 0, SEEK_SET);
   if (filelength(minofile) != MAZESIZE)
   {
      printf("\n\nERROR: 'MINO' file has wrong length!\n");
      return(-1);
   }
   read(minofile, MINO, MAZESIZE);
   close(minofile);

   /* initialize buffers */
   for (i = 0; i < MAZESIZE; i++)
   {
      /* in maze we want 0 for wall and a large number for path */
      if (MINO[i] == 0) maze[i] = 0; else maze[i] = MAZESIZE + 1;

      /* remember begin- and end point */
      if (MINO[i] == 2) begin_pos = i;
      if (MINO[i] == 3) end_pos = i;
   }

   /* exit if no begin- or no end point found */
   if ((begin_pos == -1) || (end_pos == -1))
   {
      printf("\n\nERROR: maze has no begin- or end point!\n");
      return(-1);
   }

   /* solve maze */
   printf("Working ... ");
   enumerate(begin_pos, 1);

   /* exit if no solution found */
   if (maze[end_pos] == (MAZESIZE + 1))
   {
      printf("\n\nERROR: maze has no solution!\n");
      return(-1);
   }

   printf("Done!\n\n");
   printf("Shortest path was found using %ld recursive calls, and has length %d\n", recursive_calls, maze[end_pos] - 1);

   trace_shortest(end_pos);

   /* write TOUR file */
   if ((tourfile = open("TOUR", O_WRONLY | O_CREAT | O_BINARY | O_TRUNC,
                        S_IREAD | S_IWRITE)) == -1)
   {
      printf("\n\nERROR: unable to create 'TOUR' file!\n");
      return(-1);
   }

   lseek(tourfile, 0, SEEK_SET);
   write(tourfile, MINO, MAZESIZE);
   close(tourfile);

   return(1);
}
