/*
   -===========================================================-
   | HUGI Compo #11 -- Example Maze Solver (breadth-first)     |
   | by Joergen Ibsen / Jibz <jibz@hotmail.com>                |
   -===========================================================-
*/

/*
   The basic idea is to enumerate all path-fields of the maze
   using a breadth-first algorithm. We have a heap data-
   structure which is used to hold the fields we need to
   process.

     You can think of the algorithm as filling water into the
   maze from the begin field. The water flows at a constant rate
   into all parts of the labyrinth it can reach. The algorithm
   goes like this (basically Dijkstra):

     initialize all fields to have 'infinite' count;
     initialize begin_field to have 0 count;

     insert begin_field into heap;

     while heap is not empty do

        v = minimal field from heap;

        for each field u we can reach from v do

           if u.count > v.count + 1 then
              u.count = v.count + 1;
              insert u in heap;
           endif;

        endfor;

     endwhile;
*/

#include <stdlib.h>
#include <stdio.h>
#include <io.h>
#include <sys/stat.h>
#include <fcntl.h>

/* dimensions of the maze (and thus the MINO file) */
#define MAZELINE (100)
#define MAZESIZE (MAZELINE*MAZELINE)

/* buffers */
unsigned char MINO[MAZESIZE];
unsigned int  maze[MAZESIZE];

int begin_pos = -1, end_pos = -1;

long iterations = 0;

/* -------------------------------------------------------------
   | simple heap implementation                                |
   ------------------------------------------------------------- */

typedef struct NODE {
   int val;
   int pos;
} NODE;

NODE heap[MAZESIZE];
int heap_size;

void heap_init()
{
   heap_size = 0;
}

int heap_empty()
{
   return(heap_size == 0);
}

void heap_up(int n)
{
   int father = (n - 1) / 2;

   if ((n > 0) && (heap[father].val > heap[n].val))
   {
      NODE temp = heap[father];
      heap[father] = heap[n];
      heap[n] = temp;

      heap_up(father);
   }
}

void heap_down(int n)
{
   int son = (2 * n) + 1;

   if (son < heap_size)
   {
      if ((son < heap_size - 1) && (heap[son + 1].val < heap[son].val)) son++;

      if (heap[n].val > heap[son].val)
      {
         NODE temp = heap[son];
         heap[son] = heap[n];
         heap[n] = temp;

         heap_down(son);
      }
   }
}

void heap_insert(int p, int v)
{
   heap[heap_size].pos = p;
   heap[heap_size].val = v;
   heap_size++;

   heap_up(heap_size - 1);
}

void heap_deleteMin(NODE *r)
{
   r->val = heap[0].val;
   r->pos = heap[0].pos;
   heap_size--;

   if (heap_size > 0)
   {
      heap[0] = heap[heap_size];
      heap_down(0);
   }
}

/* -------------------------------------------------------------
   | the actual maze solving functions                         |
   ------------------------------------------------------------- */

/* enumerate fields to find shortest path */
void enumerate(int pos)
{
   /* initialize heap */
   heap_init();

   /* enumerate begin field and insert into heap */
   maze[pos] = 1;
   heap_insert(pos, 1);

   /* do algorithm */
   while (!heap_empty())
   {
      int count;
      NODE v;

      /* count number of iterations used */
      iterations++;

      /* v = node with minimal count */
      heap_deleteMin(&v);

      pos = v.pos;
      count = v.val + 1;

      /* stop if we are at the end point */
      if (MINO[pos] == 3) return;

      /* enumerate neighbours and insert into heap */
      if (maze[pos + 1] > count)
      {
         maze[pos + 1] = count;
         heap_insert(pos + 1, count);
      }

      if (maze[pos + MAZELINE] > count)
      {
         maze[pos + MAZELINE] = count;
         heap_insert(pos + MAZELINE, count);
      }

      if (maze[pos - 1] > count)
      {
         maze[pos - 1] = count;
         heap_insert(pos - 1, count);
      }

      if (maze[pos - MAZELINE] > count)
      {
         maze[pos - MAZELINE] = count;
         heap_insert(pos - MAZELINE, count);
      }
   }
}

/* trace path from end to begin and put in MINO */
void trace_shortest(int pos)
{
   /* get count in end position */
   int count = maze[pos];

   do {
      count--;

      /* find the neighbouring field with value one lower than
         the current -- if there are multiple solutions, any will
         do, as the value is the distance to the begin field */
      if (maze[pos + 1] == count) pos = pos + 1;
      else if (maze[pos + MAZELINE] == count) pos = pos + MAZELINE;
      else if (maze[pos - 1] == count) pos = pos - 1;
      else if (maze[pos - MAZELINE] == count) pos = pos - MAZELINE;

      /* mark field in MINO */
      MINO[pos] = 0x90;

   } while (count > 2);
}

/* -------------------------------------------------------------
   | main                                                      |
   ------------------------------------------------------------- */

int main()
{
   int minofile = -1, tourfile = -1;
   int i;

   /* read MINO file */
   if ((minofile = open("MINO", O_RDONLY | O_BINARY)) == -1)
   {
      printf("\n\nERROR: unable to open 'MINO' file!\n");
      return(-1);
   }

   lseek(minofile, 0, SEEK_SET);
   if (filelength(minofile) != MAZESIZE)
   {
      printf("\n\nERROR: 'MINO' file has wrong length!\n");
      return(-1);
   }
   read(minofile, MINO, MAZESIZE);
   close(minofile);

   /* initialize buffers */
   for (i = 0; i < MAZESIZE; i++)
   {
      /* in maze we want 0 for wall and a large number for path */
      if (MINO[i] == 0) maze[i] = 0; else maze[i] = MAZESIZE + 1;

      /* remember begin- and end point */
      if (MINO[i] == 2) begin_pos = i;
      if (MINO[i] == 3) end_pos = i;
   }

   /* exit if no begin- or no end point found */
   if ((begin_pos == -1) || (end_pos == -1))
   {
      printf("\n\nERROR: maze has no begin- or end point!\n");
      return(-1);
   }

   /* solve maze */
   printf("Working ... ");
   enumerate(begin_pos);

   /* exit if no solution found */
   if (maze[end_pos] == (MAZESIZE + 1))
   {
      printf("\n\nERROR: maze has no solution!\n");
      return(-1);
   }

   printf("Done!\n\n");
   printf("Shortest path was found using %ld iterations, and has length %d\n", iterations, maze[end_pos] - 1);

   trace_shortest(end_pos);

   /* write TOUR file */
   if ((tourfile = open("TOUR", O_WRONLY | O_CREAT | O_BINARY | O_TRUNC,
                        S_IREAD | S_IWRITE)) == -1)
   {
      printf("\n\nERROR: unable to create 'TOUR' file!\n");
      return(-1);
   }

   lseek(tourfile, 0, SEEK_SET);
   write(tourfile, MINO, MAZESIZE);
   close(tourfile);

   return(1);
}
