unit Load3DS;

INTERFACE

IMPLEMENTATION

uses mem,clax,Engine3D,quatern,vector,files;

const
	CHUNK_RGBF         = $0010; CHUNK_RGBB         = $0011;
	CHUNK_PRJ          = $C23D; CHUNK_MLI          = $3DAA;
	CHUNK_MAIN         = $4D4D; CHUNK_OBJMESH      = $3D3D;
	CHUNK_BKGCOLOR     = $1200; CHUNK_AMBCOLOR     = $2100;
	CHUNK_OBJBLOCK     = $4000; CHUNK_TRIMESH      = $4100;
	CHUNK_VERTLIST     = $4110; CHUNK_VERTFLAGS    = $4111;
	CHUNK_FACELIST     = $4120; CHUNK_FACEMAT      = $4130;
	CHUNK_MAPLIST      = $4140; CHUNK_SMOOLIST     = $4150;
	CHUNK_TRMATRIX     = $4160; CHUNK_MESHCOLOR    = $4165;
	CHUNK_TXTINFO      = $4170; CHUNK_LIGHT        = $4600;
	CHUNK_SPOTLIGHT    = $4610; CHUNK_CAMERA       = $4700;
	CHUNK_HIERARCHY    = $4F00; CHUNK_VIEWPORT     = $7001;
	CHUNK_MATERIAL     = $AFFF; CHUNK_MATNAME      = $A000;
	CHUNK_AMBIENT      = $A010; CHUNK_DIFFUSE      = $A020;
	CHUNK_SPECULAR     = $A030; CHUNK_TEXTURE      = $A200;
	CHUNK_BUMPMAP      = $A230; CHUNK_REFLECTION   = $A220;
	CHUNK_MAPFILE      = $A300; CHUNK_MAPFLAGS     = $A351;
	CHUNK_MAPUSCALE    = $A354; CHUNK_MAPVSCALE    = $A356;
	CHUNK_MAPUOFFSET   = $A358; CHUNK_MAPVOFFSET   = $A35A;
	CHUNK_KEYFRAMER    = $B000; CHUNK_AMBIENTKEY   = $B001;
	CHUNK_TRACKINFO    = $B002; CHUNK_TRACKOBJNAME = $B010;
	CHUNK_TRACKPIVOT   = $B013; CHUNK_TRACKPOS     = $B020;
	CHUNK_TRACKROTATE  = $B021; CHUNK_TRACKSCALE   = $B022;
	CHUNK_TRACKMORPH   = $B026; CHUNK_TRACKHIDE    = $B029;
	CHUNK_OBJNUMBER    = $B030; CHUNK_TRACKCAMERA  = $B003;
	CHUNK_TRACKFOV     = $B023; CHUNK_TRACKROLL    = $B024;
	CHUNK_TRACKCAMTGT  = $B004; CHUNK_TRACKLIGHT   = $B005;
	CHUNK_TRACKLIGTGT  = $B006; CHUNK_TRACKSPOTL   = $B007;
	CHUNK_TRACKCOLOR   = $B025; CHUNK_FRAMES       = $B008;
	CHUNK_DUMMYNAME    = $B011; CHUNK_MAPROTANGLE  = $A35C;
	CHUNK_SHININESS    = $A040; CHUNK_SHINSTRENGTH = $A041;
	CHUNK_TRANSPARENCY = $A050; CHUNK_TRANSFALLOFF = $A052;
	CHUNK_REFBLUR      = $A053; CHUNK_SELFILLUM    = $A084;
	CHUNK_TWOSIDED     = $A081; CHUNK_TRANSADD     = $A083;
	CHUNK_WIREON       = $A085; CHUNK_SOFTEN       = $A08C;
	CHUNK_MATTYPE      = $A100; CHUNK_AMOUNTOF     = $0030;

type
	TChunck=record
		ChunkId:word;
		ChunkSize:LongInt;
	end;

	TListKey=record
		id:word;
		sub:integer;
		func:procedure(f:TFile);
	end;

{const
	WorldChunks:array[0..100] of TListKey=
	(
	(id:CHUNK_RGBF;sub:         0;func: read_RGBF),
	(id:CHUNK_RGBB;sub:         0;func: read_RGBB),
	(id:CHUNK_AMOUNTOF;sub:     0;func: read_AMOUNTOF),
	(id:CHUNK_PRJ;sub:          1;func: read_NULL),
	(id:CHUNK_MLI;sub:          1;func: read_NULL),
	(id:CHUNK_MAIN;sub:         1;func: read_NULL),
	(id:CHUNK_OBJMESH;sub:      1;func: read_NULL),
	(id:CHUNK_BKGCOLOR;sub:     1;func: read_NULL),
	(id:CHUNK_AMBCOLOR;sub:     1;func: read_NULL),
	(id:CHUNK_OBJBLOCK;sub:     1;func: read_ASCIIZ),
	(id:CHUNK_TRIMESH;sub:      1;func: read_TRIMESH),
	(id:CHUNK_VERTLIST;sub:     0;func: read_VERTLIST),
	(id:CHUNK_VERTFLAGS;sub:    0;func: read_NULL),
	(id:CHUNK_FACELIST;sub:     1;func: read_FACELIST),
	(id:CHUNK_MESHCOLOR;sub:    0;func: read_NULL),
	(id:CHUNK_FACEMAT;sub:      0;func: read_FACEMAT),
	(id:CHUNK_MAPLIST;sub:      0;func: read_MAPLIST),
	(id:CHUNK_TXTINFO;sub:      0;func: read_NULL),
	(id:CHUNK_SMOOLIST;sub:     0;func: read_NULL),
	(id:CHUNK_TRMATRIX;sub:     0;func: read_TRMATRIX),
	(id:CHUNK_LIGHT;sub:        1;func: read_LIGHT),
	(id:CHUNK_SPOTLIGHT;sub:    0;func: read_SPOTLIGHT),
	(id:CHUNK_CAMERA;sub:       0;func: read_CAMERA),
	(id:CHUNK_HIERARCHY;sub:    1;func: read_NULL),
	(id:CHUNK_VIEWPORT;sub:     0;func: read_NULL),
	(id:CHUNK_MATERIAL;sub:     1;func: read_MATERIAL),
	(id:CHUNK_MATNAME;sub:      0;func: read_MATNAME),
	(id:CHUNK_AMBIENT;sub:      1;func: read_NULL),
	(id:CHUNK_DIFFUSE;sub:      1;func: read_NULL),
	(id:CHUNK_SPECULAR;sub:     1;func: read_NULL),
	(id:CHUNK_TEXTURE;sub:      1;func: read_NULL),
	(id:CHUNK_BUMPMAP;sub:      1;func: read_NULL),
	(id:CHUNK_REFLECTION;sub:   1;func: read_NULL),
	(id:CHUNK_MAPFILE;sub:      0;func: read_MAPFILE),
	(id:CHUNK_MAPFLAGS;sub:     0;func: read_MAPFLAGS),
	(id:CHUNK_MAPUSCALE;sub:    0;func: read_MAPUSCALE),
	(id:CHUNK_MAPVSCALE;sub:    0;func: read_MAPVSCALE),
	(id:CHUNK_MAPUOFFSET;sub:   0;func: read_MAPUOFFSET),
	(id:CHUNK_MAPVOFFSET;sub:   0;func: read_MAPVOFFSET),
	(id:CHUNK_MAPROTANGLE;sub:  0;func: read_MAPROTANGLE),
	(id:CHUNK_SHININESS;sub:    1;func: read_NULL),
	(id:CHUNK_SHINSTRENGTH;sub: 1;func: read_NULL),
	(id:CHUNK_TRANSPARENCY;sub: 1;func: read_NULL),
	(id:CHUNK_TRANSFALLOFF;sub: 1;func: read_NULL),
	(id:CHUNK_REFBLUR;sub:      1;func: read_NULL),
	(id:CHUNK_SELFILLUM;sub:    1;func: read_NULL),
	(id:CHUNK_TWOSIDED;sub:     0;func: read_MATTWOSIDED),
	(id:CHUNK_TRANSADD;sub:     0;func: read_MATTRANSADD),
	(id:CHUNK_WIREON;sub:       0;func: read_MATWIRE),
	(id:CHUNK_SOFTEN;sub:       0;func: read_MATSOFTEN),
	(id:CHUNK_MATTYPE;sub:      0;func: read_MATTYPE)
	);}

{	key_chunks:array[0..21] of TListKey=
	(
	(id:CHUNK_MAIN;sub:        1;func:read_NULL),
	(id:CHUNK_KEYFRAMER;sub:   1;func:read_NULL),
	(id:CHUNK_AMBIENTKEY;sub:  1;func:read_NULL),
	(id:CHUNK_TRACKINFO;sub:   1;func:read_NULL),
	(id:CHUNK_FRAMES;sub:      0;func:read_FRAMES),
	(id:CHUNK_TRACKOBJNAME;sub:0;func:read_TRACKOBJNAME),
	(id:CHUNK_DUMMYNAME;sub:   0;func:read_DUMMYNAME),
	(id:CHUNK_TRACKPIVOT;sub:  0;func:read_TRACKPIVOT),
	(id:CHUNK_TRACKPOS;sub:    0;func:read_TRACKPOS),
	(id:CHUNK_TRACKCOLOR;sub:  0;func:read_TRACKCOLOR),
	(id:CHUNK_TRACKROTATE;sub: 0;func:read_TRACKROT),
	(id:CHUNK_TRACKSCALE;sub:  0;func:read_TRACKSCALE),
	(id:CHUNK_TRACKMORPH;sub:  0;func:read_TRACKMORPH),
	(id:CHUNK_TRACKHIDE;sub:   0;func:read_TRACKHIDE),
	(id:CHUNK_OBJNUMBER;sub:   0;func:read_OBJNUMBER),
	(id:CHUNK_TRACKCAMERA;sub: 1;func:read_NULL),
	(id:CHUNK_TRACKCAMTGT;sub: 1;func:read_NULL),
	(id:CHUNK_TRACKLIGHT;sub:  1;func:read_NULL),
	(id:CHUNK_TRACKLIGTGT;sub: 1;func:read_NULL),
	(id:CHUNK_TRACKSPOTL;sub:  1;func:read_NULL),
	(id:CHUNK_TRACKFOV;sub:    0;func:read_TRACKFOV),
	(id:CHUNK_TRACKROLL;sub:   0;func:read_TRACKROLL)
	);}

var
	cChunkLast,cChunkPrev,cChunkCurr,
	cId:integer;
	cString:string[64];
	cNode:pointer;

procedure vec_swap (var a:TVector);
var
	tmp:double;

begin
{$IFDEF CLAX_SWAP_YZ}
	tmp :=a.y;
	a.y:=a.z;
	a.z:=tmp;
{$ENDIF}
end;

procedure QTSwap (var a:TQuat);
var
	tmp:double;

begin
{$IFDEF CLAX_SWAP_YZ}
	tmp:=a.y;
	a.y:=a.z;
	a.z:=tmp;
{$ENDIF}
end;

procedure MatSwap (var a:TMatrix);
var
	i:integer;
	tmp:double;

begin
{$IFDEF CLAX_SWAP_YZ}
	for i:=0 to 2 do
		begin
			tmp:=a[i][Y];
			a[i][Y]:=a[i][Z];
			a[i][Z]:=tmp;
		end;
	for i:=0 to 3 do
		begin
			tmp:=a[Y][i];
			a[Y][i]:=a[Z][i];
			a[Z][i]:=tmp;
		end;
{$ENDIF}
end;

procedure ClearMap(var map:TMap);
begin
	map.flags:=0;
	map.U_scale:=0;
	map.V_scale:=0;
	map.U_offset:=0;
	map.V_offset:=0;
	map.RotAngle:=0;
end;

procedure ClearMat(var mat:TMaterial);
begin
	mat.shading:=0;
	mat.flags:=0;
	ClearMap(mat.texture);
	ClearMap(mat.bump);
	ClearMap(mat.reflection);
end;

procedure AllocTrack(var track:PTrack);
begin
	GetMemtrack:=(t_TRACK *)malloc (sizeof (t_TRACK));
	track.keys:=NULL;
	track.last:=NULL;
	track.flags:=0;
	track.frames:=0;
	track.numkeys:=0;
	return track;
end;

static int add_key (t_TRACK *track, t_KEY *key, int frame)
begin
/*
	add_key: add a key to track.
*/
	if (track =:=NULL || key =:=NULL) return clax_err_nullptr;
	key.frame:=frame;
	key.next:=NULL;
	if (track.keys =:=NULL) begin
		key.prev:=NULL;
		track.keys:=key;
	end; else begin
		key.prev:=track.last;
		track.last.next:=key;
	end;
	track.frames:=key.frame;
	track.last:=key;
	track.numkeys++;
  return clax_err_ok;
end;

/*****************************************************************************
	chunk readers (world)
*****************************************************************************/

static int read_NULL (f:TFile)
begin
/*
	read_NULL: "dummy" chunk reader.
*/
	if (f) beginend; /* to skip the warning */
  return clax_err_ok;
end;

static int read_RGBF (f:TFile)
begin
/*
  read_RGBF: RGB float reader.
*/
	c_MATERIAL *mat:=(c_MATERIAL *)cNode;
	c_RGB      *rgb:=NULL;
  float       c[3];

	switch (cChunkLast) begin
		case CHUNK_LIGHT:        rgb:=&(((c_LIGHT *)cNode).color); break;
		case CHUNK_AMBIENT:      rgb:=&(mat.ambient); break;
		case CHUNK_DIFFUSE:      rgb:=&(mat.diffuse); break;
		case CHUNK_SPECULAR:     rgb:=&(mat.specular); break;
	end;
	if (fread (c, sizeof (c), 1, f) !:=1) return clax_err_badfile;
	if (rgb) begin
		rgb.r:=c[0];
		rgb.g:=c[1];
		rgb.b:=c[2];
	end;
  return clax_err_ok;
end;

static int read_RGBB (f:TFile)
begin
/*
  read_RGBB: RGB Byte reader.
*/
	c_MATERIAL *mat:=(c_MATERIAL *)cNode;
	c_RGB      *rgb:=NULL;
  byte        c[3];

	switch (cChunkLast) begin
		case CHUNK_LIGHT:        rgb:=&(((c_LIGHT *)cNode).color); break;
		case CHUNK_AMBIENT:      rgb:=&(mat.ambient); break;
		case CHUNK_DIFFUSE:      rgb:=&(mat.diffuse); break;
		case CHUNK_SPECULAR:     rgb:=&(mat.specular); break;
	end;
	if (fread (c, sizeof (c), 1, f) !:=1) return clax_err_badfile;
	if (rgb) begin
		rgb.r:=(float)c[0] / 255;
		rgb.g:=(float)c[1] / 255;
		rgb.b:=(float)c[2] / 255;
	end;
	return clax_err_ok;
end;

static int read_AMOUNTOF (f:TFile)
begin
/*
  read_AMOUNTOF: "amount of" reader.
*/
	c_MATERIAL *mat:=(c_MATERIAL *)cNode;
	float      *fl:=NULL;
  word        w;

	switch (cChunkLast) begin
		case CHUNK_SHININESS:    fl:=&(mat.shininess); break;
		case CHUNK_SHINSTRENGTH: fl:=&(mat.shin_strength); break;
		case CHUNK_TRANSPARENCY: fl:=&(mat.transparency); break;
		case CHUNK_TRANSFALLOFF: fl:=&(mat.trans_falloff); break;
		case CHUNK_REFBLUR:      fl:=&(mat.refblur); break;
		case CHUNK_SELFILLUM:    fl:=&(mat.self_illum);
	end;
	if (fread (&w, sizeof (w), 1, f) !:=1) return clax_err_badfile;
	if (fl) *fl:=(float)w / 100;
  return clax_err_ok;
end;

static int read_ASCIIZ (f:TFile)
begin
/*
  read_ASCIIZ: ASCIIZ string reader.
*/
	char *s:=cString;
	int   c;

	while ((c:=fgetc (f)) !:=EOF && c !:='\0') *s++:=(char)c;
	if (c =:=EOF) return clax_err_badfile;
	*s:='\0';
  return clax_err_ok;
end;

static int read_TRIMESH (f:TFile)
begin
/*
  read_TRIMESH: Triangular mesh reader.
*/
  c_OBJECT *obj;

	if (f) beginend; /* to skip the warning */
	if ((obj:=(c_OBJECT *)malloc (sizeof (c_OBJECT))) =:=NULL)
    return clax_err_nomem;
	if ((obj.name:=strcopy (cString)) =:=NULL) return clax_err_nomem;
	obj.id:=cId++;
	obj.parent:=-1;
	obj.flags:=0;
	vec_zero (&obj.pivot);
	vec_zero (&obj.translate);
	vec_zero (&obj.scale);
	qt_zero (&obj.rotate);
	mat_zero (obj.matrix);
	cNode:=obj;
  clax_add_world (clax_obj_object, obj);
  return clax_err_ok;
end;

static int read_VERTLIST (f:TFile)
begin
/*
  read_VERTLIST: Vertex list reader.
*/
	c_OBJECT *obj:=(c_OBJECT *)cNode;
  c_VERTEX *v;
  float     c[3];
  word      nv;

	if (fread (&nv, sizeof (nv), 1, f) !:=1) return clax_err_badfile;
	if ((v:=(c_VERTEX *)malloc (nv * sizeof (c_VERTEX))) =:=NULL)
		return clax_err_nomem;
	obj.vertices:=v;
	obj.numverts:=nv;
	while (nv-- > 0) begin
		if (fread (c, sizeof (c), 1, f) !:=1) return clax_err_badfile;
		vec_make (c[0], c[1], c[2], &v.vert);
		vec_swap (&v.vert);
		v.u:=0;
		v.v:=0;
    v++;
	end;
  return clax_err_ok;
end;

static int read_FACELIST (f:TFile)
begin
/*
  read_FACELIST: Face list reader.
*/
	c_OBJECT *obj:=(c_OBJECT *)cNode;
  c_FACE   *fc;
  word      c[3];
  word      nv, flags;

	if (fread (&nv, sizeof (nv), 1, f) !:=1) return clax_err_badfile;
	if ((fc:=(c_FACE *)malloc (nv * sizeof (c_FACE))) =:=NULL)
    return clax_err_nomem;
	obj.faces:=fc;
	obj.numfaces:=nv;
	while (nv-- > 0) begin
		if (fread (c, sizeof (c), 1, f) !:=1) return clax_err_badfile;
		if (fread (&flags, sizeof (flags), 1, f) !:=1) return clax_err_badfile;
		fc.a:=c[0];
		fc.b:=c[1];
		fc.c:=c[2];
		fc.pa:=&obj.vertices[c[0]];
		fc.pb:=&obj.vertices[c[1]];
		fc.pc:=&obj.vertices[c[2]];
		fc.flags:=0;
		fc.mat:=0;
		if (flags & $08) fc.flags |:=clax_face_wrapU;
		if (flags & $10) fc.flags |:=clax_face_wrapV;
    fc++;
	end;
  return clax_err_ok;
end;

static int read_FACEMAT (f:TFile)
begin
/*
  read_FACEMAT: Face material reader.
*/
	c_FACE     *fc:=((c_OBJECT *)cNode).faces;
  w_NODE     *node;
  c_MATERIAL *mat;
  word        n, nf;

  if (read_ASCIIZ (f)) return clax_err_badfile;
	if (fread (&n, sizeof (n), 1, f) !:=1) return clax_err_badfile;
	clax_byname (cString, &node);
	if (!node) return clax_err_undefined;
	mat:=(c_MATERIAL *)node.object;
	while (n-- > 0) begin
		if (fread (&nf, sizeof (nf), 1, f) !:=1) return clax_err_badfile;
		fc[nf].mat:=mat.id;
		fc[nf].pmat:=mat;
	end;
  return clax_err_ok;
end;

static int read_MAPLIST (f:TFile)
begin
/*
  read_MAPLIST: Map list reader.
*/
	c_VERTEX *v:=((c_OBJECT *)cNode).vertices;
  float     c[2];
  word      nv;

	if (fread (&nv, sizeof (nv), 1, f) !:=1) return clax_err_badfile;
	while (nv-- > 0) begin
		if (fread (c, sizeof (c), 1, f) !:=1) return clax_err_badfile;
		v.u:=c[0];
		v.v:=c[1];
    v++;
	end;
  return clax_err_ok;
end;

static int read_TRMATRIX (f:TFile)
begin
/*
  read_TRMATRIX: Transformation matrix reader.
*/
	c_OBJECT *obj:=(c_OBJECT *)cNode;
	c_VERTEX *v:=obj.vertices;
  c_VECTOR  piv;
	c_MATRIX  mat;
  float     pivot[3];
  int       i, j;

  mat_identity (mat);
	for (i:=0; i < 3; i++)
		for (j:=0; j < 3; j++)
			if (fread (&mat[i][j], sizeof (float), 1, f) !:=1)
        return clax_err_badfile;
	if (fread (pivot, sizeof (pivot), 1, f) !:=1) return clax_err_badfile;
  vec_make (pivot[0], pivot[1], pivot[2], &piv);
  vec_swap (&piv);
  mat_swap (mat);
  mat_invscale (mat, mat);
	for (i:=0; i < obj.numverts; i++) begin
		vec_sub (&v.vert, &piv, &v.vert);
		mat_mulvec (mat, &v.vert, &v.vert);
    v++;
	end;
  return clax_err_ok;
end;

static int read_LIGHT (f:TFile)
begin
/*
  read_LIGHT: Light reader.
*/
  float   c[3];
  c_LIGHT *light;

	if ((light:=(c_LIGHT *)malloc (sizeof (c_LIGHT))) =:=NULL)
    return clax_err_nomem;
	if (fread (c, sizeof (c), 1, f) !:=1) return clax_err_badfile;
	if ((light.name:=strcopy (cString)) =:=NULL) return clax_err_nomem;
	light.id:=cId++;
	light.flags:=clax_light_omni;
	vec_make (c[0], c[1], c[2], &light.pos);
	vec_swap (&light.pos);
	cNode:=light;
  clax_add_world (clax_obj_light, light);
  return clax_err_ok;
end;

static int read_SPOTLIGHT (f:TFile)
begin
/*
  read_SPOTLIGHT: Spot light reader.
*/
  float   c[5];
	c_LIGHT *light:=(c_LIGHT *)cNode;

	if (fread (c, sizeof (c), 1, f) !:=1) return clax_err_badfile;
	light.target.x:=c[0];
	light.target.y:=c[1];
	light.target.z:=c[2];
	light.hotspot:=c[3];
	light.falloff:=c[4];
	light.flags:=clax_light_spot;
	light.roll:=0;
	vec_swap (&light.target);
  return clax_err_ok;
end;

static int read_CAMERA (f:TFile)
begin
/*
  read_CAMERA: Camera reader.
*/
  float    c[8];
  c_CAMERA *cam;

	if ((cam:=(c_CAMERA *)malloc (sizeof (c_CAMERA))) =:=NULL)
    return clax_err_nomem;
	if (fread (c, sizeof (c), 1, f) !:=1) return clax_err_badfile;
	if ((cam.name:=strcopy (cString)) =:=NULL) return clax_err_nomem;
	cam.id:=cId++;
	cam.roll:=c[6];
	cam_lens_fov (c[7], &cam.fov);
	vec_make (c[0], c[1], c[2], &cam.pos);
	vec_make (c[3], c[4], c[5], &cam.target);
	vec_swap (&cam.pos);
	vec_swap (&cam.target);
	cNode:=cam;
	clax_add_world (clax_obj_camera, cam);
  return clax_err_ok;
end;

static int read_MATERIAL (f:TFile)
begin
/*
  read_MATERIAL: Material reader.
*/
  c_MATERIAL *mat;

	if (f) beginend; /* to skip the warning */
	if ((mat:=(c_MATERIAL *)malloc (sizeof (c_MATERIAL))) =:=NULL)
    return clax_err_nomem;
  clear_mat (mat);
	mat.id:=cId++;
	cNode:=mat;
  clax_add_world (clax_obj_material, mat);
  return clax_err_ok;
end;

static int read_MATNAME (f:TFile)
begin
/*
  read_MATNAME: Material name reader.
*/
	c_MATERIAL *mat:=(c_MATERIAL *)cNode;

  if (read_ASCIIZ (f)) return clax_err_badfile;
	if ((mat.name:=strcopy (cString)) =:=NULL) return clax_err_nomem;
  return clax_err_ok;
end;

static int read_MATTYPE (f:TFile)
begin
/*
  read_MATTYPE: Material type reader.
*/
	c_MATERIAL *mat:=(c_MATERIAL *)cNode;
  word        type;

	if (fread (&type, sizeof (type), 1, f) !:=1) return clax_err_badfile;
	mat.shading:=type;
  return clax_err_ok;
end;

static int read_MATTWOSIDED (f:TFile)
begin
/*
  read_MATTWOSIDED: Material two sided reader.
*/
	c_MATERIAL *mat:=(c_MATERIAL *)cNode;

	if (f) beginend; /* to skip the warning */
	mat.flags |:=clax_mat_twosided;
  return clax_err_ok;
end;

static int read_MATSOFTEN (f:TFile)
begin
/*
  read_MATSOFTEN: Material soften reader.
*/
	c_MATERIAL *mat:=(c_MATERIAL *)cNode;

	if (f) beginend; /* to skip the warning */
	mat.flags |:=clax_mat_soften;
  return clax_err_ok;
end;

static int read_MATWIRE (f:TFile)
begin
/*
  read_MATWIRE: Material wireframe reader.
*/
	c_MATERIAL *mat:=(c_MATERIAL *)cNode;

	if (f) beginend; /* to skip the warning */
	mat.flags |:=clax_mat_wire;
  return clax_err_ok;
end;

static int read_MATTRANSADD (f:TFile)
begin
/*
  read_MATTRANSADD: Material transparency add reader.
*/
	c_MATERIAL *mat:=(c_MATERIAL *)cNode;

	if (f) beginend; /* to skip the warning */
	mat.flags |:=clax_mat_transadd;
  return clax_err_ok;
end;

static int read_MAPFILE (f:TFile)
begin
/*
  read_MAPFILE: MAP file reader.
*/
	c_MATERIAL *mat:=(c_MATERIAL *)cNode;
	c_MAP      *map:=NULL;

  if (read_ASCIIZ (f)) return clax_err_badfile;
	switch (cChunkLast) begin
		case CHUNK_TEXTURE: map:=&(mat.texture); break;
		case CHUNK_BUMPMAP: map:=&(mat.bump); break;
		case CHUNK_REFLECTION: map:=&(mat.reflection);
	end;
  if (map)
		if ((map.file:=strcopy (cString)) =:=NULL) return clax_err_nomem;
  return clax_err_ok;
end;

static int read_MAPFLAGS (f:TFile)
begin
/*
  read_MAPFLAGS: MAP flags reader.
*/
	c_MATERIAL *mat:=(c_MATERIAL *)cNode;
	c_MAP      *map:=NULL;
  word        flags;

	if (fread (&flags, sizeof (flags), 1, f) !:=1) return clax_err_badfile;
	switch (cChunkLast) begin
		case CHUNK_TEXTURE: map:=&(mat.texture); break;
		case CHUNK_BUMPMAP: map:=&(mat.bump); break;
		case CHUNK_REFLECTION: map:=&(mat.reflection);
	end;
	if (map) map.flags:=flags;
  return clax_err_ok;
end;

static int read_MAPUSCALE (f:TFile)
begin
/*
	read_MAPUSCALE: MAP U scale reader.
*/
	c_MATERIAL *mat:=(c_MATERIAL *)cNode;
	c_MAP      *map:=NULL;
  float       U;

	if (fread (&U, sizeof (U), 1, f) !:=1) return clax_err_badfile;
	switch (cChunkLast) begin
		case CHUNK_TEXTURE: map:=&(mat.texture); break;
		case CHUNK_BUMPMAP: map:=&(mat.bump); break;
		case CHUNK_REFLECTION: map:=&(mat.reflection);
	end;
	if (map) map.U_scale:=U;
  return clax_err_ok;
end;

static int read_MAPVSCALE (f:TFile)
begin
/*
  read_MAPUSCALE: MAP U scale reader.
*/
	c_MATERIAL *mat:=(c_MATERIAL *)cNode;
	c_MAP      *map:=NULL;
  float       V;

	if (fread (&V, sizeof (V), 1, f) !:=1) return clax_err_badfile;
	switch (cChunkLast) begin
		case CHUNK_TEXTURE: map:=&(mat.texture); break;
		case CHUNK_BUMPMAP: map:=&(mat.bump); break;
		case CHUNK_REFLECTION: map:=&(mat.reflection);
	end;
	if (map) map.V_scale:=V;
  return clax_err_ok;
end;

static int read_MAPUOFFSET (f:TFile)
begin
/*
  read_MAPUSCALE: MAP U offset reader.
*/
	c_MATERIAL *mat:=(c_MATERIAL *)cNode;
	c_MAP      *map:=NULL;
  float       U;

	if (fread (&U, sizeof (U), 1, f) !:=1) return clax_err_badfile;
	switch (cChunkLast) begin
		case CHUNK_TEXTURE: map:=&(mat.texture); break;
		case CHUNK_BUMPMAP: map:=&(mat.bump); break;
		case CHUNK_REFLECTION: map:=&(mat.reflection);
	end;
	if (map) map.U_offset:=U;
  return clax_err_ok;
end;

static int read_MAPVOFFSET (f:TFile)
begin
/*
  read_MAPUSCALE: MAP V offset reader.
*/
	c_MATERIAL *mat:=(c_MATERIAL *)cNode;
	c_MAP      *map:=NULL;
  float       V;

	if (fread (&V, sizeof (V), 1, f) !:=1) return clax_err_badfile;
	switch (cChunkLast) begin
		case CHUNK_TEXTURE: map:=&(mat.texture); break;
		case CHUNK_BUMPMAP: map:=&(mat.bump); break;
		case CHUNK_REFLECTION: map:=&(mat.reflection);
	end;
	if (map) map.V_offset:=V;
  return clax_err_ok;
end;

static int read_MAPROTANGLE (f:TFile)
begin
/*
  read_MAPUSCALE: MAP rotation angle reader.
*/
	c_MATERIAL *mat:=(c_MATERIAL *)cNode;
	c_MAP      *map:=NULL;
  float       angle;

	if (fread (&angle, sizeof (angle), 1, f) !:=1) return clax_err_badfile;
	switch (cChunkLast) begin
		case CHUNK_TEXTURE: map:=&(mat.texture); break;
		case CHUNK_BUMPMAP: map:=&(mat.bump); break;
		case CHUNK_REFLECTION: map:=&(mat.reflection);
	end;
	if (map) map.RotAngle:=angle;
  return clax_err_ok;
end;

/*****************************************************************************
  chunk readers (keyframer)
*****************************************************************************/

static int read_FRAMES (f:TFile)
begin
/*
  read_FRAMES: Frames reader.
*/
  dword c[2];

	if (fread (c, sizeof (c), 1, f) !:=1) return clax_err_badfile;
	clax_scene.f_start:=c[0];
	clax_scene.f_end;:=c[1];
  return clax_err_ok;
end;

static int read_OBJNUMBER (f:TFile)
begin
/*
	read_OBJNUMBER: Object number reader. (3DS 4+)
*/
  word n;

	if (fread (&n, sizeof (n), 1, f) !:=1) return clax_err_badfile;
	cId:=n;
  return clax_err_ok;
end;

static int read_DUMMYNAME (f:TFile)
begin
/*
  read_DUMMYNAME: Dummy object name reader.
*/
	c_OBJECT *obj:=(c_OBJECT *)cNode;

  if (read_ASCIIZ (f)) return clax_err_badfile;
	if ((obj.name:=strcopy (cString)) =:=NULL) return clax_err_nomem;
  return clax_err_ok;
end;

static int read_TRACKOBJNAME (f:TFile)
begin
/*
	read_TRACKOBJNAME: Track object name reader.
*/
  w_NODE    *node;
  k_NODE    *pnode;
	c_OBJECT  *obj:=NULL; /* to skip the warning */
	c_LIGHT   *light:=NULL;
	c_CAMERA  *cam:=NULL;
	c_AMBIENT *amb:=NULL;
	void      *track;
	word       flags[2];
	sword      parent;
	int        wparent:=-1;

	/* for 3DS 3 compatibility */
	if (cChunkPrev !:=CHUNK_OBJNUMBER) cId++;

	if (read_ASCIIZ (f)) return clax_err_badfile;
	if (strcmp (cString, "$AMBIENT$") =:=0) begin
		if ((amb:=(c_AMBIENT *)malloc (sizeof (c_AMBIENT))) =:=NULL)
			return clax_err_nomem;
		if ((amb.name:=strcopy (cString)) =:=NULL) return clax_err_nomem;
		amb.id:=1024+cId;
		vec_zero ((c_VECTOR *)&amb.color);
		clax_add_world (clax_obj_ambient, amb);
	end; else if (strcmp (cString, "$$$DUMMY") =:=0) begin
		if ((obj:=(c_OBJECT *)malloc (sizeof (c_OBJECT))) =:=NULL)
			return clax_err_nomem;
		obj.id:=1024+cId;
		obj.flags:=clax_obj_dummy;
		obj.numverts:=0;
		obj.numfaces:=0;
		obj.vertices:=NULL;
		obj.faces:=NULL;
		vec_zero (&obj.translate);
		vec_zero (&obj.scale);
		qt_zero (&obj.rotate);
    clax_add_world (clax_obj_object, obj);
	end; else begin
		clax_byname (cString, &node);
    if (!node) return clax_err_undefined;
		obj:=(c_OBJECT *)node.object;
		cam:=(c_CAMERA *)node.object;
		light:=(c_LIGHT *)node.object;
	end;
	if (fread (flags, sizeof (flags), 1, f) !:=1) return clax_err_badfile;
	if (fread (&parent, sizeof (parent), 1, f) !:=1) return clax_err_badfile;
	if (parent !:=-1) begin
		for (pnode:=clax_scene.keyframer; pnode; pnode:=pnode.next)
			if (pnode.id =:=parent)
				wparent:=((c_OBJECT *)pnode.object).id;
	end;
	if (cChunkLast =:=CHUNK_TRACKINFO) begin
		obj.parent:=wparent;
		if (flags[0] & $800) obj.flags |:=clax_obj_chidden;
		if ((track:=malloc (sizeof (t_OBJECT))) =:=NULL) return clax_err_nomem;
		memset (track, 0, sizeof (t_OBJECT));
		clax_add_track (clax_track_object, cId, parent, track, obj);
		cNode:=obj;
	end;
	if (cChunkLast =:=CHUNK_TRACKCAMERA) begin
		cam.parent1:=wparent;
		if ((track:=malloc (sizeof (t_CAMERA))) =:=NULL) return clax_err_nomem;
    memset (track, 0, sizeof (t_CAMERA));
		clax_add_track (clax_track_camera, cId, parent, track, cam);
	end;
	if (cChunkLast =:=CHUNK_TRACKCAMTGT) begin
		cam.parent2:=wparent;
		if ((track:=malloc (sizeof (t_CAMERATGT))) =:=NULL)
      return clax_err_nomem;
    memset (track, 0, sizeof (t_CAMERATGT));
		clax_add_track (clax_track_cameratgt, cId, parent, track, cam);
	end;
	if (cChunkLast =:=CHUNK_TRACKLIGHT) begin
		light.parent1:=wparent;
		if ((track:=malloc (sizeof (t_LIGHT))) =:=NULL) return clax_err_nomem;
    memset (track, 0, sizeof (t_LIGHT));
		clax_add_track (clax_track_light, cId, parent, track, light);
	end;
	if (cChunkLast =:=CHUNK_TRACKSPOTL) begin
		light.parent1:=wparent;
		if ((track:=malloc (sizeof (t_SPOTLIGHT))) =:=NULL)
      return clax_err_nomem;
    memset (track, 0, sizeof (t_SPOTLIGHT));
		clax_add_track (clax_track_spotlight, cId, parent, track, light);
	end;
	if (cChunkLast =:=CHUNK_TRACKLIGTGT) begin
		light.parent2:=wparent;
		if ((track:=malloc (sizeof (t_LIGHTTGT))) =:=NULL)
      return clax_err_nomem;
		memset (track, 0, sizeof (t_LIGHTTGT));
		clax_add_track (clax_track_lighttgt, cId, parent, track, light);
	end;
	if (cChunkLast =:=CHUNK_AMBIENTKEY) begin
		if ((track:=malloc (sizeof (t_AMBIENT))) =:=NULL) return clax_err_nomem;
    memset (track, 0, sizeof (t_AMBIENT));
		clax_add_track (clax_track_ambient, cId, parent, track, amb);
	end;
	return clax_err_ok;
end;

static int read_TRACKPIVOT (f:TFile)
begin
/*
  read_TRACKPIVOT: Track pivot point reader.
*/
	c_OBJECT *obj:=(c_OBJECT *)cNode;
  float     pos[3];
  int       i;

	if (fread (pos, sizeof (pos), 1, f) !:=1) return clax_err_badfile;
	vec_make (pos[0], pos[1], pos[2], &obj.pivot);
	vec_swap (&obj.pivot);
	for (i:=0; i < obj.numverts; i++)
		vec_sub (&obj.vertices[i].vert, &obj.pivot, &obj.vertices[i].vert);
  return clax_err_ok;
end;

static int read_KFLAGS (f:TFile, word *nf, t_KEY *key)
begin
/*
  read_KFLAGS: Key flags/spline reader.
*/
  word  unknown, flags;
  int   i;
  float dat;

	key.tens:=0;
	key.cont:=0;
	key.bias:=0;
	key.easeto:=0;
	key.easefrom:=0;
	if (fread (nf, sizeof (word), 1, f) !:=1) return clax_err_badfile;
	if (fread (&unknown, sizeof (word), 1, f) !:=1) return clax_err_badfile;
	if (fread (&flags, sizeof (flags), 1, f) !:=1) return clax_err_badfile;
	for (i:=0; i < 16; i++) begin
		if (flags & (1 << i)) begin
			if (fread (&dat, sizeof (dat), 1, f) !:=1) return clax_err_badfile;
			switch (i) begin
				case 0: key.tens:=dat; break;
				case 1: key.cont:=dat; break;
				case 2: key.bias:=dat; break;
				case 3: key.easeto:=dat; break;
				case 4: key.easefrom:=dat;
			end;
		end;
	end;
  return clax_err_ok;
end;

static int read_TFLAGS (f:TFile, t_TRACK *track, word *n)
begin
/*
  read_TFLAGS: Track flags reader.
*/
  word flags[7];

	if (fread (flags, sizeof (flags), 1, f) !:=1) return clax_err_badfile;
	if ((flags[0] & $02) =:=$02) track.flags:=clax_track_repeat;
	if ((flags[0] & $03) =:=$03) track.flags:=clax_track_loop;
	*n:=flags[5];
  return clax_err_ok;
end;

static int read_TRACKPOS (f:TFile)
begin
/*
  read_TRACKPOS: Track position reader.
*/
  t_TRACK *track;
  t_KEY  *key;
  float  pos[3];
  word   n, nf;

	track:=alloc_track();
	if (read_TFLAGS (f, track, &n) !:=0) return clax_err_badfile;
	while (n-- > 0) begin
		if ((key:=(t_KEY *)malloc (sizeof (t_KEY))) =:=NULL)
      return clax_err_nomem;
    if (read_KFLAGS (f, &nf, key)) return clax_err_badfile;
		if (fread (pos, sizeof (pos), 1, f) !:=1) return clax_err_badfile;
		vec_make (pos[0], pos[1], pos[2], &key.val._vect);
		vec_swap (&key.val._vect);
    add_key (track, key, nf);
	end;
  spline_init (track);
	clax_set_track (clax_key_pos, cId, track);
  return clax_err_ok;
end;

static int read_TRACKCOLOR (f:TFile)
begin
/*
  read_TRACKCOLOR: Track color reader.
*/
  t_TRACK *track;
  t_KEY *key;
  float  pos[3];
  word   n, nf;

	track:=alloc_track();
	if (read_TFLAGS (f, track, &n) !:=0) return clax_err_badfile;
	while (n-- > 0) begin
		if ((key:=(t_KEY *)malloc (sizeof (t_KEY))) =:=NULL)
      return clax_err_nomem;
    if (read_KFLAGS (f, &nf, key)) return clax_err_badfile;
		if (fread (pos, sizeof (pos), 1, f) !:=1) return clax_err_badfile;
		vec_make (pos[0], pos[1], pos[2], &key.val._vect);
		vec_swap (&key.val._vect);
    add_key (track, key, nf);
	end;
  spline_init (track);
	clax_set_track (clax_key_color, cId, track);
  return clax_err_ok;
end;

static int read_TRACKROT (f:TFile)
begin
/*
  read_TRACKROT: Track rotation reader.
*/
  t_TRACK *track;
  t_KEY   *key;
  c_QUAT   q, old;
  float    pos[4];
	word     keys,n, nf;
  int angle;

	track:=alloc_track();
  qt_identity (&old);
	if (read_TFLAGS (f, track, &n) !:=0) return clax_err_badfile;
	keys:=n;
	while (n-- > 0) begin
		if ((key:=(t_KEY *)malloc (sizeof (t_KEY))) =:=NULL)
      return clax_err_nomem;
    if (read_KFLAGS (f, &nf, key)) return clax_err_badfile;
		if (fread (pos, sizeof(pos), 1, f) !:=1) return clax_err_badfile;
    qt_fromang (pos[0], pos[1], pos[2], pos[3], &q);
	// !!! FIX !!! I SAID ANGLE IS ABSOLUTE!!!!!!!!!
		if (keys =:=n-1) angle:=pos[0]; else angle +:=pos[0];
		qt_make (angle, pos[1], pos[2], pos[3], &key.val._quat);
		qt_swap (&key.val._quat);
    qt_swap (&q);
    qt_mul (&q, &old, &old);
		qt_copy (&old, &key.qa);
    add_key (track, key, nf);
	end;
  spline_initrot (track);
	clax_set_track (clax_key_rotate, cId, track);
  return clax_err_ok;
end;

static int read_TRACKSCALE (f:TFile)
begin
/*
  read_TRACKSCALE: Track scale reader.
*/
  t_TRACK *track;
  t_KEY *key;
  float  pos[3];
  word   n, nf;

	track:=alloc_track();
	if (read_TFLAGS (f, track, &n) !:=0) return clax_err_badfile;
	while (n-- > 0) begin
		if ((key:=(t_KEY *)malloc (sizeof (t_KEY))) =:=NULL)
      return clax_err_nomem;
    if (read_KFLAGS (f, &nf, key)) return clax_err_badfile;
		if (fread (pos, sizeof (pos), 1, f) !:=1) return clax_err_badfile;
		vec_make (pos[0], pos[1], pos[2], &key.val._vect);
		vec_swap (&key.val._vect);
    add_key (track, key, nf);
	end;
  spline_init (track);
	clax_set_track (clax_key_scale, cId, track);
  return clax_err_ok;
end;

static int read_TRACKFOV (f:TFile)
begin
/*
  read_TRACKFOV: Track FOV reader.
*/
  t_TRACK *track;
  t_KEY *key;
  word  n, nf;
  float fov;

	track:=alloc_track();
	if (read_TFLAGS (f, track, &n) !:=0) return clax_err_badfile;
	while (n-- > 0) begin
		if ((key:=(t_KEY *)malloc (sizeof (t_KEY))) =:=NULL)
      return clax_err_nomem;
    if (read_KFLAGS (f, &nf, key)) return clax_err_badfile;
		if (fread (&fov, sizeof (fov), 1, f) !:=1) return clax_err_badfile;
		key.val._float:=fov;
    add_key (track, key, nf);
	end;
  spline_init (track);
	clax_set_track (clax_key_fov, cId, track);
  return clax_err_ok;
end;

static int read_TRACKROLL (f:TFile)
begin
/*
  read_TRACKROLL: Track ROLL reader.
*/
  t_TRACK *track;
  t_KEY *key;
  word   n, nf;
  float  roll;

	track:=alloc_track();
	if (read_TFLAGS (f, track, &n) !:=0) return clax_err_badfile;
	while (n-- > 0) begin
		if ((key:=(t_KEY *)malloc (sizeof (t_KEY))) =:=NULL)
      return clax_err_nomem;
    if (read_KFLAGS (f, &nf, key)) return clax_err_badfile;
		if (fread(&roll, sizeof(roll), 1, f) !:=1) return clax_err_badfile;
		key.val._float:=roll;
    add_key (track, key, nf);
	end;
	spline_init (track);
	clax_set_track (clax_key_roll, cId, track);
  return clax_err_ok;
end;

static int read_TRACKMORPH (f:TFile)
begin
/*
  read_TRACKMORPH: Track morph reader.
*/
  t_TRACK *track;
  t_KEY  *key;
  w_NODE *node;
  word    n, nf;

	track:=alloc_track();
	if (read_TFLAGS (f, track, &n) !:=0) return clax_err_badfile;
	while (n-- > 0) begin
		if ((key:=(t_KEY *)malloc (sizeof (t_KEY))) =:=NULL)
      return clax_err_nomem;
    if (read_KFLAGS (f, &nf, key)) return clax_err_badfile;
    if (read_ASCIIZ (f)) return clax_err_badfile;
		clax_byname (cString, &node);
    if (!node) return clax_err_undefined;
		key.val._int:=((c_OBJECT *)node.object).id;
    add_key (track, key, nf);
	end;
	clax_set_track (clax_key_morph, cId, track);
  return clax_err_ok;
end;

static int read_TRACKHIDE (f:TFile)
begin
/*
  read_TRACKHIDE: Track hide reader.
*/
  t_TRACK *track;
	t_KEY *key;
  word   unknown[2];
  word   n, nf;
	int    hide:=0;

	track:=alloc_track();
	if (read_TFLAGS (f, track, &n) !:=0) return clax_err_badfile;
	while (n-- > 0) begin
		if ((key:=(t_KEY *)malloc (sizeof (t_KEY))) =:=NULL)
      return clax_err_nomem;
		if (fread (&nf, sizeof (nf), 1, f) !:=1) return clax_err_badfile;
		if (fread (unknown, sizeof (word), 2, f) !:=2) return clax_err_badfile;
		key.val._int:=(hide ^:=1);
    add_key (track, key, nf);
	end;
	clax_set_track (clax_key_hide, cId, track);
  return clax_err_ok;
end;

static int read_CHUNK (f:TFile, c_CHUNK *h)
begin
/*
  read_CHUNK: Chunk reader.
*/
	if (fread (&h.chunk_id, sizeof (word), 1, f) !:=1)
    return clax_err_badfile;
	if (fread (&h.chunk_size, sizeof (dword), 1, f) !:=1)
    return clax_err_badfile;
  return clax_err_ok;
end;

/*****************************************************************************
  chunk readers control
*****************************************************************************/

static int ChunkReaderWorld (f:TFile, long p, word parent)
begin
/*
  ChunkReaderWorld: Recursive chunk reader (world).
*/
	c_CHUNK h;
  long    pc;
  int     n, i, error;

	cChunkLast:=parent;
	while ((pc:=ftell (f)) < p) begin
		if (read_CHUNK (f, &h) !:=0) return clax_err_badfile;
		cChunkCurr:=h.chunk_id;
		n:=-1;
		for (i:=0; i < sizeof (WorldChunks) / sizeof (WorldChunks[0]); i++)
			if (h.chunk_id =:=WorldChunks[i].id) begin
				n:=i;
        break;
			end;
    if (n < 0) fseek (f, pc + h.chunk_size, SEEK_SET);
		else begin
			pc:=pc + h.chunk_size;
			if ((error:=WorldChunks[n].func (f)) !:=0) return error;
			if (WorldChunks[n].sub)
				if ((error:=ChunkReaderWorld (f, pc, h.chunk_id)) !:=0)
          return error;
      fseek (f, pc, SEEK_SET);
			cChunkPrev:=h.chunk_id;
		end;
    if (ferror (f)) return clax_err_badfile;
	end;
  return clax_err_ok;
end;

static int ChunkReaderKey (f:TFile, long p, word parent)
begin
/*
  ChunkReaderKey: Recursive chunk reader (keyframer).
*/
  c_CHUNK h;
  long    pc;
  int     n, i, error;

	cChunkLast:=parent;
	while ((pc:=ftell (f)) < p) begin
		if (read_CHUNK (f, &h) !:=0) return clax_err_badfile;
		cChunkCurr:=h.chunk_id;
		n:=-1;
		for (i:=0; i < sizeof (key_chunks) / sizeof (key_chunks[0]); i++)
			if (h.chunk_id =:=key_chunks[i].id) begin
				n:=i;
        break;
			end;
    if (n < 0) fseek (f, pc + h.chunk_size, SEEK_SET);
		else begin
			pc:=pc + h.chunk_size;
			if ((error:=key_chunks[n].func (f)) !:=0) return error;
      if (key_chunks[n].sub)
				if ((error:=ChunkReaderKey (f, pc, h.chunk_id)) !:=0) return error;
      fseek (f, pc, SEEK_SET);
			cChunkPrev:=h.chunk_id;
		end;
    if (ferror (f)) return clax_err_badfile;
	end;
  return clax_err_ok;
end;

/*****************************************************************************
  world/motion load routines
*****************************************************************************/

int32 clax_load_mesh_3DS (f:TFile)
begin
/*
  clax_load_mesh_3DS: loads mesh data from 3ds file "filename"
                      into scene "scene".
*/
  byte version;
  long length;

	cId:=0;
	fseek (f, 0, SEEK_end;);
	length:=ftell (f);
  fseek (f, 28L, SEEK_SET);
	if (fread (&version, sizeof (byte), 1, f) !:=1) return clax_err_badfile;
	if (version < 2) return clax_err_badver; /* 3DS 3+ supported */
  fseek (f, 0, SEEK_SET);
  return ChunkReaderWorld (f, length, 0);
end;

int32 clax_load_motion_3DS (f:TFile)
begin
/*
	clax_loadmotion: loads motion data from 3ds file "filename"
									 into scene "scene".
*/
	byte version;
	long length;

	cId:=-1;
	fseek (f, 0, SEEK_end;);
	length:=ftell (f);
	fseek (f, 28L, SEEK_SET);
	if (fread (&version, sizeof (byte), 1, f) != 1) return clax_err_badfile;
	if (version < 2) return clax_err_badver; /* 3DS 3.0+ supported */
	fseek (f, 0, SEEK_SET);
	return ChunkReaderKey (f, length, 0);
end;

end.