extern char* StringToChar ( System::String* s );

// Operators
enum {	OP_PUSH_STREAM, OP_PUSH_CONST, OP_PUSH_NOTE, OP_PUSH_TIME,
		OP_SIN, OP_SAW, OP_NOISE, OP_EXP, OP_COMPRESS,
		OP_MUL, OP_ADD,
		OP_ADSR,										// Parameters: attack, decay, sustain relative time, relative sustain level
		OP_LP_FILTER, OP_HP_FILTER, OP_BP_FILTER,		// Push frequency*2*PI then quality (0..1)
		OP_PAN,											// Push % of sound on right channel
		OP_ECHO,										// Parameters: damping, delay, repeats. If delay is negative, then echo panning is performed
		OP_PUSH_CURRENT,
		OP_SIN_G, OP_SAW_G
};

// Parameter types
enum { PT_BYTE, PT_FLOAT, PT_STREAM, PT_FLT_TYPE };

// Single operation. 
// !!! NEVER add members to child classes since constant operator size is used in file loading !!!
struct Operation {
	int type;
	char name[128];
	SIZE size;
	POINT pos;
	int op;
	int inputCount;
	int paramCount;
	float params[16];
	int paramTypes[16];
	char paramNames[16][16];
	COLORREF backColor;
	// Constructor
	Operation ( ) { 
		type = -1;
		inputCount = 0; 
		paramCount = 0;
		size.cx = 50; size.cy = 15;
		ZeroMemory ( name, sizeof(name) ); 
		backColor = RGB(255,240,255);
		ZeroMemory ( params, sizeof(params) );
		ZeroMemory ( paramTypes, sizeof(paramTypes) );
		ZeroMemory ( paramNames, sizeof(paramNames) );
	}
	// Returns input position
	POINT GetInputPos ( int i ) {
		POINT pt = {pos.x + ((i+0.5) - inputCount*0.5) * (2.0*size.cx/inputCount),pos.y - size.cy};
		return pt;
	}
	// Render operation
	void Draw ( HDC hdc, bool focused, System::String* byteSequenceNames[], POINT& offs ) {
		RECT r = GetRect ( );
		HBRUSH hbr = CreateSolidBrush ( backColor );
		HBRUSH hOldBrush = (HBRUSH)SelectObject ( hdc, hbr );
		::Rectangle ( hdc, r.left + offs.x, r.top + offs.y, r.right + offs.x, r.bottom + offs.y );
		if ( focused )
			::Rectangle ( hdc, r.left+2 + offs.x, r.top+2 + offs.y, r.right-2 + offs.x, r.bottom-2 + offs.y);
		SetTextAlign ( hdc, TA_CENTER );
		COLORREF bk = SetBkColor ( hdc, backColor );
		TextOut ( hdc, pos.x + offs.x, r.top + 4 + offs.y, name, strlen(name) );
		SetBkColor ( hdc, bk );
		SelectObject ( hdc, hOldBrush );
		DeleteObject ( hbr );
		// Output
		Ellipse ( hdc, pos.x - 4 + offs.x, r.bottom - 2 + offs.y, pos.x + 4 + offs.x, r.bottom + 6 + offs.y );
		// Input
		for ( int i = 0; i < inputCount; i++ ) {
			POINT ptI = GetInputPos ( i );
			Ellipse ( hdc, ptI.x - 4 + offs.x, ptI.y + 2 + offs.y, ptI.x + 4 + offs.x, ptI.y - 6 + offs.y );
		}
		// Parameters
		SetTextAlign ( hdc, TA_LEFT );
		for ( int i = 0; i < paramCount; i++ ) {
			RECT r1 = GetParamRect ( i );
			::Rectangle ( hdc, r1.left + offs.x, r1.top + offs.y, r1.right + offs.x, r1.bottom + offs.y );
			if ( paramNames[i][0] ) {
				bk = SetBkColor ( hdc, backColor );
				TextOut ( hdc, r.left + 4 + offs.x, r1.top + 1 + offs.y, paramNames[i], strlen(paramNames[i]) );
				SetBkColor ( hdc, bk );
			}
			char buf[128] = {0};
			if ( paramTypes[i] == PT_STREAM ) {
				int k = (int)params[i];
				if ( k >= 0 )
					sprintf ( buf, "%d", k );
				else if ( byteSequenceNames[-k-1] )
						strcpy ( buf, StringToChar ( byteSequenceNames[-k-1] ) );
			}
			// Filter element is a special
			if ( paramTypes[i] == PT_FLT_TYPE ) {
				if ( type == OP_LP_FILTER )
					strcpy ( buf, "Low pass" );
				if ( type == OP_HP_FILTER )
					strcpy ( buf, "High pass" );
				if ( type == OP_BP_FILTER )
					strcpy ( buf, "Band pass" );
			}
			if ( paramTypes[i] == PT_BYTE )
				sprintf ( buf, "%d", (int)params[i] );
			if ( paramTypes[i] == PT_FLOAT )
				sprintf ( buf, "%.3f", (double)params[i] );
			TextOut ( hdc, r1.left + 2 + offs.x, r1.top + 1 + offs.y, buf, strlen(buf) );
		}
	}
	// Returns bounding rectangle for parameter
	RECT GetParamRect ( int p ) {
		RECT r = { pos.x - size.cx + 30, pos.y - size.cy + 20 + p*20, pos.x + size.cx-4, pos.y - size.cy + 20 + (p+1)*20 };
		return r;
	}
	// Returns bounding rectangle
	RECT GetRect ( ) {
		RECT r = {pos.x - size.cx, pos.y - size.cy, pos.x + size.cx, pos.y + size.cy};
		return r;
	}
	// Generates operation description on stack
	virtual void GetStackBytes ( char* stack, int* count ) { };
};
struct OperConnection {
	int operFrom;
	int operTo;
	int inputNo;
};


// =======================================================
// Push operators
struct PushConstOperation : Operation {
	PushConstOperation ( ) { 
		type = OP_PUSH_CONST;
		strcpy ( name, "Const" );
		paramCount = 1; paramTypes[0] = PT_FLOAT;
		strcpy ( paramNames[0], "Val" );
		size.cy += 10;
		backColor = RGB(230,255,240);
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 5;
		stack[0] = OP_PUSH_CONST;
		stack[1] = ((char*)params)[0];
		stack[2] = ((char*)params)[1];
		stack[3] = ((char*)params)[2];
		stack[4] = ((char*)params)[3];
	};
};
struct PushStreamOperation : Operation {
	PushStreamOperation ( ) { 
		type = OP_PUSH_STREAM;
		strcpy ( name, "Sequence" );
		paramCount = 1; paramTypes[0] = PT_STREAM;
		strcpy ( paramNames[0], "0.01*" );
		size.cy += 10;
		backColor = RGB(230,255,240);
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 2;
		stack[0] = OP_PUSH_STREAM;
		stack[1] = (char)params[0];
	};
};
struct PushNoteOperation : Operation {
	PushNoteOperation ( ) { 
		type = OP_PUSH_NOTE;
		paramCount = 1; paramTypes[0] = PT_BYTE;
		strcpy ( paramNames[0], "Offs" );
		size.cy += 10;
		strcpy ( name, "Note" );
		backColor = RGB(230,255,240);
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 2;
		stack[0] = OP_PUSH_NOTE;
		stack[1] = (char)params[0];
	};
};
struct PushTimeOperation : Operation {
	PushTimeOperation ( ) { 
		type = OP_PUSH_TIME;
		size.cx /= 2;
		strcpy ( name, "Time" );
		backColor = RGB(230,255,240);
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 1;
		stack[0] = OP_PUSH_TIME;
	};
};
struct PushCurrentOperation : Operation {
	PushCurrentOperation ( ) { 
		type = OP_PUSH_CURRENT;
		size.cx /= 2;
		strcpy ( name, "Current" );
		backColor = RGB(230,255,240);
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 1;
		stack[0] = OP_PUSH_CURRENT;
	};
};


// =======================================================
// Oscillators
struct OscSinOperation : Operation {
	OscSinOperation ( ) { 
		type = OP_SIN;
		strcpy ( name, "fr   Sin    ph" );
		inputCount = 2; 
		backColor = RGB(230,240,255);
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 1;
		stack[0] = OP_SIN;
	};
};
struct OscSinGOperation : Operation {
	OscSinGOperation ( ) { 
		type = OP_SIN_G;
		strcpy ( name, "fr   SinG   ph" );
		inputCount = 2; 
		backColor = RGB(230,240,255);
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 1;
		stack[0] = OP_SIN_G;
	};
};
struct OscSawOperation : Operation {
	OscSawOperation ( ) { 
		type = OP_SAW;
		strcpy ( name, "fr   Saw   ph" );
		inputCount = 2; 
		backColor = RGB(230,240,255);
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 1;
		stack[0] = OP_SAW;
	};
};
struct OscSawGOperation : Operation {
	OscSawGOperation ( ) { 
		type = OP_SAW_G;
		strcpy ( name, "fr   SawG  ph" );
		inputCount = 2; 
		backColor = RGB(230,240,255);
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 1;
		stack[0] = OP_SAW_G;
	};
};
struct OscNoiseOperation : Operation {
	OscNoiseOperation ( ) { 
		type = OP_NOISE;
		size.cx /= 2;
		strcpy ( name, "Noise" );
		backColor = RGB(230,240,255);
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 1;
		stack[0] = OP_NOISE;
	};
};
struct OscExpOperation : Operation {
	OscExpOperation ( ) { 
		type = OP_EXP;
		size.cx /= 2;
		strcpy ( name, "Exp" );
		inputCount = 1; 
		backColor = RGB(230,240,255);
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 1;
		stack[0] = OP_EXP;
	};
};


// =======================================================
// Arithmetics
struct AddOperation : Operation {
	AddOperation ( ) { 
		type = OP_ADD;
		size.cx /= 2;
		strcpy ( name, "Add" );
		inputCount = 2;
		backColor = RGB(255,255,240);
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 1;
		stack[0] = OP_ADD;
	};
};
struct MulOperation : Operation {
	MulOperation ( ) { 
		type = OP_MUL;
		size.cx /= 2;
		strcpy ( name, "Mul" );
		inputCount = 2;
		backColor = RGB(255,255,240);
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 1;
		stack[0] = OP_MUL;
	};
};


// =======================================================
// Filters
struct FilterOperation : Operation {
	FilterOperation ( int tp ) { 
		type = tp;
		strcpy ( name, "fr      q      in" );
		inputCount = 3; 
		paramCount = 1; paramTypes[0] = PT_FLT_TYPE;
		strcpy ( paramNames[0], "Type" );
		size.cy += 10;
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 1;
		stack[0] = type;
	};
};


// =======================================================
// Others
struct AdsrOperation : Operation {
	AdsrOperation ( ) { 
		type = OP_ADSR;
		strcpy ( name, "ADSR" );
		inputCount = 1;
		paramCount = 4; paramTypes[0] = PT_BYTE; paramTypes[1] = PT_BYTE; paramTypes[2] = PT_BYTE; paramTypes[3] = PT_BYTE;
		strcpy ( paramNames[0], "A%" ); strcpy ( paramNames[1], "D%" ); strcpy ( paramNames[2], "S%" ); strcpy ( paramNames[3], "SL%" );
		size.cy += 40;	
		params[0] = 10; params[1] = 20; params[2] = 70; params[3] = 80;
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 5;
		stack[0] = OP_ADSR;
		stack[1] = (char)params[0];
		stack[2] = (char)params[1];
		stack[3] = (char)params[2];
		stack[4] = (char)params[3];
	};
};
struct PanOperation : Operation {
	PanOperation ( ) { 
		type = OP_PAN;
		strcpy ( name, "r%   Pan   in" );
		inputCount = 2; 
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 1;
		stack[0] = OP_PAN;
	};
};
struct EchoOperation : Operation {
	EchoOperation ( ) { 
		type = OP_ECHO;
		strcpy ( name, "Echo" );
		inputCount = 1; 
		paramCount = 3; paramTypes[0] = PT_BYTE; paramTypes[1] = PT_BYTE; paramTypes[2] = PT_BYTE;
		strcpy ( paramNames[0], "Dmp" ); strcpy ( paramNames[1], "Del" ); strcpy ( paramNames[2], "Rep" );
		size.cy += 30;	
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 4;
		stack[0] = OP_ECHO;
		stack[1] = (char)params[0];
		stack[2] = (char)params[1];
		stack[3] = (char)params[2];
	};
};
struct CompressOperation : Operation {
	CompressOperation ( ) { 
		type = OP_COMPRESS;
		strcpy ( name, "Compress" );
		inputCount = 1; 
		paramCount = 2; paramTypes[0] = PT_BYTE; paramTypes[1] = PT_BYTE;
		strcpy ( paramNames[0], "Low" ); strcpy ( paramNames[1], "Hi" );
		size.cy += 20;	
	}
	void GetStackBytes ( char* stack, int* count ) { 
		*count = 3;
		stack[0] = OP_COMPRESS;
		stack[1] = (char)params[0];
		stack[2] = (char)params[1];
		if ( stack[1] == stack[2] )
			*count = -1;
	};
};