#ifndef _GS_H_
#define _GS_H_

#include "defines.h"
#include "gif.h"
#include "dma.h"


//===========================================================================
// General definitions
//===========================================================================
typedef enum
{
    PAL_256_256_32 = 0,
	PAL_320_256_32,
	PAL_384_256_32,
	PAL_512_256_32,
	PAL_640_256_32,
    PAL_256_512_32,
	PAL_320_512_32,
	PAL_384_512_32,
	PAL_512_512_32,
	PAL_640_512_32,

	NTSC_256_224_32,
	NTSC_320_224_32,
	NTSC_384_224_32,
	NTSC_512_224_32,
	NTSC_640_224_32,
	NTSC_256_448_32,
	NTSC_320_448_32,
	NTSC_384_448_32,
	NTSC_512_448_32,
	NTSC_640_448_32
} GS_ENUM_VMODE;

typedef struct
{
    uint8 int_mode;     // interlaced flag (1=on, 0=off)
    uint8 ntsc_pal;     // 2=NTSC, 3=PAL
    uint8 field_mode;   // 1=FRAME, 2=FIELD
    uint16 width;       // screen width
    uint16 height;      // screen height
    uint8 magh;         // horiz. magnitude
    uint8 psm;          // pixel storage format
} GS_VMODE_T __attribute__((aligned(16)));

typedef struct
{
    uint8 int_mode;     // interlaced flag (1=on, 0=off)
    uint8 ntsc_pal;     // 2=NTSC, 3=PAL
    uint8 field_mode;   // 1=FRAME, 2=FIELD
    uint16 width;       // screen width
    uint16 height;      // screen height
    uint8 magh;         // horiz. magnitude
    uint8 psm;          // pixel storage format

    uint8 bpp;          // bits per pixel
    uint8 fbw;          // frame buffer width

    uint32 fbp_1;       // frame base pointer: fbp1
    uint32 fbp_2;       // fbp2 (base ptr)
    uint32 zbp;         // z buffer (base ptr)
    uint32 tbp;         // texture buffer (base ptr)

    uint32 frame_1;     // screen switching data...
    uint32 dispfb2_1;   //
    uint32 frame_2;     //
    uint32 dispfb2_2;   //
    uint8  visible;     //
} GS_VMODE __attribute__((aligned(16)));


#define NON_INTERLACED  0
#define INTERLACED      1
#define NTSC            0
#define PAL             1
#define FIELD           0
#define FRAME           1

#define PSMCT32         0
#define PSMCT24         1
#define PSMCT16         2
#define PSMCT16S        10

#define PSMT8           19
#define PSMT4           20
#define PSMT8H          27

#define PSMZ32          0
#define PSMZ24          1
#define PSMZ16          2
#define PSMZ16S         10


//===========================================================================
// externals (asm)
//===========================================================================
extern void gs_reset_dma(void);
extern void gs_send_gif(void *packet_ptr, uint32 packet_size);
extern void gs_wait_gif(void);
extern void gs_reset(uint8 int_mode, uint8 ntsc_pal, uint8 field_mode);
extern void gs_vsync(void);
extern void gs_hsync(void);
extern void gs_install_vrh(void *funct);


//===========================================================================
// externals
//===========================================================================
extern GS_VMODE gs_vmode;

extern uint8 gs_detect_mode(void);  // returns PAL or NTSC
extern void gs_init(uint8 int_mode, uint8 ntsc_pal, uint8 field_mode, uint16 width, uint16 height, uint8 magh, uint8 psm);
extern void gs_init_vmode(GS_ENUM_VMODE vm);
extern void gs_clear_screen(void);
extern void gs_swap_screen(void);
extern void gs_finish(void);
extern void gs_set_viewport(uint16 width, uint16 height);
extern void gs_texflush(void);
extern void gs_clear_zbuff(void);
extern void gs_init_zbuff(uint8 psmz, uint32 z_init);
extern void gs_ee2vram(uint16 tbp, uint16 x, uint16 y, void *img, uint16 w, uint16 h, uint8 psm);
extern void gs_draw_sprite(int16 x, int16 y, uint32 z, uint16 w, uint16 h, uint16 tx, uint16 ty, uint16 tw, uint16 th, uint16 tbp, uint8 psm, uint16 cbp, uint8 cpsm, uint8 tcc, uint8 tfx, uint32 rgba, uint32 flags);

extern void gs_tbuf_reset(void);
extern int16 gs_tbuf_load(void *img, uint16 w, uint16 h, uint8 psm, void *clut, uint8 cpsm, uint8 tcc);
//extern int16 gs_tbuf_draw_texture(int16 x, int16 y, uint32 z, uint16 scale_x, uint16 scale_y, int16 th, uint32 tfx, uint32 flags);
extern int16 gs_tbuf_draw_sprite(int16 x, int16 y, uint32 z, uint16 w, uint16 h, uint16 tx, uint16 ty, uint16 tw, uint16 th, int16 thndl, uint32 flags);
extern int16 gs_tbuf_draw_sprite_mod(int16 x, int16 y, uint32 z, uint16 w, uint16 h, uint16 tx, uint16 ty, uint16 tw, uint16 th, int16 thndl, uint32 rgba, uint32 flags);


//===========================================================================
// Privileged Register Macros
//===========================================================================
#define GS_SET_REGISTER(WHICH,VALUE) \
    *(WHICH) = (uint64)(VALUE);
//---------------------------------------------------------------------------
#define GS_CSR  ((volatile uint64 *)(0x12001000))
#define GS_SET_CSR(SIGNAL,FINISH,HSINT,VSINT,EDWINT,FLUSH,RESET,NFIELD,FIELD,FIFO,REV,ID) \
    *GS_CSR = \
    ((uint64)(SIGNAL)   << 0)       | \
    ((uint64)(FINISH)   << 1)       | \
    ((uint64)(HSINT)    << 2)       | \
    ((uint64)(VSINT)    << 3)       | \
    ((uint64)(EDWINT)   << 4)       | \
    ((uint64)(FLUSH)    << 8)       | \
    ((uint64)(RESET)    << 9)       | \
    ((uint64)(NFIELD)   << 12)      | \
    ((uint64)(FIELD)    << 13)      | \
    ((uint64)(FIFO)     << 14)      | \
    ((uint64)(REV)      << 16)      | \
    ((uint64)(ID)       << 24)
//---------------------------------------------------------------------------
#define GS_RESET() \
    *GS_CSR = ((uint64)(1) << 9)
//---------------------------------------------------------------------------
#define GS_PMODE    ((volatile uint64 *)(0x12000000))
#define GS_SET_PMODE(EN1,EN2,MMOD,AMOD,SLBG,ALP) \
    *GS_PMODE = \
    ((uint64)(EN1)  << 0)   | \
    ((uint64)(EN2)  << 1)   | \
    ((uint64)(001)  << 2)   | \
    ((uint64)(MMOD) << 5)   | \
    ((uint64)(AMOD) << 6)   | \
    ((uint64)(SLBG) << 7)   | \
    ((uint64)(ALP)  << 8)
//---------------------------------------------------------------------------
#define GS_SMODE2   ((volatile uint64 *)(0x12000020))
#define GS_SET_SMODE2(INT,FFMD,DPMS) \
    *GS_SMODE2 = \
    ((uint64)(INT)  << 0)   | \
    ((uint64)(FFMD) << 1)   | \
    ((uint64)(DPMS) << 2)
//---------------------------------------------------------------------------
#define GS_DISPFB1  ((volatile uint64 *)(0x12000070))
#define GS_SET_DISPFB1(FBP,FBW,PSM,DBX,DBY) \
    *GS_DISPFB1 = \
    ((uint64)(FBP)  << 0)   | \
    ((uint64)(FBW)  << 9)   | \
    ((uint64)(PSM)  << 15)  | \
    ((uint64)(DBX)  << 32)  | \
    ((uint64)(DBY)  << 43)
//---------------------------------------------------------------------------
#define GS_DISPLAY1 ((volatile uint64 *)(0x12000080))
#define GS_SET_DISPLAY1(DX,DY,MAGH,MAGV,DW,DH) \
    *GS_DISPLAY1 = \
    ((uint64)(DX)   << 0)   | \
    ((uint64)(DY)   << 12)  | \
    ((uint64)(MAGH) << 23)  | \
    ((uint64)(MAGV) << 27)  | \
    ((uint64)(DW)   << 32)  | \
    ((uint64)(DH)   << 44)
//---------------------------------------------------------------------------
#define GS_DISPFB2  ((volatile uint64 *)(0x12000090))
#define GS_SET_DISPFB2(FBP,FBW,PSM,DBX,DBY) \
    *GS_DISPFB2 = \
    ((uint64)(FBP)  << 0)   | \
    ((uint64)(FBW)  << 9)   | \
    ((uint64)(PSM)  << 15)  | \
    ((uint64)(DBX)  << 32)  | \
    ((uint64)(DBY)  << 43)
//---------------------------------------------------------------------------
#define GS_DISPLAY2 ((volatile uint64 *)(0x120000a0))
#define GS_SET_DISPLAY2(DX,DY,MAGH,MAGV,DW,DH) \
    *GS_DISPLAY2 = \
    ((uint64)(DX)   << 0)   | \
    ((uint64)(DY)   << 12)  | \
    ((uint64)(MAGH) << 23)  | \
    ((uint64)(MAGV) << 27)  | \
    ((uint64)(DW)   << 32)  | \
    ((uint64)(DH)   << 44)
//---------------------------------------------------------------------------
#define GS_BGCOLOR  ((volatile uint64 *)(0x120000e0))
#define GS_SET_BGCOLOR(R,G,B) \
    *GS_BGCOLOR = \
    ((uint64)(R)    << 0)       | \
    ((uint64)(G)    << 8)       | \
    ((uint64)(B)    << 16)
//---------------------------------------------------------------------------



//===========================================================================
// General Purpose Register Macros
//===========================================================================

//---------------------------------------------------------------------------
// ALPHA_x Registers - Setup Alpha Blending Parameters
//   Alpha Formula is: Cv = (A-B)*C>>7 + D
//   For A,B,D - (0=texture, 1=frame buffer, 2=0)
//   For C - (0=texture, 1=frame buffer, 2=use FIX field for Alpha)
//---------------------------------------------------------------------------
#define _GS_ALPHA(A,B,C,D,FIX) \
    (((uint64)(A)   << 0)       | \
     ((uint64)(B)   << 2)       | \
     ((uint64)(C)   << 4)       | \
     ((uint64)(D)   << 6)       | \
     ((uint64)(FIX) << 32))
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// BITBLTBUF Register - Setup Image Transfer Between EE and GS
//   SBP  - Source buffer address (Address/256)
//   SBW  - Source buffer width (Pixels/64)
//   SPSM - Source pixel format (0 = 32bit RGBA)
//   DBP  - Destination buffer address (Address/256)
//   DBW  - Destination buffer width (Pixels/64)
//   DPSM - Destination pixel format (0 = 32bit RGBA)
//
// - When transferring from EE to GS, only the Detination fields
//   need to be set. (Only Source fields for GS->EE, and all for GS->GS).
//---------------------------------------------------------------------------
#define _GS_BITBLTBUF(SBP,SBW,SPSM,DBP,DBW,DPSM) \
    (((uint64)(SBP)     << 0)       | \
     ((uint64)(SBW)     << 16)      | \
     ((uint64)(SPSM)    << 24)      | \
     ((uint64)(DBP)     << 32)      | \
     ((uint64)(DBW)     << 48)      | \
     ((uint64)(DPSM)    << 56))
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// FRAME_x Register
//---------------------------------------------------------------------------
#define _GS_FRAME(FBP,FBW,PSM,FBMSK) \
    (((uint64)(FBP)     << 0)       | \
     ((uint64)(FBW)     << 16)      | \
     ((uint64)(PSM)     << 24)      | \
     ((uint64)(FBMSK)   << 32))
//---------------------------------------------------------------------------
#define _GS_DISPFB(FBP,FBW,PSM,DBX,DBY) \
   (((uint64)(FBP)  << 0)   | \
    ((uint64)(FBW)  << 9)   | \
    ((uint64)(PSM)  << 15)  | \
    ((uint64)(DBX)  << 32)  | \
    ((uint64)(DBY)  << 43))
//---------------------------------------------------------------------------


//===========================================================================
// GS General Purpose Registers
//===========================================================================
#define GS_PRIM         0x00    // Select and configure current drawing primitive
#define GS_RGBAQ        0x01    // Setup current vertex color
#define GS_ST           0x02    // ...
#define GS_UV           0x03    // Specify Vertex Texture Coordinates
#define GS_XYZF2        0x04    // Set vertex coordinate
#define GS_XYZ2         0x05    // Set vertex coordinate and 'kick' drawing
#define GS_TEX0_1       0x06    // Texture Buffer Setup (Context 1)
#define GS_TEX0_2       0x07    // Texture Buffer Setup (Context 2)
#define GS_CLAMP_1      0x08    // ...
#define GS_CLAMP_2      0x09    // ...
#define GS_FOG          0x0a    // ...
#define GS_XYZF3        0x0c    // ...
#define GS_XYZ3         0x0d    // ...
#define GS_TEX1_1       0x14    // ...
#define GS_TEX1_2       0x15    // ...
#define GS_TEX2_1       0x16    // ...
#define GS_TEX2_2       0x17    // ...
#define GS_XYOFFSET_1   0x18    // Mapping from Primitive to Window coordinate system (Context 1)
#define GS_XYOFFSET_2   0x19    // Mapping from Primitive to Window coordinate system (Context 2)
#define GS_PRMODECONT   0x1a    // ...
#define GS_PRMODE       0x1b    // ...
#define GS_TEXCLUT      0x1c    // ...
#define GS_SCANMSK      0x22    // ...
#define GS_MIPTBP1_1    0x34    // ...
#define GS_MIPTBP1_2    0x35    // ...
#define GS_MIPTBP2_1    0x36    // ...
#define GS_MIPTBP2_2    0x37    // ...
#define GS_TEXA         0x3b    // ...
#define GS_FOGCOL       0x3d    // ...
#define GS_TEXFLUSH     0x3f    // Write to this register before using newly loaded texture
#define GS_SCISSOR_1    0x40    // Setup clipping rectangle (Context 1)
#define GS_SCISSOR_2    0x41    // Setup clipping rectangle (Context 2)
#define GS_ALPHA_1      0x42    // Setup Alpha Blending Parameters (Context 1)
#define GS_ALPHA_2      0x43    // Setup Alpha Blending Parameters (Context 2)
#define GS_DIMX         0x44    // ...
#define GS_DTHE         0x45    // ...
#define GS_COLCLAMP     0x46    // ...
#define GS_TEST_1       0x47    // ...
#define GS_TEST_2       0x48    // ...
#define GS_PABE         0x49    // ...
#define GS_FBA_1        0x4a    // ...
#define GS_FBA_2        0x4b    // ...
#define GS_FRAME_1      0x4c    // Frame buffer settings (Context 1)
#define GS_FRAME_2      0x4d    // Frame buffer settings (Context 2)
#define GS_ZBUF_1       0x4e    // ...
#define GS_ZBUF_2       0x4f    // ...
#define GS_BITBLTBUF    0x50    // Setup Image Transfer Between EE and GS
#define GS_TRXPOS       0x51    // Setup Image Transfer Coordinates
#define GS_TRXREG       0x52    // Setup Image Transfer Size
#define GS_TRXDIR       0x53    // Set Image Transfer Directon + Start Transfer
#define GS_HWREG        0x54    // ...
#define GS_SIGNAL       0x60    // ...
#define GS_FINISH       0x61    // ...
#define GS_LABEL        0x62    // ...

//---------------------------------------------------------------------------
// GS_PRIM Register - Setup Drawing Primitive
//   PRI  - Primitive type
//   IIP  - Shading method (0=flat, 1=gouraud)
//   TME  - Texture mapping (0=off, 1=on)
//   FGE  - Fog (0=off, 1=on)
//   ABE  - Alpha Blending (0=off, 1=on)
//   AA1  - Antialiasing (0=off,1=on)
//   FST  - Texture coordinate specification
//              (0=use ST/RGBAQ register, 1=use UV register)
//              (UV means no perspective correction, good for 2D)
//   CTXT - Drawing context (0=1, 1=2)
//   FIX  - ?? Fragment value control (use 0)
//---------------------------------------------------------------------------
#define GS_PRIM_POINT          0
#define GS_PRIM_LINE           1
#define GS_PRIM_LINE_STRIP     2
#define GS_PRIM_TRI            3
#define GS_PRIM_TRI_STRIP      4
#define GS_PRIM_TRI_FAN        5
#define GS_PRIM_SPRITE         6

#define GS_PRIM_GOURAUD    (1<<3)
#define GS_PRIM_TEXTURE    (1<<4)
#define GS_PRIM_FOGGING    (1<<5)
#define GS_PRIM_BLENDING   (1<<6)
#define GS_PRIM_ANTIALIAS  (1<<7)
#define GS_PRIM_TEXT_UV    (1<<8)
#define GS_PRIM_CONTEXT_2  (1<<9)
#define GS_PRIM_FVC_FIXED  (1<<10)

#define _GS_PRIM(PRI,IIP,TME,FGE,ABE,AA1,FST,CTXT,FIX) \
    (((uint64)(PRI)     << 0)       | \
     ((uint64)(IIP)     << 3)       | \
     ((uint64)(TME)     << 4)       | \
     ((uint64)(FGE)     << 5)       | \
     ((uint64)(ABE)     << 6)       | \
     ((uint64)(AA1)     << 7)       | \
     ((uint64)(FST)     << 8)       | \
     ((uint64)(CTXT)    << 9)       | \
     ((uint64)(FIX)     << 10))

//---------------------------------------------------------------------------
// RGBAQ Register
//---------------------------------------------------------------------------
#define _GS_RGBAQ(R,G,B,A,Q) \
    (((uint64)(R)       << 0)       | \
     ((uint64)(G)       << 8)       | \
     ((uint64)(B)       << 16)      | \
     ((uint64)(A)       << 24)      | \
     ((uint64)(Q)       << 32))

//---------------------------------------------------------------------------
// SCISSOR_x Register
//---------------------------------------------------------------------------
#define _GS_SCISSOR(X0,X1,Y0,Y1) \
    (((uint64)(X0)      << 0)       | \
     ((uint64)(X1)      << 16)      | \
     ((uint64)(Y0)      << 32)      | \
     ((uint64)(Y1)      << 48))

//---------------------------------------------------------------------------
// TEST_x Register - Pixel Test Settings
//   ATE   - Alpha Test (0=off, 1=on)
//   ATST  - Alpha Test Method
//             0=NEVER:  All pixels fail.
//             1=ALWAYS: All pixels pass.
//             2=LESS:   Pixels with A less than AREF pass.
//             3=LEQUAL, 4=EQUAL, 5=GEQUAL, 6=GREATER, 7=NOTEQUAL
//   AREF  - Alpha value compared to.
//   AFAIL - What to do when a pixel fails a test.
//             0=KEEP:    Don't update anything.
//             1=FBONLY:  Update frame buffer only.
//             2=ZBONLY:  Update z-buffer only.
//             3=RGBONLY: Update only the frame buffer RGB.
//   DATE  - Destination Alpha Test (0=off, 1=on)
//   DATM  - DAT Mode (0=pass pixels whose destination alpha is 0)
//   ZTE   - Depth Test (0=off, 1=on)
//   ZTST  - Depth Test Method.
//             0=NEVER, 1=ALWAYS, 2=GEQUAL, 3=GREATER
//---------------------------------------------------------------------------
#define GS_ATST_NEVER      0
#define GS_ATST_ALWAYS     1
#define GS_ATST_LESS       2
#define GS_ATST_LEQUAL     3
#define GS_ATST_EQUAL      4
#define GS_ATST_GEQUAL     5
#define GS_ATST_GREATER    6
#define GS_ATST_NOTEQUAL   7

#define GS_AFAIL_KEEP      0
#define GS_AFAIL_FBONLY    1
#define GS_AFAIL_ZBONLY    2
#define GS_AFAIL_RGBONLY   3

#define GS_ZTST_NEVER      0
#define GS_ZTST_ALWAYS     1
#define GS_ZTST_GEQUAL     2
#define GS_ZTST_GREATER    3

#define _GS_TEST(ATE,ATST,AREF,AFAIL,DATE,DATM,ZTE,ZTST) \
    (((uint64)(ATE)     << 0)       | \
     ((uint64)(ATST)    << 1)       | \
     ((uint64)(AREF)    << 4)       | \
     ((uint64)(AFAIL)   << 12)      | \
     ((uint64)(DATE)    << 14)      | \
     ((uint64)(DATM)    << 15)      | \
     ((uint64)(ZTE)     << 16)      | \
     ((uint64)(ZTST)    << 17))

//---------------------------------------------------------------------------
// TEX0_x Register - Set Texture Buffer Information
//   TBP0 - Texture Buffer Base Pointer (Address/256)
//   TBW  - Texture Buffer Width (Texels/64)
//   PSM  - Pixel Storage Format (0 = 32bit RGBA)
//   TW   - Texture Width (Width = 2^TW)
//   TH   - Texture Height (Height = 2^TH)
//   TCC  - Tecture Color Component
//            0=RGB,
//            1=RGBA, use Alpha from TEXA reg when not in PSM
//   TFX  - Texture Function (0=modulate, 1=decal, 2=hilight, 3=hilight2)
//---------------------------------------------------------------------------
#define GS_TEX_MODULATE    0
#define GS_TEX_DECAL       1
#define GS_TEX_HILIGHT     2
#define GS_TEX_HILIGHT2    3

#define _GS_TEX0(TBP0,TBW,PSM,TW,TH,TCC,TFX,CBP,CPSM,CSM,CSA,CLD) \
    (((uint64)(TBP0)    << 0)   | \
     ((uint64)(TBW)     << 14)  | \
     ((uint64)(PSM)     << 20)  | \
     ((uint64)(TW)      << 26)  | \
     ((uint64)(TH)      << 30)  | \
     ((uint64)(TCC)     << 34)  | \
     ((uint64)(TFX)     << 35)  | \
     ((uint64)(CBP)     << 37)  | \
     ((uint64)(CPSM)    << 51)  | \
     ((uint64)(CSM)     << 55)  | \
     ((uint64)(CSA)     << 56)  | \
     ((uint64)(CLD)     << 61))

//---------------------------------------------------------------------------
// TEX1_x Register - Set Texture Information
//   LCM   - LOD calculation method
//   MXL   - Maximum MIP level (0-6)
//   MMAG  - Filter when expanding (0=NEAREST, 1=LINEAR)
//   MMIN  - Filter when reducing (0=NEAREST, 1=LINEAR)
//   MTBA  - MIP Base specified by (0=MIPTBP1&2, 1=Automatic)
//   L     - LOD parameter L
//   K     - LOD parameter K
//---------------------------------------------------------------------------
#define GS_FILTER_NEAREST      0
#define GS_FILTER_LINEAR       1

#define _GS_TEX1(LCM,MXL,MMAG,MMIN,MTBA,L,K) \
    (((uint64)(LCM)     << 0)   | \
     ((uint64)(MXL)     << 2)   | \
     ((uint64)(MMAG)    << 5)   | \
     ((uint64)(MMIN)    << 6)   | \
     ((uint64)(MTBA)    << 9)   | \
     ((uint64)(L)       << 19)  | \
     ((uint64)(K)       << 32))

//---------------------------------------------------------------------------
// TRXDIR Register - Set Image Transfer Directon, and Start Transfer
//   XDIR - (0=EE->GS, 1=GS->EE, 2=GS->GS, 3=Transmission is deactivated)
//---------------------------------------------------------------------------
#define GS_XDIR_EE_GS          0
#define GS_XDIR_GS_EE          1
#define GS_XDIR_GS_GS          2
#define GS_XDIR_DEACTIVATE     3

#define _GS_TRXDIR(XDIR) \
    ((uint64)(XDIR))

//---------------------------------------------------------------------------
// TRXPOS Register - Setup Image Transfer Coordinates
//   SSAX - Source Upper Left X
//   SSAY - Source Upper Left Y
//   DSAX - Destionation Upper Left X
//   DSAY - Destionation Upper Left Y
//   DIR  - Pixel Transmission Order (00 = top left -> bottom right)
//
// - When transferring from EE to GS, only the Detination fields
//   need to be set. (Only Source fields for GS->EE, and all for GS->GS).
//---------------------------------------------------------------------------
#define _GS_TRXPOS(SSAX,SSAY,DSAX,DSAY,DIR)  \
    (((uint64)(SSAX)    << 0)       | \
     ((uint64)(SSAY)    << 16)      | \
     ((uint64)(DSAX)    << 32)      | \
     ((uint64)(DSAY)    << 48)      | \
     ((uint64)(DIR)     << 59))

//---------------------------------------------------------------------------
// TRXREG Register - Setup Image Transfer Size
//   RRW - Image Width
//   RRH - Image Height
//---------------------------------------------------------------------------
#define _GS_TRXREG(RRW,RRH)  \
    (((uint64)(RRW) << 0)       | \
     ((uint64)(RRH) << 32))

//---------------------------------------------------------------------------
// UV Register - Specify Texture Coordinates
//---------------------------------------------------------------------------
#define _GS_UV(U,V)  \
    (((uint64)(U)   << 0)       | \
     ((uint64)(V)   << 16))

//---------------------------------------------------------------------------
// XYZ2 Register
//---------------------------------------------------------------------------
#define _GS_XYZ2(X,Y,Z)  \
    (((uint64)(X)       << 0)       | \
     ((uint64)(Y)       << 16)      | \
     ((uint64)(Z)       << 32))

//---------------------------------------------------------------------------
// XYZF2 Register
//---------------------------------------------------------------------------
#define _GS_XYZF2(X,Y,Z,F)  \
    (((uint64)(X)       << 0)       | \
     ((uint64)(Y)       << 16)      | \
     ((uint64)(Z)       << 32)      | \
     ((uint64)(F)       << 56))

//---------------------------------------------------------------------------
// XYOFFSET_x Register
//---------------------------------------------------------------------------
#define _GS_XYOFFSET(OFX,OFY)    \
    (((uint64)(OFX)     << 0)       | \
     ((uint64)(OFY)     << 32))

//---------------------------------------------------------------------------
// ZBUF_x Register
//---------------------------------------------------------------------------
#define _GS_ZBUF(ZBP,PSM,ZMSK)   \
    (((uint64)(ZBP)     << 0)       | \
     ((uint64)(PSM)     << 24)      | \
     ((uint64)(ZMSK)    << 32))


#endif // _GS_H_
