#include "console.h"
#include "keyboard/keyboard.h"
#include "screen/screen.h"
#include "konsole.h"
#include "command/command_manager.h"
#include <stdio.h>
#include <sys/types.h>
#include <sys/unistd.h>

CConsole::CConsole( void ) :
    Path( "ms0:/" ),
    CursorSpeed( 1 ),
    MaximumLineCount( 256 ),
    MaximumCommandCount( 32 ),
    ScreenLineCount( 10 ),
    LineIndex( 0 ),
    CommandIndex( 0 ),
    CursorIndex( 0 ),
    ItHasEchoOn( true )
{
    char buffer[ MAXPATHLEN ];

    if( getcwd( buffer, MAXPATHLEN ) != 0 )
        SetPath( buffer );

    ScreenLineCount = KONSOLE_Screen().GetTextModeHeight() - 2;

    CommandArray.AddLine( CString( "" ) );

    Write( "Welcome in Konsole !" );
}

CConsole::~CConsole( void )
{
}

void CConsole::SetPath( const CString & path )
{
    Path = path;

    if( Path.GetBuffer()[ Path.GetLength() - 1 ] != '/' )
        Path += '/';
}

const CString & CConsole::GetPath( void ) const
{
    return Path;
}

void CConsole::SetHasEchoOn( bool it_has_echo_on )
{
    ItHasEchoOn = it_has_echo_on;
}

void CConsole::Update( void )
{
    CheckInput();
    Render();
}

void CConsole::Write( CString string )
{
    LineArray.AddLine( string );
    
    if( LineArray.GetLinesCount() > MaximumLineCount )
    {
        LineArray.RemoveFirstLine();
    }
    else
    {
        LineIndex ++;
    }
}

void CConsole::Clear( void )
{
    LineIndex = 0;
    LineArray.SetEmpty();
}

void CConsole::CheckInput( void )
{
    uint32 key_code;
    int32 alt_key, ctrl_key, shift_key;

    getMetaKeys( &alt_key, &ctrl_key, &shift_key );

    key_code = KONSOLE_Keyboard().GetKeyJustPressed();

    if( ctrl_key != 0 )
    {
        if( key_code == KEY_1 ) { KONSOLE_ActivateConsole( 0 ); return; }
        if( key_code == KEY_2 ) { KONSOLE_ActivateConsole( 1 ); return; }
        if( key_code == KEY_3 ) { KONSOLE_ActivateConsole( 2 ); return; }
        if( key_code == KEY_4 ) { KONSOLE_ActivateConsole( 3 ); return; }
        if( key_code == KEY_5 ) { KONSOLE_ActivateConsole( 4 ); return; }
    }

    switch( key_code )
    {
        case NONE:
        {
        } break;

        case KEY_DOWN:
        {
            if( shift_key == 0 )
            {
                if( ctrl_key == 0 )
                {
                    if( CommandArray.GetLinesCount() > 1 )
                    {
                        CommandIndex ++;

                        if( CommandIndex >= CommandArray.GetLinesCount() )
                            CommandIndex = 1;

                        CommandArray[ 0 ] = CommandArray[ CommandIndex ];
                        CursorIndex = CommandArray[ 0 ].GetLength();
                    }
                }
                else
                {
                    LineIndex = LineArray.GetLinesCount();
                }
            }
            else
            {
                if( LineArray.GetLinesCount() > ScreenLineCount
                    && LineIndex < LineArray.GetLinesCount()
                    )
                {
                    LineIndex ++;
                }
            }
        } break;

        case KEY_UP:
        {
            if( shift_key == 0 )
            {
                if( ctrl_key == 0 )
                {
                    if( CommandArray.GetLinesCount() > 1 )
                    {
                        if( CommandIndex > 0 )
                            CommandIndex --;

                        if( CommandIndex < 1 )
                        {
                            CommandIndex = CommandArray.GetLinesCount() - 1;
                        }

                        CommandArray[ 0 ] = CommandArray[ CommandIndex ];
                        CursorIndex = CommandArray[ 0 ].GetLength();
                    }
                }
                else
                {
                    if( LineArray.GetLinesCount() > ScreenLineCount )
                        LineIndex = ScreenLineCount;
                }
            }
            else
            {
                if( LineIndex > ScreenLineCount )
                {
                    LineIndex --;
                }
            }
        } break;

        case KEY_LEFT:
        {
            if( CursorIndex > 0 )
                CursorIndex --;
        } break;

        case KEY_RIGHT:
        {
            if( CursorIndex < CommandArray[ 0 ].GetLength() )
                CursorIndex ++;
        } break;

        case KEY_BACKSPACE:
        {
            if( CursorIndex > 0 )
            {
                CursorIndex --;
                CommandArray[ 0 ].Remove( CursorIndex );
            }

            CommandIndex = 0;
        } break;

        case KEY_DELETE:
        {
            if( CursorIndex < CommandArray[ 0 ].GetLength() )
                CommandArray[ 0 ].Remove( CursorIndex );

            CommandIndex = 0;
        } break;

        case KEY_ENTER:
        {
            if( CommandArray[ 0 ].GetLength() > 0 )
            {
                CString command( CommandArray[ 0 ] );

                if( command[ 0 ] == '@' )
                {
                    command.Remove( 0 );
                }
                else if( ItHasEchoOn )
                {
                    Write( Path + ">" + CommandArray[ 0 ] );
                }

                if( command.GetLength() > 0 )
                {
                    if( !KONSOLE_CommandManager().Execute( command, Path, *this ) )
                        Write( "Invalid Command" );
            
                    CommandArray.AddLine( command );
                    CommandArray[ 0 ] = "";
                }
            }

            CommandIndex = 0;
            CursorIndex = 0;

            if( CommandArray.GetLinesCount() >= MaximumCommandCount )
                CommandArray.RemoveLine( 1 );
        
        } break;

        default:
        {
            CString text( KONSOLE_Keyboard().GetStringFromKey( key_code ) );

            if( text.GetLength() > 0 )
            {
                CommandArray[ 0 ].Insert( text, CursorIndex );
                
                if( CursorIndex < CommandArray[ 0 ].GetLength() )
                    CursorIndex ++;
            }
        } break;
    }
}

void CConsole::Render( void )
{
    ulong line_index, line_count;
    
    line_count = ScreenLineCount;

    if( line_count > LineArray.GetLinesCount() )
        line_count = LineArray.GetLinesCount();
    
    for( line_index = 0 ; line_index < line_count ; line_index ++ )
    {
        KONSOLE_Screen().Print( LineArray[ LineIndex - line_count + line_index ], 0, line_index );
    }    

    CString command_line;

    command_line.Set( "%s\n\n", CommandArray[ 0 ].GetBuffer() );
    
    if( ( KONSOLE_GetTime() * CursorSpeed / 400 ) % 2 == 0 )
        command_line[ CursorIndex ] = '_';
    
    command_line = Path + ">" + command_line;

    KONSOLE_Screen().Print( command_line.GetBuffer(), 0, line_index );
}
