#if 0
The MIT License (MIT)

Copyright (c) 2014 inmensabolademanteca@gmail.com

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
#endif

; ----------------------------------------------------------------------------
; CIDLESA's Altair arcade (1981) port to the ZX Spectrum.
;
; Project started 23 Sep 2013.
;
; Assemble with TASM (Telemark Cross Assembler)
;	tasm -80 -b altair.asm
;
; Then use Bin2Tap to generate the .tap file:
;	bin2tap -b altair.obj
; ----------------------------------------------------------------------------

#include "const.asm"

	.org $8000

start	di

; Set Stack Pointer.

	ld sp,0

; Install a jump to our interrupt routine at address INTERR.

	ld a,$c3
	ld (INTERR),a
	ld hl,interrupt
	ld (INTERR+1),hl

; Create interrupt table and enable interrupts.

	ld hl,INTTAB
	ld bc,257
	ld a,h
	ld i,a
	ld a,INTERR&255
	call memset
	im 2
	ei

; Premare alien images.

	ld hl,pmir0
	call mirims

; Clear VRAM.

	ld hl,vram
	ld bc,VRAMSZ+ARAMSZ
	xor a
	call memset

; Clear Back Buffer and Back Buffer color.

	call clrbbuf

#if MASTER_VERSION == 0

; Generate ABUFW LDI instructions and a RET.

	ld hl,ldit
	ld b,ABUFW
start0	ld (hl),$ed
	inc hl
	ld (hl),$a0
	inc hl
	djnz start0
	ld (hl),$c9

#endif

; ********************************************
; * VRAM to Back Buffer addresses generation *
; ********************************************

; We are going to generate a table of tuples (vram address, backbuffer
; address). Each Back Buffer address is the address of the first pixel
; of a scanline that must be copied, beginning at the corresponding vram
; address. This includes data scanlines and attribute scanlines, so it
; has 8 tuples for pixel scanlines, 1 tuple for the corresponding attribute
; scanline, and so on.
;
; The screen addresses are in the form:
;	010b b000 cccx xxxx
;		bb = one of the three blocks of screen
;		ccc = character row in block
;		xxxxx = character column in row
; Given our Back Buffer Width, we generate the addresses so they will be
; centered horizontally on the screen.

; -----------------------------------------------------------------------
; 1. First generate only the vram addresses and vram attribute addresses.
; -----------------------------------------------------------------------

	ld hl,vadrss
	ld d,BBUFY
	ld e,BBUFX
	ld b,ABUFH
start3	push bc

; Generate 8 scanlines for this character line.

	push de
	call scradr
	ld b,8
start2	ld (hl),e
	inc hl
	ld (hl),d
	inc hl
	inc hl
	inc hl
	inc d
	djnz start2
	pop de

; Generate the attribute address.

	push de
	call atradr
	ld (hl),e
	inc hl
	ld (hl),d
	inc hl
	inc hl
	inc hl
	pop de

; Increment y character position and go to next.

	inc d
	pop bc
	djnz start3

; -----------------------------------------------------------
; 2. Generate the Back Buffer and Attribute Buffer addresses.
; -----------------------------------------------------------

	ld hl,vadrss
	ld de,bbuf
	ld ix,abuf
	ld b,ABUFH
start5	push bc

; Generate 8 Back Buffer scanline addresses.

	ld a,8
	ld bc,ABUFW
start4	inc hl
	inc hl
	ld (hl),e
	inc hl
	ld (hl),d
	inc hl
	ex de,hl
	add hl,bc
	ex de,hl
	dec a
	jr nz,start4

; Generate the Attribute Buffer scanline address.

	push de
	push ix
	pop de
	inc hl
	inc hl
	ld (hl),e
	inc hl
	ld (hl),d
	inc hl
	ex de,hl
	add hl,bc
	ex de,hl
	push de
	pop ix
	pop de

; ABUFH times.

	pop bc
	djnz start5

; ----------------------------------
; End of Scanline Address Generation
; ----------------------------------

	ld a,STATE_DISCLAIMER
	call set_state

theloop
	call update_state

; Increment frame counter.

	ld hl,frames
	inc (hl)

	jr theloop

; -------------------------
; 'chstrf' Change Starfield
; -------------------------
; 	Increments Star Mode Counter and when it reaches 0, we swap the
;	starfield.

chstrf

; Increment Star Mode Counter.

	ld a,(starmc)
	add a,16
	ld (starmc),a
	ret nz

; Star Mode Counter reached 0.
; Configure Draw Stars to erase and erase all stars.

	xor a
	ld (drstrsv),a
	call drstrs

; Configure Draw Stars to draw again.

	ld a,8
	ld (drstrsv),a

; Swap Star Mode.

	ld a,(starm)
	xor 1
	ld (starm),a
	ret

; -------------------
; 'drstrs' Draw Stars
; -------------------
; 	Draws the starfield on Back Buffer.
;	This algorithm can be configured to draw or to erase, by changing
;	the byte at address 'drstrsv'.

drstrs

; Set the start address where to take random values to draw the stars.
; We use two addresses depending on Star Mode, address 0 and adress 192.

	ld a,(starm)
	or a
	jr nz,drstrs1
	ld e,0
	jr drstrs2
drstrs1	ld e,192
drstrs2 ld d,0

; Set HL to backbuf, B to number of lines to paint.

	ld hl,bbuf
	ld b,ABUFH

drstrs8	push bc

; In C we will count the x offset on a line.

	ld c,0

; Get an offset where to paint a new star between 0 and 7.

drstrs7	ld a,(de)
	inc de
	and 7

; Save in B and add to Line Offset.

	ld b,a
	add a,c

; If we have surpassed the line length, go to next line.

	cp ABUFW
	jr nc,drstrs6

; Set new Line Offset.

	ld c,a

; Add the offset to HL.

	ld a,b
	add a,l
	ld l,a
	ld a,0
	adc a,h
	ld h,a

; Draw the star. This is 'ld (hl),8'.
; 'drstarv' is modified from outside so the same routine clears or paints.

	.db $36
drstrsv	.db 8

; Go to next star.

	inc hl
	inc c
	jr drstrs7

; Go to the start of the next line for ABUFH lines.

drstrs6 neg
	add a,ABUFW*8
	add a,l
	ld l,a
	ld a,0
	adc a,h
	ld h,a
	pop bc
	djnz drstrs8
	ret

; -----------------------
; 'dump' Dump Back Buffer
; -----------------------
;	Transfer Back Buffer to Video RAM.

dump	ld hl,vadrss

; Number of scanlines to transfer (pixel scans + attribute scans).

	ld a,BBUFH+ABUFH

dump1	

; Take Video RAM address.

	ld e,(hl)
	inc hl
	ld d,(hl)
	inc hl

; Take Back Buffer or Attribute Buffer address.

	ld c,(hl)
	inc hl
	ld b,(hl)
	inc hl

; Set HL back buffer, DE Video RAM address, and transfer scanline.

	push hl
	ld h,b
	ld l,c
	#if MASTER_VERSION
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	ldi
	#else
	call ldit
	#endif
	pop hl

; For all scanlines.

	dec a
	;jr nz,dump1
	jp nz,dump1
	ret

; ----------
; 'colorize'
; ----------
;	Colorizes the back attribute buffer by changin its paper color.
; Only used for effects. Uses bbuf_paper and bbuf_last_paper. The game
; can change bbuf_paper to apply a new paper color.
;
; Saves	D.

colorize

	ld a,(bbuf_paper)
	cp ABUFCLR
	jr nz,colorize_do
	ld hl,bbuf_last_paper
	cp (hl)
	ret z

colorize_do

	ld (bbuf_last_paper),a

; Draw the color as paper in backbuffer attributes.

	ld hl,abuf
	ld bc,ABUFSZ
	ld e,a
	
colorize_loop

	ld a,(hl)
	and %10000111
	or e
	ld (hl),a
	inc hl
	dec bc
	ld a,b
	or c
	jr nz,colorize_loop
	ret

; ------------------------
; 'drimt' Draw Image Table
; ------------------------
;	An image table is the address of a maximum of 8 words, which specify
;	adresses of images.
;	Which of these 8 images is drawn, is selected by the lower 3 bits of
;	the x pixel position where we want to draw.
;	If we always draw byte aligned, we only need one entry in the image
;	table.
;
; In	HL start of Image table. D,E x,y position in pixels.

drimt

; Calc start of image.

	ld a,d
	call cimadr

; Calc position x in bytes.

	ld a,d
	call tochrp
	ld d,a

; Fall!

; -----------------
; 'drim' Draw Image
; -----------------
;	Draws an image in the Back Buffer, draws its color, and saves the
;	rects where we have painted to be restored later.
;	An image starts with two bytes. The first is the width in bytes; the
;	second is the height in pixels. Then comes the actual image data.
;
; In	HL start of image. D,E x,y position in bytes.

drim

; Calc Back Buffer paint address.

	ex de,hl
	ld bc,bbuf
	call cbpos
	ex de,hl

; Set C width and B height.

	ld c,(hl)
	inc hl
	ld b,(hl)
	inc hl

; Now, HL start of image data. DE address in Back Buffer. B,C height, width.
; Save rect to restore later, and paint.

	push hl
	xor a
	call savrec
	pop hl
	call cpim
	ret

; ------------------------------
; 'tochrp' To Character Position
; ------------------------------
;
; In	A pixel coordinate.
; Out	A character coordinates.
; Saves	BC, DE, HL.

tochrp	and $f8
	rrca
	rrca
	rrca
	ret

; --------------------
; 'inirecs' Init Rects
; --------------------
;	Sets Current Rect Pointer to the start of the Rect List and sets
;	the Number of Rects to 0.
;
; Saves	DE, BC.

inirecs ld hl,reclst
	ld (rectp),hl
	xor a
	ld (nrects),a
	ret

; -------------------------------
; 'savrec' Save Rect in Rect List
; -------------------------------
;	Saves a rect in the rect list if there is space enough (MAXRECS) and
;	points Current Rect Pointer to the next rect and increments Number of
;	Rects.
;
; In	B,C height,width. DE address of first byte of rect. A restore value.
; Saves BC, DE.

savrec	push af
	ld a,(nrects)
	cp MAXRECS
	jr nz,savrec1
	pop af
	ret
savrec1	inc a
	ld (nrects),a
	pop af
	ld hl,(rectp)
	ld (hl),c
	inc hl
	ld (hl),b
	inc hl
	ld (hl),e
	inc hl
	ld (hl),d
	inc hl
	ld (hl),a
	inc hl
	ld (rectp),hl
	ret

; ---------------------
; 'blkrecs' Blank Rects
; ---------------------
;	Goes through the Rect List and calls 'erase' for each.

blkrecs	ld hl,reclst
	ld a,(nrects)
blkrec1	or a
	ret z
	push af
	ld c,(hl)
	inc hl
	ld b,(hl)
	inc hl
	ld e,(hl)
	inc hl
	ld d,(hl)
	inc hl
	ld a,(hl)
	inc hl
	push hl
	call erase
	pop hl
	pop af
	dec a
	jr blkrec1

; ------------------
; 'erase' Erase Rect
; ------------------
;	Fills a rect with a value. The rect is assummed in a buffer with a
;	with of ABUFW bytes.
;
; In:	B,C height,width in bytes. DE address of first byte.
;	A value to erase to.

erase

; Set DE as the offset we have to add to go to the next line at destiny.

	ex af,af'
	ld a,ABUFW
	sub c
	ld l,a
	ex af,af'
	ld h,0
	ex de,hl

; Fill rect.

erase2	push bc
erase1 	ld (hl),a
	inc hl
	dec c
	jr nz,erase1
	pop bc
	add hl,de
	djnz erase2
	ret	

; ---------------------------
; 'cimadr' Calc Image Address
; ---------------------------
;	Given an Image Table, returns the address of the start of an Image
;	given the lower 3 bits of the x position where the image is to be
;	painted.
;
; In	HL Start of an Image table. A x position in pixels.
; Out	HL address of Image.
; Saves	DE.

cimadr	and 7
	rlca
	ld c,a
	ld b,0
	add hl,bc
	ld a,(hl)
	inc hl
	ld h,(hl)
	ld l,a
	ret

; -----------------
; 'cpim' Copy Image
; -----------------
;	Copies a set of bytes into a buffer, arranged as rows and columns.
;	The buffer where to paint is assumed to be have a width of ABUFW
;	bytes.
;
; In	B,C is height in bytes, width in bytes. HL start of image data.
; 	DE address where to paint.

cpim	

; Calc offset between buffer lines.

	ld a,ABUFW
	sub c

; Copy all lines.

cpim1	push bc

; Copy a line.

	ld b,0
	ldir

#if 0
	push af
	ld b,c
cpim2	ld a,(de)
	or (hl)
	ld (de),a
	inc hl
	inc de
	djnz cpim2
	pop af
#endif

; Here B is 0. Go to next line at destiny.

	ex de,hl
	ld c,a
	add hl,bc
	ex de,hl

; For height lines.

	pop bc
	djnz cpim1
	ret

; ----------------------------
; 'cbpos' Calc Buffer position
; ----------------------------
;	Calculates y * 24 + x + offset.
;
; In	BC start of buffer. H,L x in bytes, y in bytes.
; Out	HL address in buffer.
; Saves	DE.

cbpos	push bc
	ld a,h
	ld h,0

; Multiply y position by 8, save in BC.

	add hl,hl
	add hl,hl
	add hl,hl
	ld c,l
	ld b,h

; Y position multiplied by 16.

	add hl,hl

; Set HL = y*16 + y*8 = y*24.

	add hl,bc

; Add x position.

	ld c,a
	ld b,0
	add hl,bc

; Add offset, ie buffer address.

	pop bc
	add hl,bc
	ret

; ---------------------
; 'drsprs' Draw Sprites
; ---------------------
;	Runs through the Sprite Table and draws all sprites.

drsprs	ld b,NSPRS
	ld hl,sprtab
drsprs1	push bc
	call drspr
	pop bc
	djnz drsprs1
	ret

; -------------------
; 'drspr' Draw Sprite
; -------------------
;	Draws a Sprite in the Back Buffer and Attribute Buffer.
;
; In	HL sprite ptr.
; Out	HL next sprite ptr.

drspr

; Set IX as Sprite pointer.

	push hl
	pop ix

; HL Image Table address.

	ld l,(ix+SP_ITL)
	ld h,(ix+SP_ITH)

; If address is 0, go to next sprite.

	ld a,h
	or l
	jr z,drspr1

; Set IY as the Image address.

	ld a,(ix+SP_PX)
	call cimadr
	push hl
	pop iy

; Byte align x position and draw image.

	ld a,(ix+SP_PX)
	call tochrp
	ld d,a
	ld e,(ix+SP_PY)
	call drim

; Check if we colorize or apply color pattern.

	ld a,(ix+SP_COH)
	or a
	jr z,drspr2
	ld a,(ix+SP_PX)
	or (ix+SP_PY)
	and 7
	jr nz,drspr2

; Apply color pattern.
; 1. Transform positions to character positions.

	ld a,(ix+SP_PX)
	call tochrp
	ld d,a
	ld a,(ix+SP_PY)
	call tochrp
	ld e,a

; 2. Get pattern address and draw.

	ld l,(ix+SP_COL)
	ld h,(ix+SP_COH)
	call drptrn
	jr drspr1

; Colorize.
; Fix y position and dimension to draw color.

drspr2	ld e,(ix+SP_PY)
	ld c,(iy+IM_HEI)
	call chrfix

; Byte align x position.

	ld a,(ix+SP_PX)
	call tochrp

; Calc draw adress in HL.

	ld h,a
	ld l,e
	push bc
	ld bc,abuf
	call cbpos
	pop bc

; Set BC height, width.

	ld b,c
	ld c,(iy+IM_WID)

; Save color rect.

	ex de,hl
	ld a,ABUFCLR
	call savrec

; Draw color using Erase.

	ld a,(ix+SP_COL)
	call erase

; Set HL to point to the next Sprite.

drspr1	push ix
	pop hl
	ld bc,SPRSZ
	add hl,bc
	ret

; ---------------------------
; 'drptrn' Draw Color Pattern
; ---------------------------
;
; In	HL address of color pattern. DE x,y in characters.

drptrn

; Calc first byte position.

	ex de,hl
	ld bc,abuf
	call cbpos

; Take dimensions of pattern.

	ex de,hl
	ld c,(hl)
	inc hl
	ld b,(hl)
	inc hl

; Save rect.

	push hl
	ld a,ABUFCLR
	call savrec
	pop hl

; Draw pattern.

	call cpim
	ret

; --------------------------------
; 'chrfix' Fix Dimension For Color
; --------------------------------
;	Given the y position in pixels and the height in pixels, calculate
;	the y position character aligned and the height in characters that
;	is needed to fill the area.
;
; In	E y position in pixels. C height in pixels.
; Out	E y in characters. C height in characters.
; Saves	HL

chrfix	

; Set height in characters to (y & 7 + h + 7) / 8.

	ld a,e
	and 7
	add a,c
	add a,7
	call tochrp
	ld c,a

; Set y position in characters.

	ld a,e
	call tochrp
	ld e,a
	ret
 
; ----------------------------
; 'scradr' Calc Screen Address
; ----------------------------
;
; In	DE y,x position in characters.
; Out	DE screen address.
; Saves	HL, BC

; A screen address has the form 010B B000 LLLC CCCC and on entry we have
; 000B BLLL 000C CCCC.

scradr	ld a,d
	and 7
	rrca
	rrca
	rrca
	or e
	ld e,a
	ld a,d
	and $18
	or $40
	ld d,a
	ret

; -------------------------------
; 'atradr' VRAM Attribute Address
; -------------------------------
;	Given position in characters, calculates the corresponding vram
;	attribute adddress.
;
; In	D,E y,x in character coordinates.
; Out	DE vram attribute address.
; Saves	HL, BC

; An attribute address has the form 010110BB LLLC CCCC and on entry
; we have 000B BLLL 000C CCCC .

atradr	ld a,d
	rrca
	rrca
	rrca
	ld d,a
	and $e0
	or e
	ld e,a
	ld a,d
	and 3
	or $58
	ld d,a
	ret

; ---------------------------------
; 'drchrc' Draw Character With Color
; ---------------------------------
;	Draws a character directly on the screen, and its color.
;
; In	HL address to 8 bytes of char data. DE y,x in chars. A color.
; Saves	A,C

drchrc	push af
	call scradr

; Draw character.

	push de
	ld b,8
drchrc1	ld a,(hl)
	ld (de),a
	inc hl
	inc d
	djnz drchrc1
	pop hl

; Draw attribute.

	ld a,h
	rrca
	rrca
	rrca
	and 7
	or $58
	ld h,a
	pop af
	ld (hl),a
	ret

; -----------------------
; 'romchar' Rom character
; -----------------------
;	Returns the address of character data in ROM or custom font.
;
; In	A character code.
; Out	HL address of character data.
; Saves	BC, DE

romchar 

; Check if it is from A-Z, use our font in this case.

	cp $41
	jr c,romchar1
	cp $5B
	jr nc,romchar1
	sub $41
	ld hl,font_A
	jr romchar2

romchar1

; For the rest of characters use ROM font.
; $3d00 is address of char 32 (space) in ROM.

	sub 32
	ld hl,$3d00

romchar2

	push bc
	ld b,h
	ld c,l
	ld l,a
	ld h,0
	add hl,hl
	add hl,hl
	add hl,hl
	add hl,bc
	pop bc
	ret

; -------------------
; 'drstr' Draw String
; -------------------
;	Draws a string on VRAM.
;	A string is a series of characters from 32 to 127.
;	If 0 is encountered, it signals the end of the string.
;	If 1 is encountered, the next byte is a color to paint the rest
;	of the string.
;
; In	HL string address. DE y,x position in characters. A color.

drstr

; Set initial color.

	ld c,a

; Get character or command. Return if 0.

drstr2	ld a,(hl)
	inc hl
	or a
	ret z
	cp 1
	jr nz,drstr1

; Set new color.

	ld c,(hl)
	inc hl
	jr drstr2

drstr1

; Find address in rom.

	push hl
	call romchar
	push de

; Set color and draw.

	ld a,c
	call drchrc

	pop de
	inc e
	pop hl
	jr drstr2

; ----------------------
; 'strlen' String length
; ----------------------
;	Calculates the length of a string.
;
; In	HL str address.
; Out	B len.
; Saves	DE, C.

strlen	ld b,0

strlen_next

	ld a,(hl)
	inc hl

; If 0 end of string.

	or a
	ret z

; If 1 the next byte is a color.

	cp 1
	jr z,strlen_color

; Increment length.

	inc b
	jr strlen_next

strlen_color
	inc hl
	jr strlen_next
	
; -----------------------------
; 'getspr' Get Free Sprite Slot
; -----------------------------
;	Finds a free sprite slot between [D,E[.
;
; In	[D,E[ slots where to find.
; Out	HL address of slot. CY 0 if found, 1 if no free slot.

getspr	ld hl,sprtab
	ld c,SPRSZ
	ld b,NSPRS
	call gtslot
	ret

#if 0
getspr	ld hl,sprtab
	ld bc,SPRSZ

; A will count until we reach E or NSPRS.

	xor a
getspr2	cp NSPRS
	jr z,getspr3
	cp e
	jr z,getspr3

; We only consider the slot if it is equal or greater than D.

	cp d
	jr c,getspr1

; Check if we have a free slot.

	push af
	ld a,(hl)
	inc hl
	or (hl)
	dec hl
	jr z,getspr4
	pop af

; Go to next slot.

getspr1	add hl,bc
	inc a
	jr getspr2

; Find none. Exit with CY 1.

getspr3	scf
	ret

; Found. Exit with CY 0.

getspr4	pop af
	or a
	ret
#endif

; ----------------------------
; 'getwt' Get Word in Table
; ----------------------------
;	Implements LD HL,(HL+A*2)
;
; In	A index in table. HL table.
; Out	HL word at HL+A*2.
; Saves	AF, DE, BC.

getwt	push af
	push bc
	ld c,a
	ld b,0
	add hl,bc
	add hl,bc
	ld a,(hl)
	inc hl
	ld h,(hl)
	ld l,a
	pop bc
	pop af
	ret

; ----------------------------
; 'getbt' Get Byte in Table
; ----------------------------
;	Implements LD A,(HL+A)
;
; In	A index in table. HL table.
; Out	A byte.
; Saves	HL, DE, BC.

getbt	push hl
	add a,l
	ld l,a
	ld a,h
	adc a,0
	ld h,a
	ld a,(hl)
	pop hl
	ret

; --------------------
; 'freesp' Free Sprite
; --------------------
;	Frees sprite pointed by IY.
;
; In	IY Sprite Pointer.
; Saves	AF, BC, DE, HL.

freesp	ld (iy+SP_ITL),0
	ld (iy+SP_ITH),0
	ld (iy+SP_ANL),0
	ld (iy+SP_ANH),0
	ret

; --------------------
; 'freeob' Free Object
; --------------------
;	Frees object pointed by IX.
;
; In	IX Object Pointer.
; Saves	AF, BC, DE, HL.

freeob	ld (ix+OB_FUL),0
	ld (ix+OB_FUH),0
	ret

; ------------------
; 'getob' Get Object
; ------------------
;	Finds a free object slot in the Object Table.
;
; In	[D,E[ slots where to find.
; Out	HL address of slot. CY 0 if found, 1 if no free slot.

getob	ld hl,objtab
	ld c,OBJSZ
	ld b,NOBJS
	call gtslot
	ret

; -----------------
; 'gtslot' Get Slot
; -----------------
;	Finds a structure in an array of structures. If the first word is 0
;	this means that the structure is free.
;
; In	[D,E[ slots where to find. C size of structure.
;	B table size. HL table address.
; Out	HL address of slot. CY 0 if found, 1 if no free slot.

gtslot	

; A will count until we reach E or NOBJS.

	xor a
gtslot2	cp b
	jr z,gtslot3
	cp e
	jr z,gtslot3

; We only consider the slot if it is equal or greater than D.

	cp d
	jr c,gtslot1

; Check if we have a free slot.

	push af
	ld a,(hl)
	inc hl
	or (hl)
	dec hl
	jr z,gtslot4
	pop af

; Go to next slot.

gtslot1	push bc
	ld b,0
	add hl,bc
	pop bc
	inc a
	jr gtslot2

; Find none. Exit with CY 1.

gtslot3	scf
	ret

; Found. Exit with CY 0.

gtslot4	pop af
	or a
	ret

; --------------------------------
; 'exobs' Execute Object Behaviors
; --------------------------------
;	Runs through the Object Table and executes the behavior for each if
;	if any.

exobs	ld b,NOBJS
	ld hl,objtab
exobs1	push bc
	call exob
	pop bc
	djnz exobs1
	ret

; ------------------------------
; 'exob' Execute Object Behavior
; ------------------------------
;	Executes the behavior of an object, if any. Goes to the next object.
;
; In	HL Object address.
; Out	HL Adrress of next Object.

exob

; Set IX to Object pointer.

	push hl
	push hl
	pop ix

; Get behavior function.

	ld l,(ix+OB_FUL)
	ld h,(ix+OB_FUH)

; If behavior function address is 0, go to next object.
; Else execute behavior.

	ld a,h
	or l
	jr z,exob1
	call jphl

; Point HL to next Object.

exob1	pop hl
	ld bc,OBJSZ
	add hl,bc
	ret

; -----------------
; 'jphl' Juml to HL
; -----------------
;	This can be used with 'call jphl' to call the routine contained in HL.

jphl	jp (hl)

; -------------
; 'modb' Modulo
; -------------
;	Calculates A mod E. If E is 0, returns A, that is, acts as A mod 256.
;
; In	A, E.
; Out	A = A mod E
; Saves HL, BC, DE

; Works by subtracting E, E * 2, E * 4, ... and again E, E * 2, E * 4 until
; no more can be subtracted.

modb

; Check if E is 0.

	rlc e
	jr nz,modb3

; It's 0.

	rrc e
	ret

modb3

; Not 0. Restore and continue.

	rrc e
modb4	push de
modb1	cp e
	jr c,modb2
	sub e
	sla e
	jr nz,modb1
modb2	pop de
	cp e
	jr nc,modb4
	ret

; ------------------------
; 'rand' Get Random Number
; ------------------------
;	Returns a random byte.
;
; Out	A random number.
; Saves	HL, DE, BC.

rand	

;	ld a,r
;	ret

; Run through the ROM using Random Pointer and take next value.
; From address $0500 seems more random.

	push hl
	ld a,(randp)
	ld l,a
	ld h,$05
	inc a
	ld (randp),a
	ld a,(hl)
	pop hl
	ret

; This one is taken from the Internet. x(i+1)=(5*x(i)+1) mod 256
; To be tested later.
;	ld a,seed
;	ld b,a
;	add a,a
;	add a,a
;	add a,b
;	inc a

; -----------------------
; 'randr' Random In Range
; -----------------------
;	Calcs a random number in range [D,E].
;
; In	[D,E] range.
; Out	A random number.
; Saves	HL, BC

randr	inc e
	ld a,e
	sub d
	ld e,a
	call rand
	call modb
	add a,d
	ret

; -------------------
; 'memset' Memory Set
; -------------------
;	Like the C function. Sets memory to a value.
;
; In	HL address of first byte. BC how many to set. A value to set to.

memset

; Save value to set in e.

	ld e,a

; If 0 return.

	ld a,b
	or c
	ret z

; Set first value.

	ld (hl),e

; We are going to set BC - 1, as the first is already set.

	dec bc

; If we are done return.

	ld a,b
	or c
	ret z

; Copy from the previous to the next byte.

	ld d,h
	ld e,l
	inc de
	ldir
	ret

; -------------------
; 'addnum' Add number
; -------------------
;	Decimal addition. The numbers are stored each digit in a byte.
; The number in address pointed by HL is added to the one pointed by DE
; and stores in this same sequence pointed by DE.
;
; If HL points to these bytes 			0019
; And DE to these             			0001
; Then the bytes pointed by DE will contain	0020
;
; In	HL array of digits. DE array of digits. B number of digits.
; Saves	None.

addnum

; Go to the end of the numbers.

	ld c,b
	ld b,0
	dec c
	add hl,bc
	ex de,hl
	add hl,bc
	ex de,hl
	ld b,c
	inc b

; Reset CY.

	or a

addnum1 ld a,(de)
	adc a,(hl)
	cp 10
	jr c,addnum0

; The sum is equal or more than 10.

	sub 10
	ld (de),a
	scf
	jr addnum2

; The sum is less than 10.

addnum0	ld (de),a

; Reset CY.

	or a

addnum2	dec hl
	dec de
	djnz addnum1
	ret

; -------------------
; 'drnum' Draw number
; -------------------
;
; In	B number of digits. HL pointer to digits. DE y,x in chars. C color.
; Saves	C

drnum

; In the begining, while it is a zero digit, draw a space.

	dec b

drnum_zero

	ld a,(hl)
	or a
	jr nz,drnum_rest
	ld a,' '
	call drchrsf
	inc e
	inc hl
	djnz drnum_zero

drnum_rest

; Now, some digits remain.

	inc b

drnum_num

	ld a,(hl)
	add a,'0'

; Find in rom and draw with color.

	call drchrsf

drnum_next

; Go to next digit.

	inc e
	inc hl
	djnz drnum_num
	ret

; -------------------------
; 'drchrsf' Draw char safe.
; -------------------------
;	Draws a character, preserves most of registers.
;
; In	A character code. DE y,x in characters. C color.
; Saves	BC, DE, HL.

drchrsf push hl
	call romchar
	push de
	push bc
	ld a,c
	call drchrc
	pop bc
	pop de
	pop hl
	ret

; --------
; 'minnum'
; --------
;	Selects the minimum of two decimal numbers.
;
; In	HL addr number 1. DE addr number 2. B digits.
; Out	A 0 if equal, -1 if HL is the minimum, 1 if DE is the minimum.
; Saves	HL, DE.

minnum
	push de
	push hl

minnum_loop

	ld a,(de)
	cp (hl)
	jr nz,minnum_cp

; Digits are equal, go to next.

	inc hl
	inc de
	djnz minnum_loop
	xor a
	jr minnum_end

minnum_cp

; The digits are different.

	jr c,minnum_is_2
	ld a,-1
	jr minnum_end

minnum_is_2

; The second number is less than the first.

	ld a,1

minnum_end
	pop hl
	pop de
	ret

; --------------------
; 'digit' Gets a digit
; --------------------
;	Gets a digit of a number, starting from the most significant.
;
; In	HL number address. B digit to obtain (number length - digits from
;	the least significant).
; Out	A digit. B 0. HL points at digit.

digit	dec hl
	inc b

digit_loop
	inc hl
	ld a,(hl)
	djnz digit_loop
	ret

; ---------------------
; 'clrscr' Clear Screen
; ---------------------
;	Clear screen to a color.
;
; In	A color.

clrscr	push af
	ld hl,vram
	ld bc,VRAMSZ
	xor a
	call memset
	ld hl,aram
	ld bc,ARAMSZ
	pop af
	call memset
	ret

; ---------------------
; 'clrwin' Clear window
; ---------------------
;	Clears some rect of the screen to a color. (Needs optimization).
;
; In	A color. D,E y,x char coords. B,C height,width in chars.

clrwin	

; Get in HL the address of SPACE character.

	push af
	ld a,32
	call romchar
	pop af

clrwiny

; For all rows.

	push bc
	push de

clrwinx

; For all columns.

	push de
	push hl
	call drchrc
	pop hl
	pop de
	inc e
	dec c
	jr nz,clrwinx

; Next row.

	pop de
	pop bc
	inc d
	djnz clrwiny
	ret

; ---------
; 'clrbbuf'
; ---------
;	Clears back buffer and back buffer color and resets nrects.

clrbbuf

; Clear backbuffer.

	ld hl,bbuf
	ld bc,BBUFSZ
	xor a
	call memset

; Set Back Buffer color.

	ld hl,abuf
	ld bc,ABUFSZ
	ld a,ABUFCLR
	call memset

; Reset rects.

	xor a
	ld (nrects),a
	ret

; ------
; 'loop'
; ------
;	Loops.
;
; In	BC cicles.
; Out	BC 0.
; Saves DE, HL.

loop	dec bc
	ld a,c
	or b
	jr nz,loop
	ret

; Paper to colorize back buffer for effects.
bbuf_paper	.db ABUFCLR
bbuf_last_paper	.db ABUFCLR

; Star Mode Counter.
; Grows until some point, where Star Mode is switched. 
starmc	.db 0

; Star Mode.
; Takes values 0 or 1 to set two different starfield patterns.
starm	.db 0

; N rects filled and current rect pointer.
nrects	.db 0
rectp	.dw reclst

; 0-255. Incremented each time we call rand.
;
randp	.db 0

; Frame counter.
frames	.db 0

#include "states.asm"
#include "gameplay_st.asm"
#include "drmach.asm"
#include "menu_st.asm"
#include "attract_st.asm"
#include "options_st.asm"
#include "killed_st.asm"
#include "round_st.asm"
#include "name_st.asm"
#include "disclaimer_st.asm"
#include "dedicate_st.asm"
#include "gameover_st.asm"
#include "hud.asm"
#include "anim.asm"
#include "obfun.asm"
#include "timer.asm"
#include "keyb.asm"
#include "prepare.asm"
#include "interr.asm"
#include "images.asm"
#include "font.asm"

endp	.equ $

#include "ram.asm"

	.echo "Number of sprites "
	.echo NSPRS
	.echo "\nNumber of rects "
	.echo MAXRECS
	.echo "\nTotal program memory: "
	.echo (eofram - start)
	.echo " bytes.\n"

	.end
