/*
   Reverie
   starfield.cpp
   Copyright (C)2003 Dan Potter
*/

/* Pulled and greatly adapted from Tryptonite */

/*

Ideas for doing real 3D stars:

Give each star an actual 3D position [-1..1], and the starfield two vectors:

1) View direction
2) Movement direction

Stars are moved along the movement direction linearly. When they reach the
edge of the bounding box, they are repositioned at the opposite end of the
box. Stars are then rotated by the view angles and projected on the screen
using the standard matrix stuff.

To achieve an intertia effect, the view direction should either be slightly
ahead or slightly behind the movement direction (behind I think). Some
experimentation will be needed to determine the exact desired effect.

*/


#include "global.h"
#include "starfield.h"

#define BBXF 1.0f

StarField::StarField(int ns) {
	int i;

	m_sns = m_ns = ns;
	m_stars = new Vector[m_ns];
	m_starsMoveDir = new Vector[m_ns];
	m_starsTran = new StarVector[m_ns];
	for (i=0; i<m_ns; i++) {
		m_stars[i].x = BBXF*1.5f*(randnum(1024) / 512.0f - 1.0f);
		m_stars[i].y = BBXF*1.5f*(randnum(1024) / 512.0f - 1.0f);
		m_stars[i].z = BBXF*(randnum(1024) / 512.0f - 1.0f);

		m_starsMoveDir[i].w = 5.0f;
	}

	m_moveDir = m_moveDirGoal = Vector(0.0f, 0.0f, BBXF*1.0f/120.0f);
	m_viewDir = Vector(0.0f, 0.0f, BBXF*-1.0f/120.0f);

	m_snowMode = false;
}

StarField::~StarField() {
	delete[] m_stars;
	delete[] m_starsTran;
}

void StarField::setSnowMode() {
	m_snowMode = true;
	m_moveDirGoal = m_moveDir = Vector(0.0f, -0.005f, 0.0f);

	// Prep the starfield (takes a few frames to make it look right)
	for (int i=0; i<120; i++)
		nextFrame();
}

void StarField::set3dMode() {
	m_snowMode = false;
	m_moveDirGoal = Vector(0.0f, 0.0f, BBXF*1.0f/120.0f);
	for (int i=0; i<m_ns; i++)
		m_starsMoveDir[i] = Vector(0,0,0,5.0f);
}

void StarField::setActive(int sns) {
	m_sns = sns;
}

// Hehe... "wrapStar"...
//
// Anyway, what we need to do here is look at the star motion vector and
// move the star back into the view cube so that if the motion continues in
// the same direction, things will continue to more or less be filled up.
// The way we do this is to pick a random X/Y value and place the star in
// the plane at the rear of the motion vector.
void StarField::wrapStar(const Vector & moveNorm, Vector & toFix) {
	// Pick random X/Y values
	//float nx = 1.5f*(randnum(1024) / 512.0f - 1.0f);
	//float ny = 1.5f*(randnum(1024) / 512.0f - 1.0f);

	// Move back along the movement vector by 1 unit.

	/* while (toFix.z > 1.0f)
		toFix.z -= 2.0f; */

	// Laziness...!
	toFix.x = BBXF*1.5f*(randnum(1024) / 512.0f - 1.0f);
	toFix.y = BBXF*1.5f*(randnum(1024) / 512.0f - 1.0f);
	toFix.z = BBXF*(randnum(1024) / 512.0f - 1.0f);
}

void StarField::setCourse(const Vector & dir) {
	m_moveDir = Vector(BBXF*dir.x/240.0f, BBXF*dir.y/240.0f,
		BBXF*1.0f/120.0f);
}

void StarField::draw(int list) {
	if (list != PLX_LIST_OP_POLY)
		return;

	int i;
	float xn, yn, zn, wn, a, r;

	/* Color col = getColor();
	Vector pos = getPosition(); */

	/* Send polygon header to the TA using store queues */
	plx_cxt_texture(NULL);
	plx_cxt_send(list);

	plx_dr_state_t dr;
	plx_dr_init(&dr);

	// Setup rotation matrices
	point_t eye = {0,0,0};
	point_t lookat = {m_viewDir.x, m_viewDir.y, m_viewDir.z};
	point_t up = {0,1,0};
	plx_mat3d_identity();
	plx_mat3d_lookat(&eye, &lookat, &up);
	
	plx_mat_identity();
	plx_mat3d_apply(PLX_MAT_MODELVIEW);

	Vector pos = getPosition();

	// Transform all of the stars to absolute world coordinates
	for (i=0; i<m_ns; i++) {
		// Skip inactive stars
		if (m_stars[i].w == -1.0f) {
			m_starsTran[i].active = false;
			continue;
		}

		// Get the star values
		xn = m_stars[i].x;
		yn = m_stars[i].y;
		zn = m_stars[i].z;
		wn = 1.0f;

		// Do some rotations
		plx_mat_tfip_3dw(xn, yn, zn, wn);

		if (zn >= 0.0f) {
			m_starsTran[i].active = false;
		} else {
			m_starsTran[i] = Vector(xn, yn, zn * 5.0f);
			m_starsTran[i].b = (((zn/BBXF) + 1.0f) / 1.3f);
			if (m_starsTran[i].b < 0.0f)
				m_starsTran[i].b = 0.0f;
			if (m_starsTran[i].b > 1.0f)
				m_starsTran[i].b = 1.0f;
		}
	}

	// Now that the stars are in absolute world coords with shading,
	// transform them to screen coords and render.
	plx_mat_identity();
	plx_mat3d_apply(PLX_MAT_SCREENVIEW);
	plx_mat3d_apply(PLX_MAT_PROJECTION);

	r = getColor().r;

	for (i=0; i<m_ns; i++) {
		if (!m_starsTran[i].active)
			continue;

		// Get the star values
		xn = m_starsTran[i].x;
		yn = m_starsTran[i].y;
		zn = m_starsTran[i].z;
		a = m_starsTran[i].b * r;

		// Do some rotations
		plx_mat_tfip_3d(xn, yn, zn);

		if (zn <= 0.0f)
			continue;

		plx_vert_fnd(&dr, PLX_VERT, xn, yn+2, pos.z, 1.0f, a, a, a);
		plx_vert_fnd(&dr, PLX_VERT, xn, yn, pos.z, 1.0f, a, a, a);
		plx_vert_fnd(&dr, PLX_VERT, xn+2, yn+2, pos.z, 1.0f, a, a, a);
		plx_vert_fnd(&dr, PLX_VERT_EOS, xn+2, yn, pos.z, 1.0f, a, a, a);
	}
}

void StarField::nextFrame() {
	for (int i=0; i<m_ns; i++) {
		// Move this star opposite the move direction vector. If
		// it goes outside the bounding box, place it on the opposite
		// side for wraparound.
		m_stars[i] += m_moveDir;
		m_stars[i] += m_starsMoveDir[i];
		if (m_stars[i].x < -BBXF || m_stars[i].x > BBXF ||
			m_stars[i].y < -BBXF || m_stars[i].y > BBXF ||
			m_stars[i].z < -BBXF || m_stars[i].z > BBXF)
		{
			if (i >= m_sns)
				m_stars[i].w = -1.0f;
			else
				wrapStar(Vector(), m_stars[i]);
		}
	}

	if (!m_snowMode) {
		if (m_moveDir == m_moveDirGoal) {
			m_viewDir.x = m_viewDir.x + (-m_moveDir.x - m_viewDir.x)/8.0f;
			m_viewDir.y = m_viewDir.y + (-m_moveDir.y - m_viewDir.y)/8.0f;
		//	m_viewDir.z = m_viewDir.z + (m_moveDir.z - m_viewDir.z)/4.0f;
		} else {
			for (int i=0; i<m_ns; i++) {
				float a = m_starsMoveDir[i].x;
				a = a < 0 ? -a : a;
				if (a < 0.001f)
					a = 0.0f;
				else
					a = m_starsMoveDir[i].x * 0.5f;
				m_starsMoveDir[i].x = a;
			}
		}
	} else {
		for (int i=0; i<m_ns; i++) {
			m_starsMoveDir[i].w -= 1.0f;
			if (m_starsMoveDir[i].w <= 0.0f) {
				m_starsMoveDir[i].x += ((randnum(100) - 50) / 50.0f) * 0.001f;
				if (m_starsMoveDir[i].x <= -0.0025f)
					m_starsMoveDir[i].x = -0.0025f;
				if (m_starsMoveDir[i].x >= 0.0025f)
					m_starsMoveDir[i].x = 0.0025f;
				m_starsMoveDir[i].w = 5.0f;
			}
		}
	}

	if (m_moveDirGoal != m_moveDir) {
		Vector dv;

		dv = m_moveDirGoal - m_moveDir;
		if (dv.length() < 0.001f)
			m_moveDir = m_moveDirGoal;
		else
			m_moveDir += dv * 0.025f;
	}

	Drawable::nextFrame();
}
