/*
   Reverie
   phongspheres.cpp
   Copyright (C)2003 Dan Potter
*/

#include "global.h"
#include "phongspheres.h"

/*

Good texture map, some environment mapping, quick flick of the wrist,
and boom, instant phong shading! :D

This is a little complex but it more or less works like a pipeline to
simulate ~2-order raytracing:

1) Transform the center point of each sphere.
2) Draw each sphere with the base phong texture, rotated to point at
   the light source.
3) Transform the light reflection point of each sphere (done the same
   time as #2).
4) For each sphere, draw 3 partial spheres with the smaller reflection
   texture, same process as step #2 (except the points produced in step
   #3 are now the light sources).

The effect is still not all that fantastic when you have time to really
study the output, but it'll only be on the screen for a few secs ;)

*/

PhongSpheres::PhongSpheres(Texture * txr, Texture * txrh, Texture * txrb) {
	m_txr = txr;
	m_txr->setFilter(Texture::FilterBilinear);
	m_txr->setUVClamp(Texture::UVClamp, Texture::UVClamp);

	m_txrh = txrh;
	m_txrh->setFilter(Texture::FilterBilinear);
	m_txrh->setUVClamp(Texture::UVClamp, Texture::UVClamp);

	m_txrb = txrb;
	m_txrb->setFilter(Texture::FilterBilinear);
	m_txrb->setUVClamp(Texture::UVClamp, Texture::UVClamp);
}

PhongSpheres::~PhongSpheres() {
}

void PhongSpheres::drawSphere(float a, float za, bool partial) {
	float	x, y, z;
	pvr_dr_state_t	dr_state;

	Color c = getColor();
	float i = c.r;

	const float radius = 4.0f;
	const int slices = 21;
	const int stacks = 40;

	// For the secondary highlights, we only need a small chunk
	// of the full sphere. This saves a lot of rendering time.
	int slStart, slEnd;
	int stStart, stEnd;
	if (partial) {
		slStart = slices/2 - slices/4;
		slEnd = slices/2 + slices/4;
		stStart = stacks/2 - stacks/4;
		stEnd = stacks/2 + stacks/4;
	} else {
		slStart = 0;
		slEnd = slices;
		stStart = 0;
		stEnd = stacks-1;
	}

	/* Setup our Direct Render state: pick a store queue and setup QACR0/1 */
	pvr_dr_init(dr_state);

	/* Initialize xmtrx with the values from KGL */
	plx_mat_identity();
	plx_mat3d_apply(PLX_MAT_SCREENVIEW);
	plx_mat3d_apply(PLX_MAT_PROJECTION);
	plx_mat3d_apply(PLX_MAT_MODELVIEW);

	/* Iterate over stacks */
	for (int iy=stStart; iy<=stEnd; iy++) {
		float wx1 = fsin(iy * M_PI / stacks) * radius;
		float wy1 = fcos(iy * M_PI / stacks) * radius;

		float wx2 = fsin((iy+1) * M_PI / stacks) * radius;
		float wy2 = fcos((iy+1) * M_PI / stacks) * radius;

		float v1 = 1.0f * iy / stacks;
		float v2 = 1.0f * (iy+1) / stacks;

		/* Iterate over slices: each entire stack will be one
		   long triangle strip. */
		for (int ix=slStart; ix<=slEnd; ix++ ) {
			float u = 2.0f * ix / slices;
			float sx = fcos(ix * 2*M_PI / slices);
			float sz = fsin(ix * 2*M_PI / slices);

			/* x, y+1 */
			x = sx * wx2;
			y = wy2;
			z = sz * wx2;
			mat_trans_single(x, y, z);			/* Use ftrv to transform */
			z += za;
			plx_vert_ffd(&dr_state, PVR_CMD_VERTEX,
				x, y, z,
				a,i,i,i,
				u, v2);
                
			/* x, y */
			x = sx * wx1;
			y = wy1;
			z = sz * wx1;
			mat_trans_single(x, y, z);
			z += za;

			if (ix == slEnd) {
				plx_vert_ffd(&dr_state, PVR_CMD_VERTEX_EOL,
					x, y, z,
					a,i,i,i,
					u, v1);
			} else {
				plx_vert_ffd(&dr_state, PVR_CMD_VERTEX,
					x, y, z,
					a,i,i,i,
					u, v1);
			}
		}
	}
}

void PhongSpheres::drawLight(const Vector & pos) {
	plx_mat3d_identity();
	plx_mat3d_translate(pos.x, pos.y, pos.z);
	plx_mat3d_rotate(m_theta, 1.0f, 0.0f, 0.0f);
	plx_mat3d_rotate(m_theta, 0.0f, 1.0f, 0.0f);
	plx_mat3d_rotate(m_theta, 0.0f, 0.0f, 1.0f);
	plx_mat3d_scale(0.5f, 0.5f, 0.5f);

	// Save the light position in world coordinates, for later.
	plx_mat_identity();
	plx_mat3d_apply(PLX_MAT_MODELVIEW);
	m_lightTrans = Vector(0,0,0,0);
	plx_mat_tfip_2d(m_lightTrans.x, m_lightTrans.y, m_lightTrans.z);

	// First, the cube itself.
	plx_cxt_texture(NULL);
	plx_cxt_send(PLX_LIST_TR_POLY);

	plx_mat_identity();
	plx_mat3d_apply(PLX_MAT_SCREENVIEW);
	plx_mat3d_apply(PLX_MAT_PROJECTION);
	plx_mat3d_apply(PLX_MAT_MODELVIEW);

	uint32 color = getColor();
	plx_dr_state_t dr;

	plx_dr_init(&dr);

	// Left face
	plx_vert_indm3(&dr, PLX_VERT, -1, -1, -1, color);
	plx_vert_indm3(&dr, PLX_VERT, -1,  1, -1, color);
	plx_vert_indm3(&dr, PLX_VERT, -1, -1,  1, color);
	plx_vert_indm3(&dr, PLX_VERT_EOS, -1,  1,  1, color);

	// Right face
	plx_vert_indm3(&dr, PLX_VERT, 1, -1, -1, color);
	plx_vert_indm3(&dr, PLX_VERT, 1,  1, -1, color);
	plx_vert_indm3(&dr, PLX_VERT, 1, -1,  1, color);
	plx_vert_indm3(&dr, PLX_VERT_EOS, 1,  1,  1, color);

	// Front face
	plx_vert_indm3(&dr, PLX_VERT, -1,  1, -1, color);
	plx_vert_indm3(&dr, PLX_VERT, -1, -1, -1, color);
	plx_vert_indm3(&dr, PLX_VERT,  1,  1, -1, color);
	plx_vert_indm3(&dr, PLX_VERT_EOS,  1, -1, -1, color);

	// Back face
	plx_vert_indm3(&dr, PLX_VERT, -1,  1, 1, color);
	plx_vert_indm3(&dr, PLX_VERT, -1, -1, 1, color);
	plx_vert_indm3(&dr, PLX_VERT,  1,  1, 1, color);
	plx_vert_indm3(&dr, PLX_VERT_EOS,  1, -1, 1, color);

	// Top face
	plx_vert_indm3(&dr, PLX_VERT, -1, -1, -1, color);
	plx_vert_indm3(&dr, PLX_VERT, -1, -1,  1, color);
	plx_vert_indm3(&dr, PLX_VERT,  1, -1, -1, color);
	plx_vert_indm3(&dr, PLX_VERT_EOS,  1, -1,  1, color);

	// Bottom face
	plx_vert_indm3(&dr, PLX_VERT, -1, 1, -1, color);
	plx_vert_indm3(&dr, PLX_VERT, -1, 1,  1, color);
	plx_vert_indm3(&dr, PLX_VERT,  1, 1, -1, color);
	plx_vert_indm3(&dr, PLX_VERT_EOS,  1, 1,  1, color);

	// Now go back and put the bloom on the cube
	float x = 0, y = 0, z = 0;
	plx_mat_tfip_3d(x, y, z);

	plx_cxt_texture(*m_txrb);
	plx_cxt_blending(PLX_BLEND_SRCALPHA, PLX_BLEND_ONE);
	plx_cxt_culling(PLX_CULL_NONE);
	plx_cxt_send(PLX_LIST_TR_POLY);

	plx_spr_inp(64.0f, 64.0f, x, y, z + 0.001f, color);
	plx_cxt_blending(PLX_BLEND_SRCALPHA, PLX_BLEND_INVSRCALPHA);
}

void PhongSpheres::draw(int list) {
	// Make and submit the context.
	plx_cxt_culling(PLX_CULL_NONE);

	// Basic scene setup
	plx_mat3d_identity();
	plx_mat3d_translate(0.0f, 0.0f, -40.0f);

	if (list == PLX_LIST_OP_POLY) {
		// For each sphere, we'll rotate differently. Figure out
		// where the spheres are so we can point them at the light.
		plx_mat3d_push();
		for (int i=0; i<4; i++) {
			// Do the transforms.
			if (i != 0)
				plx_mat3d_peek();
			//plx_mat3d_rotate(m_theta*0.3f, 1.0f, 0.0f, 0.0f);
			plx_mat3d_rotate(m_theta*1.5f + i*90, 0.0f, 1.0f, 0.0f);
			if (!(i%2))
				plx_mat3d_translate(12.0f + fcos(m_theta*2*M_PI/100.0f) * 2.0f,
					fsin(m_theta*2*M_PI/140.0f) * 4.0f, 0.0f);
			else
				plx_mat3d_translate(12.0f + fcos(m_theta*2*M_PI/100.0f) * 2.0f,
					fsin(m_theta*2*M_PI/140.0f + M_PI) * 4.0f, 0.0f);

			// Get the sphere's final position in world coords.
			m_sphereTrans[i] = Vector(0,0,0,0);
			plx_mat_tfip_2d(m_sphereTrans[i].x, m_sphereTrans[i].y, m_sphereTrans[i].z);
		}
		plx_mat3d_pop();

		// Select the main sphere texture.
		plx_cxt_texture(*m_txr);
		plx_cxt_send(list);

		// Draw the basic spheres.
		for (int i=0; i<4; i++) {
			// Move it into its 3d position.
			plx_mat3d_identity();
			plx_mat3d_translate(m_sphereTrans[i].x, m_sphereTrans[i].y, m_sphereTrans[i].z);

			// Now we also need to have it looking at the light source.
			Vector d = m_lightTrans - m_sphereTrans[i];
			float yaw = -(float)atan2(d.z, d.x) * 360.0f / (2*M_PI) + 90;
			float pitch = -(float)atan2(d.y, d.length()) * 360.0f / (2*M_PI);

			plx_mat3d_rotate(yaw, 0.0f, 1.0f, 0.0f);
			plx_mat3d_rotate(pitch, 1.0f, 0.0f, 0.0f);

			// Draw the base sphere.
			drawSphere(1.0f, 0.000f, false);

			// Also translate its light reflection point and store that for later.
			plx_mat_identity();
			plx_mat3d_apply(PLX_MAT_MODELVIEW);
			m_sphereLights[i] = Vector(0,0,2);
			plx_mat_tfip_2d(m_sphereLights[i].x, m_sphereLights[i].y, m_sphereLights[i].z);
		}
	} else if (list == PLX_LIST_TR_POLY) {
		// Setup our context for blending
		plx_cxt_blending(PLX_BLEND_SRCALPHA, PLX_BLEND_ONE);
		plx_cxt_texture(*m_txrh);
		plx_cxt_send(list);

		// For each sphere...
		for (int i=0; i<4; i++) {
			// Move it into its 3d position.
			plx_mat3d_identity();
			plx_mat3d_translate(m_sphereTrans[i].x, m_sphereTrans[i].y, m_sphereTrans[i].z);
			plx_mat3d_push();

			// And cross with each other sphere...
			for (int j=0; j<4; j++) {
				// No need to light ourselves.
				if (i == j)
					continue;

				// Get the light vector from the reflection light we're referencing.
				Vector d = m_sphereLights[j] - m_sphereTrans[i];

				float yaw = -(float)atan2(d.z, d.x) * 360.0f / (2*M_PI) + 90;
				float pitch = -(float)atan2(d.y, d.length()) * 360.0f / (2*M_PI);

				if (j != 0)
					plx_mat3d_peek();
				plx_mat3d_rotate(yaw, 0.0f, 1.0f, 0.0f);
				plx_mat3d_rotate(pitch, 1.0f, 0.0f, 0.0f);

				// Draw the sphere.
				float a = 0.6f - 0.4f * (d.length() - 12.0f) / (20.0f - 12.0f);
				if (a < 0.2f)
					a = 0.1f;
				if (a > 0.6f)
					a = 0.6f;
				drawSphere(a, 0.0005f * (j+1), true);
			}

			plx_mat3d_pop();
		}
		plx_cxt_blending(PLX_BLEND_SRCALPHA, PLX_BLEND_INVSRCALPHA);

		// And draw a visible light source
		drawLight(m_light - Vector(0,0,40));
	}
}

void PhongSpheres::nextFrame() {
	m_theta++;

	// Our light will trace out an hourglass shape.
	float y = fsin(m_theta * 2*M_PI / 360.0f) * 8.0f;
	m_light = Vector(
		fcos(m_theta * 2*M_PI / 90.0f) * y,
		y,
		fsin(m_theta * 2*M_PI / 90.0f) * y);

	Drawable::nextFrame();
}
