/*
   Reverie
   demo.cpp
   Copyright (C)2003 Dan Potter
*/

/* This pulls everything together into one happy Tsunami scene graph and
   lets it go. */

#include "global.h"
#include "starfield.h"
#include "delaycube.h"
#include "melter.h"
#include "delay.h"
#include "amigasphere.h"
#include "perfmeter.h"
#include "phongspheres.h"
#include "screenblinds.h"
#include "linearmover.h"
#include "screenwave.h"
#include "rotocube.h"
#include "endscroller.h"
#include "copperbars.h"

class Demo : public GenericMenu, public RefCnt, public OneShot::Target {
public:
	Demo();
	virtual ~Demo();

	// OneShot::Target interface
	virtual void shoot(int code);
	virtual RefCnt * getRefCnt() { return this; }

	// We'll attach this to someone else to fade us.
	class Fader : public Animation {
	public:
		Fader(Demo * d) { m_d = d; }
		virtual ~Fader() { }

		virtual void nextFrame(Drawable * d) {
			bool done = true;
			for (int i=0; i<3; i++) {
				m_d->m_bg[i] -= 1.0f / 60.0f;
				if (m_d->m_bg[i] <= 0.0f)
					m_d->m_bg[i] = 0.0f;
				else
					done = false;
			}
			if (done)
				complete(d);
		}
	private:
		Demo * m_d;
	};

protected:
	virtual void startExit();
	virtual void inputEvent(const Event & evt);

private:
	Animation *	part1Setup();
	Animation *	part2Setup(Animation *);
	Animation *	part3Setup(Animation *);
	Animation *	part4Setup(Animation *);
	Animation *	part5Setup(Animation *);
	Animation *	part6Setup(Animation *);

	RefPtr<StarField>	m_sf;
	RefPtr<DelayCube>	m_dc;
};

Demo::Demo() {
	// Now then, we'll use Tsunami triggers to build up basically
	// the whole demo here. This really just screams for a scripting
	// language but this'll have to do for now. :)
	Animation * sd;
	sd = part1Setup();
	sd = part2Setup(sd);
	sd = part3Setup(sd);
	sd = part4Setup(sd);
	sd = part5Setup(sd);
	sd = part6Setup(sd);

	// Throw this in here to watch the performance. (DEBUG)
	/*PerfMeter * pm = new PerfMeter(new Font("sanserif.txf"));
	pm->setTranslate(Vector(30, 480 - 30, 4500));
	m_scene->subAdd(pm); */
}

Demo::~Demo() {
}

// We use this during the part4->part5 transition, to switch
// background colors.
void Demo::shoot(int code) {
	setBg(1,1,1);
}

void Demo::startExit() {
	GenericMenu::startExit();
}

void Demo::inputEvent(const Event & evt) {
	static int i = 0;
	static char fn[256] = "/pc/tmp/shotXX.ppm";

	if (evt.type == Event::EvtKeypress) {
		switch (evt.key) {
		// DEBUG
		/*case Event::KeyStart:
			startExit();
			break; */

		case Event::KeyMiscY: {
			bool pushed = false;

			MAPLE_FOREACH_BEGIN(MAPLE_FUNC_CONTROLLER, cont_state_t, st)
				if (st->ltrig > 0 && st->rtrig > 0)
					pushed = true;
			MAPLE_FOREACH_END()
			
			fn[12] = '0' + (i / 10);
			fn[13] = '0' + (i % 10);
			vid_screen_shot(fn);
			i++;
			break;
		}
		
		}
	}
}


// Part 1 ///////////////////////////////////////////////////
Animation * Demo::part1Setup() {
	// We'll start out with a snowfield, but black.
	m_sf = new StarField(4096);
	m_sf->setSnowMode();
	m_sf->setTint(Color(1,0,0,0));
	m_sf->setTranslate(Vector(0,0,0.0001f));
	m_scene->subAdd(m_sf);

	// Slowly fade in the snowfield.
	TintFader * tf = new TintFader(Color(1,1,1,1), Color(0,1.0f/60.0f,1.0f/60.0f,1.0f/60.0f));
	m_sf->animAdd(tf);

	// When the snowfield is faded in, wait a couple of seconds.
	Delay * delay = new Delay(2*60);
	tf->triggerAdd(new ChainAnimation(delay));

	// After that, create a melter with our first banner texture.
	Texture * txr = new Texture("ca_presents.png", true);
	Melter * mt = new Melter(txr);
	mt->setTranslate(Vector(320,240,100));
	Birth * birth = new Birth(mt, m_scene);
	delay->triggerAdd(birth);

	// And wait a few seconds...
	delay = new Delay(3*60);
	mt->animAdd(delay);

	// And melt that out.
	Melter::MeltOut * mo = new Melter::MeltOut();
	delay->triggerAdd(new ChainAnimation(mo));
	mo->triggerAdd(new Death());


	// Second melter, with our second banner texture.
	Delay * delay2 = new Delay(2*60);
	delay->triggerAdd(new ChainAnimation(delay2, m_sf));
	delay = delay2;

	txr = new Texture("little_demo.png", true);
	mt = new Melter(txr);
	mt->setTranslate(Vector(320,240,100));
	birth = new Birth(mt, m_scene);
	delay->triggerAdd(birth);

	// And wait a few seconds...
	delay = new Delay(4*60);
	mt->animAdd(delay);

	// And melt that out.
	mo = new Melter::MeltOut();
	delay->triggerAdd(new ChainAnimation(mo));
	mo->triggerAdd(new Death());


	// One more banner.
	delay2 = new Delay(2*60);
	delay->triggerAdd(new ChainAnimation(delay2, m_sf));
	delay = delay2;

	txr = new Texture("reverie.png", true);
	mt = new Melter(txr);
	mt->setTranslate(Vector(320,240,100));
	birth = new Birth(mt, m_scene);
	delay->triggerAdd(birth);

	// And wait a few seconds...
	delay = new Delay(5*60 + 45);
	mt->animAdd(delay);

	// And melt that out.
	mo = new Melter::MeltOut();
	delay->triggerAdd(new ChainAnimation(mo));
	mo->triggerAdd(new Death());


	// Make a scene delay also and return that to hook onto.
	delay2 = new Delay(2*60);
	delay->triggerAdd(new ChainAnimation(delay2, m_sf));
	return delay2;
}

// Part 2 ///////////////////////////////////////////////////
Animation * Demo::part2Setup(Animation * delay) {
	// Start out this section by converting the snowfield to a starfield.
	delay->triggerAdd(new ChainAnimation(new StarField::SnowToStars(), m_sf));

	// Make a delay cube, at brightness zero.
	DelayCube * dc = new DelayCube();
	m_dc = dc;
	dc->setTint(Color(1,0,0,0));
	delay->triggerAdd(new Birth(dc, m_scene));

	// And start the starfield zooming around
	StarField::ZoomAround * za = new StarField::ZoomAround(10*90);
	delay->triggerAdd(new ChainAnimation(za, m_sf));

	// Fade the cube in.
	TintFader * tf;
	tf = new TintFader(Color(1,1,1,1), Color(0,1.0f/60.0f,1.0f/60.0f,1.0f/60.0f));
	dc->animAdd(tf);

	// Switch to all the different patterns, ending up on the sphere.
	RefPtr<DelayCube::SwitchObject> so;
	for (int i=1; i<=11; i++) {
		// Wait a second.
		delay = new Delay(88);
		if (i == 1)
			tf->triggerAdd(new ChainAnimation(delay));
		else
			so->triggerAdd(new ChainAnimation(delay));

		// Switch patterns.
		so = new DelayCube::SwitchObject(i % 4);
		delay->triggerAdd(new ChainAnimation(so));
	}

	// Wait a second for the last object shift.
	delay = new Delay(60);
	so->triggerAdd(new ChainAnimation(delay));

	return delay;
}

// Part 3 ///////////////////////////////////////////////////
Animation * Demo::part3Setup(Animation * delay) {
	// We'll want our own sub-scene so we can fade it out at once.
	Scene * scene = new Scene();
	delay->triggerAdd(new Birth(scene, m_scene));

	// Start out this section by fading in an Amiga sphere
	AmigaSphere * as = new AmigaSphere(m_dc, false);
	as->setTint(Color(0,1,1,1));
	delay->triggerAdd(new Birth(as, scene));

	TintFader * tf = new TintFader(Color(1,1,1,1), Color(1.0f/30.0f,0,0,0));
	as->animAdd(tf);

	// And fade out the old delay cube.
	tf = new TintFader(Color(1,0,0,0), Color(0,-1.0f/30.0f,-1.0f/30.0f,-1.0f/30.0f));
	tf->triggerAdd(new Death());
	delay->triggerAdd(new ChainAnimation(tf, m_dc));

	// As well as the starfield.
	tf = new TintFader(Color(1,0,0,0), Color(0,-1.0f/30.0f,-1.0f/30.0f,-1.0f/30.0f));
	tf->triggerAdd(new Death());
	delay->triggerAdd(new ChainAnimation(tf, m_sf));

	// Wait a few seconds.
	delay = new Delay((int)(8*60 - 30));
	as->animAdd(delay);

	// And switch the sphere to bounce mode.
	AmigaSphere::ToBounce * tb = new AmigaSphere::ToBounce();
	delay->triggerAdd(new ChainAnimation(tb, as));

	// Then, one by one, add the other spheres.
	delay = new Delay(2*60);
	Delay * d2;
	tb->triggerAdd(new ChainAnimation(delay, as));

	for (int i=0; i<25; i++) {
		int x = i % 5;
		int y = i / 5;
		as = new AmigaSphere(NULL, true);
		as->setTranslate(Vector(-25.0f + 10.0f*x, 14.0f, -25.0f + 10.0f*y));
		delay->triggerAdd(new Birth(as, scene));

		d2 = new Delay((int)(1.2f*60));
		delay->triggerAdd(new ChainAnimation(d2, as));
		delay = d2;
	}

	// On the tag-end of that last delay, fade it all out.
	tf = new TintFader(Color(1,0,0,0), Color(0,-1.0f/60.0f,-1.0f/60.0f,-1.0f/60.0f));
	delay->triggerAdd(new ChainAnimation(tf, scene));

	// Put a new delay parented from the big scene so we can trash the sub-scene.
	delay = new Delay(1);
	tf->triggerAdd(new ChainAnimation(delay, m_scene));
	tf->triggerAdd(new Death(scene));	// Death of the scene...! ;)

	return delay;
}

// Part 4 ///////////////////////////////////////////////////
Animation * Demo::part4Setup(Animation * delay) {
	// Setup our "phong shaded" spheres
	Texture * txr = new Texture("phong_red.png", false);
	Texture * txrh = new Texture("phong_highlight_2.png", true);
	Texture * txrb = new Texture("phong_bloom.png", true);
	PhongSpheres * ps = new PhongSpheres(txr, txrh, txrb);
	ps->setTint(Color(1,0,0,0));
	delay->triggerAdd(new Birth(ps, m_scene));

	// Fade them in from black.
	TintFader * tf = new TintFader(Color(1,1,1,1), Color(0,1.0f/60.0f,1.0f/60.0f,1.0f/60.0f));
	ps->animAdd(tf);

	// Let 'em stay on for 12 seconds.
	Delay * d = new Delay(12*60);
	tf->triggerAdd(new ChainAnimation(d));

	// Use a screenblinds to get rid of them.
	ScreenBlinds * sb = new ScreenBlinds(true);
	sb->setTint(Color(0,1,1,1));
	d->triggerAdd(new Birth(sb, m_scene));

	// Fade the screenblinds from transparent to opaque.
	tf = new TintFader(Color(1,1,1,1), Color(1.0f/60.0f,0,0,0));
	d->triggerAdd(new ChainAnimation(tf, sb));

	// When the tintfader is done, kill off the spheres.
	tf->triggerAdd(new Death(ps));

	// ... and switch the background color.
	tf->triggerAdd(new OneShot(this, 0));

	// ... and spawn a new delay on the big scene.
	delay = new Delay(1);
	tf->triggerAdd(new ChainAnimation(delay, m_scene));

	// ... and kill off the screenblinds.
	tf->triggerAdd(new Death(sb));

	return delay;
}

// Part 5 ///////////////////////////////////////////////////
Animation * Demo::part5Setup(Animation * delay) {
	// Make a new ScreenWave object, off-screen.
	ScreenWave * sw = new ScreenWave();
	sw->setTint(Color(1,0.75f,0.75f,0.75f));
	sw->setTranslate(Vector(0,480.0f,0.0001f));
	delay->triggerAdd(new Birth(sw, m_scene));

	// Move it upwards linearly.
	LinearMover * lm = new LinearMover(Vector(0,0,0.0001f), Vector(0,-480.0f/120.0f,0));
	sw->animAdd(lm);

	// And wait a few seconds...
	Delay * d = new Delay(4*60);
	lm->triggerAdd(new ChainAnimation(d, m_scene));

	// Bring in a rotocube, completely transparent.
	RotoCube * rc = new RotoCube();
	rc->setTint(Color(0,0.6f,0.6f,0.6f));
	d->triggerAdd(new Birth(rc, m_scene));

	// And fade it in.
	TintFader * tf = new TintFader(Color(1,0.6f,0.6f,0.6f), Color(1/60.0f,0,0,0));
	rc->animAdd(tf);

	// Wait a few seconds...
	d = new Delay(7*60);
	tf->triggerAdd(new ChainAnimation(d, m_scene));

	// Fade the rotocube to translucent.
	tf = new TintFader(Color(0.6f, 0.7f, 0.7f, 0.7f), Color(-0.4f/60.0f,0.1f/60.0f,0.1f/60.0f,0.1f/60.0f));
	d->triggerAdd(new ChainAnimation(tf, rc));

	// Wait a few seconds...
	d = new Delay(7*60 - 30);
	tf->triggerAdd(new ChainAnimation(d, m_scene));

	// Fade the bgcolor of our menu to black.
	Fader * f = new Fader(this);
	d->triggerAdd(new ChainAnimation(f, m_scene));

	// Wait a few seconds...
	d = new Delay(3*60);
	f->triggerAdd(new ChainAnimation(d, m_scene));

	// Fade the rotocube out.
	tf = new TintFader(Color(1.0f, 0.0f, 0.0f, 0.0f), Color(0,-1.0f/60.0f,-1.0f/60.0f,-1.0f/60.0f));
	d->triggerAdd(new ChainAnimation(tf, rc));

	// ...and the ScreenWave
	tf = new TintFader(Color(1.0f, 0.0f, 0.0f, 0.0f), Color(0,-0.75f/60.0f,-0.75f/60.0f,-0.75f/60.0f));
	d->triggerAdd(new ChainAnimation(tf, sw));

	// Kill off both when they're done fading.
	tf->triggerAdd(new Death(rc));
	tf->triggerAdd(new Death(sw));

	// And make a new animation to hang off of.
	delay = new Delay(1);
	tf->triggerAdd(new ChainAnimation(delay, m_scene));

	return delay;
}

// Part 6 ///////////////////////////////////////////////////
Animation * Demo::part6Setup(Animation * delay) {
	// Endscroller setup
	Texture * txr = new Texture("font.png", true);
	EndScroller * es = new EndScroller(txr);
	es->setTint(Color(1,1,1,1));
	es->setTranslate(Vector(0,0,200));
	delay->triggerAdd(new Birth(es, m_scene));

	// And copperbars.
	CopperBars * cb = new CopperBars();
	cb->setTint(Color(1,0,0,0));
	cb->setTranslate(Vector(0,0,100));
	delay->triggerAdd(new Birth(cb, m_scene));

	// Fade the copper bars in
	TintFader * tf = new TintFader(Color(1,1,1,1), Color(0,1.0f/60.0f,1.0f/60.0f,1.0f/60.0f));
	cb->animAdd(tf);

	// Aaaand that's all she wrote! EndScroller will take care of
	// the rest from here.

	return delay;
}

void modplug_start(const char * fn);
void vmuInit();
void vmuShutdown();
void doDemo() {
	RefPtr<Demo> p1 = new Demo();
	vmuInit();
	modplug_start("reverie.s3m");
	p1->doMenu();
	vmuShutdown();
}
