#include "config.h"
/******************************************************************************
 * tunnel.c - Partie principale: calcul et affichage du tunnel                *
 ******************************************************************************
 * Projet       : Code(R) - Tunnel                                            *
 * Auteur       : Arnaud Storq (http://norecess.planet-d.net)/Posse-Press 2003*
 ******************************************************************************/
#include "tunnel.h"

//---------------------------------------------------------------------------- CONSTANTES LOCALES
#define TUNNEL_AFFICHAGE_WIREFRAME      0       // 0=normal  1=wireframe  

#define TUNNEL_VITESSE                  0.2f    // vitesse d'animation du tunnel
#define TUNNEL_TEXTURE_VITESSE          0.0035f // vitesse du dplacement de la texture applique sur le tunnel

#define TUNNEL_CERCLE_POINTS_NOMBRE     32      // nombre de points par cercle
#define TUNNEL_CERCLE_NOMBRE            24      // nombre de cercles composant le tunnel

#define TUNNEL_CERCLE_DISTANCE          -1.3f   // distance entre chaques cercles composant le tunnel

#define TUNNEL_DEPLACEMENT_RATIO        0.2f    // ratio du dplacement du tunnel en profondeur
#define TUNNEL_PROFONDEUR_VITESSE       20.0f   // vitesse d'animation du tunnel en profondeur

#define TUNNEL_DEPLACEMENT_X_VITESSE_1  1.0f    // vitesse du deplacement du tunnel en X (1er cas)
#define TUNNEL_DEPLACEMENT_X_VITESSE_2  0.69f   // vitesse du deplacement du tunnel en X (2eme cas)
#define TUNNEL_DEPLACEMENT_Y_VITESSE_1  0.81f   // vitesse du deplacement du tunnel en Y (1er cas)
#define TUNNEL_DEPLACEMENT_Y_VITESSE_2  1.3f    // vitesse du deplacement du tunnel en Y (2eme cas)

#define TUNNEL_TEXTURE_WRAPPING         2.0f    // doit etre puissance de 2

#define FRUSTRUM_RATIO                  0.5f    // ratio du frustrum permettant un zoom 2D sur tout l'ecran

#define BROUILLARD_COMPOSANTE_R         0.7f    // composante R (rouge) du brouillard
#define BROUILLARD_COMPOSANTE_G         0.7f    // composante G (vert) du brouillard
#define BROUILLARD_COMPOSANTE_B         0.8f    // composante B (bleu) du brouillard
#define BROUILLARD_DENSITE              0.13f   // densite du brouillard

//---------------------------------------------------------------------------- MACRO LOCALES
#define MATH_PI				3.1415926535897932f
#define MATH_RADIAN(a)		(((float) a)*MATH_PI/180.0f)

//---------------------------------------------------------------------------- VARIABLES LOCALES
static unsigned int textureOpenGL = 0;
static float timerTextureDeplacement = 0.0f;

//---------------------------------------------------------------------------- BROUILLARD
static void
tunnelBrouillardActiver(void)
{
    float fogColor[4] =
    {
        BROUILLARD_COMPOSANTE_R, BROUILLARD_COMPOSANTE_G, BROUILLARD_COMPOSANTE_B, 1.0f
    };
    glFogfv(GL_FOG_COLOR, fogColor); // couleur
    
    glFogi(GL_FOG_MODE, GL_EXP2); // type du brouillard
    glFogf(GL_FOG_DENSITY, BROUILLARD_DENSITE); // epaisseur
    
    glFogf(GL_FOG_START, 0.0f); // le brouillard commence  Z=0..
    glFogf(GL_FOG_END, 1.0f); // et s'acheve  1 !
    
    glEnable(GL_FOG); // active le brouillard
}

//---------------------------------------------------------------------------- INITIALISER
void
tunnelInitialiser(unsigned char *texture, int largeur, int hauteur)
{
    glGenTextures(1, &textureOpenGL);

    glBindTexture(GL_TEXTURE_2D, textureOpenGL);

    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);    
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	
    glTexImage2D(GL_TEXTURE_2D, 0, 3, largeur, hauteur, 0, GL_RGB, GL_UNSIGNED_BYTE, texture);
}

//---------------------------------------------------------------------------- AFFICHER
void
tunnelAfficher(int time)
{
    int c; // cercle courrant
    float p; // point dans le cercle courrant
    
    float tunnelDeplacement; // deplacement en profondeur
    float incCercle; // increment pour un cercle complet
    
    float depX = 0.0f; // deplacement X du cercle courrant
    float depY = 0.0f; // deplacement Y du cercle courrant
    
    // initialise les timers permettant de simuler le dplacement
    float timerTunnelDeplacement = (float) time*TUNNEL_VITESSE;
    timerTextureDeplacement += TUNNEL_TEXTURE_VITESSE;
    
    // efface le Z-Buffer uniquement
    glClearDepth(1.0f);
    glClearColor(BROUILLARD_COMPOSANTE_R, BROUILLARD_COMPOSANTE_G, BROUILLARD_COMPOSANTE_B, 1.0f);
    glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
    
    // active le Z-Buffer
    // les points inferieurs (GL_LESS) au Z courant dans le Z-Buffer passent
    glEnable(GL_DEPTH_TEST);
    glDepthFunc(GL_LESS);

    // active l'application du brouillard
    tunnelBrouillardActiver();

    // active l'application de texture aux triangles
    glBindTexture(GL_TEXTURE_2D, textureOpenGL);
    glEnable(GL_TEXTURE_2D);
    
    // active le mode wireframe si necessaire
    if (TUNNEL_AFFICHAGE_WIREFRAME==1)
    {
        glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    }
    
    // initialise la matrice de projection
    // la constante FRUSTRUM_RATIO permet d'effectuer un Zoom 2D sur l'ecran
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glFrustum(-FRUSTRUM_RATIO, FRUSTRUM_RATIO, -FRUSTRUM_RATIO, FRUSTRUM_RATIO, 1.0f, 600.0f);
    
    // initialise la matrice de visualisation
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    
    // precalcule l'increment necessaire pour effectuer un cercle complet
    incCercle = 360.0f/TUNNEL_CERCLE_POINTS_NOMBRE;
    
    // pour chaques cercles..
    for (c=0; c<TUNNEL_CERCLE_NOMBRE; c++)
    {
        // deplace le cercle courant avec les valeurs precedemment calcules
        // la constante TUNNEL_CERCLE_DISTANCE permet d'avancer dans la prodondeur du tunnel
        // l'astuce du tunnel reside ici : la matrice courrante est repositionn
        // plus loin en profondeur  chaques appels  glTranslatef !
        glTranslatef(depX, depY, TUNNEL_CERCLE_DISTANCE);

        // calcule le deplacement X et Y du cercle courrant
        // les fonctions sinus/cosinus prennent des radians en entre
        // il est donc ncessaire de convertir les degrs en radians
        // la constante TUNNEL_DEPLACEMENT_RATIO permet de donner un mouvement
        // agrable  la profondeur du tunnel
        depX = sinf(MATH_RADIAN(timerTunnelDeplacement*TUNNEL_DEPLACEMENT_X_VITESSE_1))*TUNNEL_DEPLACEMENT_RATIO;
        depY = cosf(MATH_RADIAN(timerTunnelDeplacement*TUNNEL_DEPLACEMENT_Y_VITESSE_1))*TUNNEL_DEPLACEMENT_RATIO;
        depX -= cosf(MATH_RADIAN(timerTunnelDeplacement*TUNNEL_DEPLACEMENT_X_VITESSE_2))*TUNNEL_DEPLACEMENT_RATIO;
        depY -= sinf(MATH_RADIAN(timerTunnelDeplacement*TUNNEL_DEPLACEMENT_X_VITESSE_2))*TUNNEL_DEPLACEMENT_RATIO;
        
        // commence l'affichage des primitives du type Quadrangles (4 points par polygones)
        glBegin(GL_QUADS);
    
        // pour chaques points dans le cercle courrant..
        for (p=0.0f; p<360.0f; p+=incCercle)
        {    
                // les couples x1, y1 et x2, y2 sont positionns sur le cercle courrant
                // les couples x3, y3 et x4, y4 sont positionns sur le cercle suivant
                float x1 = sinf(MATH_RADIAN(p));
                float y1 = cosf(MATH_RADIAN(p));
                float x2 = x1+depX;
                float y2 = y1+depY;
                float x3 = sinf(MATH_RADIAN(p+incCercle));
                float y3 = cosf(MATH_RADIAN(p+incCercle));
                float x4 = x3+depX;
                float y4 = y3+depY;
                
                // calcule les coordonnes de texture pour le polygone actuel
                // les coordonnes horizontales u1 et u2 sont calculs grace  la position par rapport au cercle courrant
                // les coordonnes verticules v1 et v2 sont calcules grace  l'index du cercle courrant
                // la constante TUNNEL_TEXTURE_WRAPPING permet de redimensionner la texture
                float u1 = (p*TUNNEL_TEXTURE_WRAPPING)/360.0f;
                float u2 = ((p+incCercle)*TUNNEL_TEXTURE_WRAPPING)/360.0f;
                float v1 = (((float) c)*TUNNEL_TEXTURE_WRAPPING)/(float) TUNNEL_CERCLE_NOMBRE;
                float v2 = (((float) (c+1))*TUNNEL_TEXTURE_WRAPPING)/(float) TUNNEL_CERCLE_NOMBRE;
                
                // simule le dplacement dans le tunnel en modifiant les coordonnes u et v
                // et oui! les vertices n'avancent pas en Z, seule la texture se deplace..
                v1 += timerTextureDeplacement;
                v2 += timerTextureDeplacement;
                u1 += timerTextureDeplacement;
                u2 += timerTextureDeplacement;
                
                // dessine le quadrangle en specifiant les coordonnes texture puis les points
                glTexCoord2f(u1, v1);
                glVertex3f(x1, y1, 0.0f);                
                glTexCoord2f(u1, v2);
                glVertex3f(x2, y2, TUNNEL_CERCLE_DISTANCE);                
                glTexCoord2f(u2, v2);
                glVertex3f(x4, y4, TUNNEL_CERCLE_DISTANCE);     
                glTexCoord2f(u2, v1);
                glVertex3f(x3, y3, 0.0f);
        }
            
        // cesse l'affichage des quadrangles
        glEnd();
        
        // mise  jour du deplacement vers la profondeur
        timerTunnelDeplacement += TUNNEL_PROFONDEUR_VITESSE;
    }
}

//---------------------------------------------------------------------------- DETRUIRE
void
tunnelDetruire(void)
{
    // detruit la texture OpenGL
    glDeleteTextures(1, &textureOpenGL);
}

