/*******************************************************

Original Author...Håvard Christensen
Purpose...........vector math

********************************************************/


//TODO:
/*
- needs to be tested in action to iron out any hidden bugs.
*/

/*!
Description:
This is the vector math structures for fusausion.
They are templetized to support different types.
there are some shorthands allready defined.
cVecxf where x is a number from 2 to 4, is a shorthand for cVecx<float> (cVec2f,cVec3f,cVec4f)

cVecxfp4 where x is a number from 2 to 4, is a shorthand for
	cVecx<cFPValFrac4Signed> (cVec2fp4,cVec3fp4,cVec4fp4) a vector with fixed point ( 4 bits for fraction)

cVecxfp8 where x is a number from 2 to 4, is a shorthand for
	cVecx<cFPValFrac8Signed> (cVec2fp8,cVec3fp8,cVec4fp8) a vector with fixed point ( 8 bits for fraction)

cVecxfp12 where x is a number from 2 to 4, is a shorthand for
	cVecx<cFPValFrac12Signed> (cVec2fp12,cVec3fp12,cVec4fp12) a vector with fixed point ( 12 bits for fraction)

cVecxfp14 where x is a number from 2 to 4, is a shorthand for
	cVecx<cFPValFrac14Signed> (cVec2fp14,cVec3fp14,cVec4fp14) a vector with fixed point ( 14 bits for fraction)
*/


#ifndef H_xxFUSA_VECTORMATHxx_H
#define H_xxFUSA_VECTORMATHxx_H
#include <cmath>
#include <iostream>
#include "fusa_fixedpointTypes.h"

namespace fusa
{

	/*!
	@class cVec2
	@brief cVec2 is a mathematical vector/point of dimension two. the datatype is templetized and can be fixed-point,float, double etc.
	*/
	template<class Ta>
	class cVec2
	{
	public:
		cVec2()
		{
			x = y = 0;
		}

		cVec2(Ta xIn,Ta yIn)
		{
			x=xIn;
			y=yIn;
		}

		cVec2(const cVec2& vec)
		{
			x=vec.x;
			y=vec.y;
		}
		
		///adds two vectors and returns result
		cVec2 operator+(const cVec2& vec)const
		{
			return cVec2(x + vec.x,y + vec.y);
		}

		///subtracts two vectors and returns result
		cVec2 operator-(const cVec2& vec)const
		{
			return cVec2(x - vec.x,y - vec.y);
		}

		///returns a negated vector
		cVec2 operator-()const
		{
			return cVec2(-x,-y);
		}
		
		///multiplies the vector components with scalar value val
		cVec2 operator*(Ta val)const
		{
			return cVec2(x * val,y * val);
		}
		
		///divides the vector components with scalar value val
		cVec2 operator/(Ta val)const
		{
			return cVec2(x / val,y / val);
		}

		///adds a vector vec to the calling vector
		void operator+=(const cVec2& vec)
		{
			x+=vec.x;
			y+=vec.y;
		}
		
		///subtracts a vector vec from the calling vector
		void operator-=(const cVec2& vec)
		{
			x-=vec.x;
			y-=vec.y;
		}

		///multiplies the calling vector with scalar value val
		void operator*=(Ta val)
		{
			x*=val;
			y*=val;
		}

		///divides the calling vector with scalar value val
		void operator/=(Ta val)
		{
			x/=val;
			y/=val;
		}

		///gets the magnitude of the vector.
		Ta magnitude()const
		{
			return Ta(std::sqrt(x * x + y * y));
		}

		Ta squareLength()const
		{
			return (x * x + y * y);
		}

		///normalize the vector
		void normalize()
		{
			Ta magDiv = Ta(1)/magnitude();
			x*=magDiv;
			y*=magDiv;
		}

		///return a normalized version of the vector.
		cVec2 getNormalized()const
		{
			cVec2 res(x,y);
			res.normalize();
			return res;
		}

		///returns which component is smallest in absolute value
		unsigned char smallestComponentAbs()const
		{
			Ta xAbs=x;
			Ta yAbs=y;
			if(xAbs < yAbs)
			{
				return 0;
			}
			return 1;
		}

		///returns which component is biggest in absolute value
		unsigned char biggestComponentAbs()const
		{
			Ta xAbs=x;
			Ta yAbs=y;
			if(xAbs > yAbs)
			{
				return 0;
			}
			return 1;
		}


		///returns a normalized vector perpendicular to the vector it is called on.
		cVec2 perpendicular()const
		{
			cVec2 res(y,-x);
			res.normalize();
			return res;
		}

		///return the dot product between the calling vector and parameter vec
		float dotProd(const cVec2& vec)const
		{
			return (x * vec.x + y * vec.y);
		}

		///returns a reflected vector.
		cVec2 reflect()const
		{
			return cVec2(-x,y);
		}

		///reflects the calling vector.
		void reflectMe()
		{
			x=-x;
		}

		///get vector element at index acc.
		Ta& operator[](unsigned int acc)
		{
			return (&x)[acc];
		}

		///get const vector element at index acc.
		const Ta& operator[](unsigned int acc)const
		{
			return (&x)[acc];
		}

		void set(const Ta &x,const Ta &y)
		{
			this->x = x;
			this->y = y;
		}

		bool equals(const cVec2 & vec,Ta epsilon=0.0)
		{
			if(vec.x >= (x - epsilon) && vec.x <= (x + epsilon))
				if(vec.y >= (y - epsilon) && vec.y <= (y + epsilon))
					return true;
			return false;
		}


		Ta x,y;
	};

	/*!
	@class cVec3
	@brief cVec3 is a mathematical vector/point of dimension three. the datatype is templetized and can be fixed-point,float, double etc.
	*/
	template<class Ta>
	class cVec3
	{
	public:
		cVec3()
		{
			x=y=z=0;
		}
		cVec3(Ta xIn,Ta yIn,Ta zIn)
		{
			x=xIn;
			y=yIn;
			z=zIn;
		}
		cVec3(const cVec3& vec)
		{
			x=vec.x;
			y=vec.y;
			z=vec.z;
		}

		const cVec3& operator=(const cVec3& vec)
		{
			x=vec.x;
			y=vec.y;
			z=vec.z;
			return *this;
		}

		///adds two vectors and returns result
		cVec3 operator+(const cVec3& vec)const
		{
			return cVec3(x + vec.x,y + vec.y,z + vec.z);
		}

		///subtracts two vectors and returns result
		cVec3 operator-(const cVec3& vec)const
		{
			return cVec3(x - vec.x,y - vec.y,z - vec.z);
		}

		///returns a negated vector
		cVec3 operator-()const
		{
			return cVec3(-x,-y,-z);
		}

		///multiplies the vector components with scalar value val
		cVec3 operator*(Ta val)const
		{
			return cVec3(x * val,y * val,z * val);
		}

		///divides the vector components with scalar value val
		cVec3 operator/(Ta val)const
		{
			return cVec3(x / val,y / val,z / val);
		}

		///adds a vector vec to the calling vector
		void operator+=(const cVec3& vec)
		{
			x+=vec.x;
			y+=vec.y;
			z+=vec.z;
		}

		///subtracts a vector vec from the calling vector
		void operator-=(const cVec3& vec)
		{
			x-=vec.x;
			y-=vec.y;
			z-=vec.z;
		}

		///multiplies the calling vector with scalar value val
		void operator*=(Ta val)
		{
			x*=val;
			y*=val;
			z*=val;
		}

		///divides the calling vector with scalar value val
		void operator/=(Ta val)
		{
			x/=val;
			y/=val;
			z/=val;
		}

		///returns magnitude
		Ta magnitude()const
		{
			return Ta(std::sqrt(x*x + y*y + z*z));
		}

		Ta squareLength()const
		{
			return (x * x + y * y + z * z);
		}

		///normalizes the vector
		void normalize()
		{
			Ta magDiv = Ta(1)/magnitude();
			x*=magDiv;
			y*=magDiv;
			z*=magDiv;
		}

		///returns normalized vector
		cVec3 getNormalized()const
		{
			cVec3 res(x,y,z);
			res.normalize();
			return res;
		}

			bool equals(const cVec3 & vec,Ta epsilon=0.0)
		{
			if(vec.x >= (x - epsilon) && vec.x <= (x + epsilon))
				if(vec.y >= (y - epsilon) && vec.y <= (y + epsilon))
					if(vec.z >= (z - epsilon) && vec.z <= (z + epsilon))
					return true;
			return false;
		}

		///returns which component is smallest in absolute value, 0 is x, 1 is y , 2 is z.
		unsigned char smallestComponentAbs()const
		{
			//returns which component is smallest.
			Ta xAbs = std::abs(x);
			Ta yAbs = std::abs(y);
			Ta zAbs = std::abs(z);

			if(xAbs < yAbs)
			{//either x or z is smallest.
				if(xAbs<zAbs)
				{
					//x is smallest
					return 0;
				}
				else
				{
					//z is smallest
					return 2;
				}
			}
			//x can not be smallest, either y or z is smallest.
			if(yAbs < zAbs)
			{// y is smallest.
				return 1;
			}
			//z is smallest.
			return 2;
		}

		///returns which component is biggest in absolute value, 0 is x, 1 is y , 2 is z.
		unsigned char biggestComponentAbs()const
		{
			//returns which component is biggest.
			Ta xAbs = std::abs(x);
			Ta yAbs = std::abs(y);
			Ta zAbs = std::abs(z);

			if(xAbs > yAbs)
			{//either x or z is biggest.
				if(xAbs>zAbs)
				{
					//x is biggest
					return 0;
				}
				else
				{
					//z is biggest
					return 2;
				}
			}
			//x can not be biggest, either y or z is biggest.
			if(yAbs > zAbs)
			{// y is biggest.
				return 1;
			}
			//z is biggest.
			return 2;
		}

		///returns a normalized vector perpendicular to the vector it is called on.
		cVec3 perpendicular()const
		{
			cVec3 res;
			char smallComp = smallestComponentAbs();
			if(smallComp==0)
			{
				res.y=-z;
				res.z=y;
				res.normalize();
				return res;
			}
			if(smallComp==1)
			{
				res.x=-z;
				res.z=x;
				res.normalize();
				return res;
			}

			res.x=-y;
			res.y=x;
			res.normalize();
			return res;

		}

		///returns dot product between calling vector and parameter vec
		Ta dotProd(const cVec3& vec)const
		{
			return (x * vec.x + y * vec.y + z * vec.z);
		}

		///returns cross product between calling vector and parameter vec
		cVec3 crossProd(const cVec3& vec)const
		{

			return cVec3((y*vec.z)-(z*vec.y),(z*vec.x)-(x*vec.z),(x*vec.y)-(y*vec.x));
		}

		///reflects the calling vector around normal n
		void reflectMe(const cVec3 &n)
		{
			*this = n*Ta(2) * (n.dotProd(*this)) - (*this);
		}

		///returns calling vector reflected around n.
		cVec3 reflect(const cVec3 &n)const
		{
			return cVec3(n*Ta(2) * (n.dotProd(*this)) - (*this));
		}

		///accesss value at index acc. (acc = 0, is x, 1 is y, 2 is z)
		Ta& operator[](unsigned int acc)
		{
			return (&x)[acc];
		}

		///accesss value at index acc. (acc = 0, is x, 1 is y, 2 is z)
		const Ta& operator[](unsigned int acc)const
		{
			return (&x)[acc];
		}

		///returns true if all components is equal to val, uses epsilol as epsilon value.
		bool allComponents(Ta val,Ta epsilol=0.00001)
		{
			//FIX_ME does not work for some reason so I just check if all comps zero here.

			Ta dotProd = this->dotProd(*this);
			if(dotProd > -epsilol && dotProd < epsilol)
				return true;
			return false;
			if( !(x > val - epsilol && x < val + epsilol) )
				return false;
			if(!(y > val - epsilol && y < val + epsilol) )
				return false;
			if(!(z > val - epsilol && z < val + epsilol) )
				return false;
			
			return true;
		}

		void set(const Ta &x,const Ta &y, const Ta &z)
		{
			this->x = x;
			this->y = y;
			this->z = z;
		}

		Ta x,y,z;
	};

	/*!
	@class cVec4
	@brief cVec4 is a mathematical vector/point of dimension four. the datatype is templetized and can be fixed-point,float, double etc.
	*/
	template<class Ta>
	class cVec4
	{
	public:
		cVec4()
		{
			x=y=z=w=0;
		}
		cVec4(Ta xIn,Ta yIn,Ta zIn,Ta wIn)
		{
			x=xIn;
			y=yIn;
			z=zIn;
			w=wIn;
		}
		cVec4(const cVec4& vec)
		{
			x=vec.x;
			y=vec.y;
			z=vec.z;
			w=vec.w;
		}
		
		///adds two vectors and returns result
		cVec4 operator+(const cVec4& vec)const
		{
			return cVec4(x + vec.x,y + vec.y,z + vec.z,w + vec.w);
		}

		///subtracts two vectors and returns result
		cVec4 operator-(const cVec4& vec)const
		{
			return cVec4(x - vec.x,y - vec.y,z - vec.z,w - vec.w);
		}

		///returns a negated vector.
		cVec4 operator-()const
		{
			return cVec4(-x,-y,-z,-w);
		}

		///multiplies the vector components with scalar value val
		cVec4 operator*(Ta val)const
		{
			return cVec4(x * val,y * val,z * val,w * val);
		}

		///divides the vector components with scalar value val
		cVec4 operator/(Ta val)const
		{
			return cVec4(x / val,y / val,z / val,w / val);
		}

		///adds a vector vec to the calling vector
		void operator+=(const cVec4& vec)
		{
			x+=vec.x;
			y+=vec.y;
			z+=vec.z;
			w+=vec.w;
		}
	
		///subtracts a vector vec from the calling vector
		void operator-=(const cVec4& vec)
		{
			x-=vec.x;
			y-=vec.y;
			z-=vec.z;
			w-=vec.w;
		}

		///multiplies the calling vector with scalar value val
		void operator*=(Ta val)
		{
			x*=val;
			y*=val;
			z*=val;
			w*=val;
		}

		///divides the calling vector with scalar value val
		void operator/=(Ta val)
		{
			val = Ta(1)/val;
			x*=val;
			y*=val;
			z*=val;
			w*=val;
		}

		///returns the magnitude
		float magnitude()const
		{
			return std::sqrt(x*x + y*y + z*z + w*w);
		}

		Ta squareLength()const
		{
			return (x * x + y * y + z * z + w * w);
		}

		///normalize the vector
		void normalize()
		{
			float magDiv = Ta(1)/magnitude();
			x*=magDiv;
			y*=magDiv;
			z*=magDiv;
			w*=magDiv;
		}

		///returns the normalized vector
		cVec4 getNormalized()const
		{
			cVec4 res(x,y,z,w);
			res.normalize();
			return res;
		}

		bool equals(const cVec4 & vec,Ta epsilon=0.0)
		{
			if(vec.x >= (x - epsilon) && vec.x <= (x + epsilon))
				if(vec.y >= (y - epsilon) && vec.y <= (y + epsilon))
					if(vec.z >= (z - epsilon) && vec.z <= (z + epsilon))
						if(vec.w >= (w - epsilon) && vec.w <= (w + epsilon))
					return true;
			return false;
		}

		///returns which component is smallest in absolute value, 0 is x, 1 is y , 2 is z, 3 is w
		unsigned char smallestComponentAbs()const
		{
			//returns which component is smallest.
			Ta xAbs = std::abs(x);
			Ta yAbs = std::abs(y);
			Ta zAbs = std::abs(z);
			Ta wAbs = std::abs(w);

			if(xAbs < yAbs)
			{//either x or z or w is smallest.
				if(xAbs<zAbs)
				{
					//x or w is smallest
					if(xAbs < wAbs)
					{
						return 0;
					}
					// w is smallest.
					return 3;

				}
				else
				{
					//z is smallest or w is smallest.
					if(zAbs < wAbs)
					{
						return 2;
					}

					return 3;
				}
			}
			//x can not be smallest, either y or z or w is smallest.
			if(yAbs < zAbs)
			{// y is smallest or w
				if(yAbs<wAbs)
				{
					return 1;
				}
				return 3;
			}
			//z is smallest or w
			if(zAbs < wAbs)
				return 2;
			return 3;
		}

		///returns which component is biggest in absolute value, 0 is x, 1 is y , 2 is z, 3 is w
		unsigned char biggestComponentAbs()const
		{
			//returns which component is biggest.
			Ta xAbs = std::abs(x);
			Ta yAbs = std::abs(y);
			Ta zAbs = std::abs(z);
			Ta wAbs = std::abs(w);

			if(xAbs > yAbs)
			{//either x or z or w is biggest.
				if(xAbs > zAbs)
				{
					//x or w is biggest
					if(xAbs > wAbs)
					{
						return 0;
					}
					// w is smallest.
					return 3;

				}
				else
				{
					//z is biggest or w is smallest.
					if(zAbs > wAbs)
					{
						return 2;
					}

					return 3;
				}
			}
			//x can not be biggest, either y or z or w is biggest.
			if(yAbs > zAbs)
			{// y is biggest or w
				if(yAbs > wAbs)
				{
					return 1;
				}
				return 3;
			}
			//z is biggest or w
			if(zAbs > wAbs)
				return 2;
			return 3;
		}


		///returns the dotproduct between the calling vector and parameter vec
		float dotProd(const cVec4& vec)const
		{
			return (x*vec.x + y*vec.y + z*vec.z + w*vec.w);
		}

		///access vector component at index acc
		Ta& operator[](unsigned int acc)
		{
			return (&x)[acc];
		}

		///access vector component at index acc
		const Ta& operator[](unsigned int acc)const
		{
			return (&x)[acc];
		}

		void set(const Ta &x,const Ta &y, const Ta &z,const Ta &w)
		{
			this->x = x;
			this->y = y;
			this->z = z;
			this->w = w;
		}

		Ta x,y,z,w;
	};


	//operators outside class.

	template<class Ta>
	cVec2<Ta> operator*(float f,const cVec2<Ta> &val)
	{
		return cVec2<Ta> (f*val.x,f*val.y);
	}

	template<class Ta>
	cVec3<Ta> operator*(float f,const cVec3<Ta> &val)
	{
		return cVec3<Ta> (f*val.x,f*val.y,f*val.z);
	}

	template<class Ta>
	cVec4<Ta> operator*(float f,const cVec4<Ta> &val)
	{
		return cVec4<Ta> (f*val.x,f*val.y,f*val.z,f*val.w);
	}

	template<class Ta>
	cVec2<Ta> operator/(float f,const cVec2<Ta> &val)
	{
		return cVec2<Ta> (f/val.x,f/val.y);
	}

	template<class Ta>
	cVec3<Ta> operator/(float f,const cVec3<Ta> &val)
	{
		return cVec3<Ta> (f/val.x,f/val.y,f/val.z);
	}

	template<class Ta>
	cVec4<Ta> operator/(float f,const cVec4<Ta> &val)
	{
		return cVec4<Ta> (f/val.x,f/val.y,f/val.z,f/val.w);
	}

	template<class Ta>
	std::ostream& operator<<(std::ostream &os,const cVec2<Ta> &val)
	{
		return os<<val.x<<" "<<val.y;
	}

	template<class Ta>
	std::istream& operator>>(std::istream &is,cVec2<Ta> &val)
	{
		return is>>val.x>>val.y;
	}

	template<class Ta>
	std::ostream& operator<<(std::ostream &os,const cVec3<Ta> &val)
	{
		return os<<val.x<<" "<<val.y<<" "<<val.z;
	}

	template<class Ta>
	std::istream& operator>>(std::istream &is,cVec3<Ta> &val)
	{
		return is>>val.x>>val.y>>val.z;
	}

	template<class Ta>
	std::ostream& operator<<(std::ostream &os,const cVec4<Ta> &val)
	{
		return os<<val.x<<" "<<val.y<<" "<<val.z<<" "<<val.w;
	}

	template<class Ta>
	std::istream& operator>>(std::istream &is,cVec4<Ta> &val)
	{
		return is>>val.x>>val.y>>val.z>>val.w;
	}

	///cVec2f is a vector of two dimensions with float as data type
	typedef cVec2<float> cVec2f;
	///cVec2f is a vector of three dimensions with float as data type
	typedef cVec3<float> cVec3f;
	///cVec2f is a vector of four dimensions with float as data type
	typedef cVec4<float> cVec4f;

	///fixed point 4 bits fractional part vectors
	/*typedef cVec2<cFPSignShort4> cVec2fpShort4;
	typedef cVec3<cFPSignShort4> cVec3fpShort4;
	typedef cVec4<cFPSignShort4> cVec4fpShort4;

	///fixed point 8 bits fractional part vectors
	typedef cVec2<cFPSignShort8> cVec2fpShort8;
	typedef cVec3<cFPSignShort8> cVec3fpShort8;
	typedef cVec4<cFPSignShort8> cVec4fpShort8;

	///fixed point 12 bits fractional part vectors
	typedef cVec2<cFPSignShort12> cVec2fpShort12;
	typedef cVec3<cFPSignShort12> cVec3fpShort12;
	typedef cVec4<cFPSignShort12> cVec4fpShort12;

	///fixed point 14 bits fractional part vectors
	typedef cVec2<cFPSignShort14> cVec2fpShort14;
	typedef cVec3<cFPSignShort14> cVec3fpShort14;
	typedef cVec4<cFPSignShort14> cVec4fpShort14;*/
}
#endif

