/*******************************************************
Original Author...Hvard Christensen
Purpose...........to provide an interface for animational controllers

Description: A Controller is something that takes in a a number t, 
which is in the range 0 <= t <= 1 and then returns a calculated value
based on the type of controller and t. i.e, a linear controller 
brings a linear transitions between its animation keys.


********************************************************/


//TODO:
/*
- circle controller not implemented
- slerp controller not implemented 
- squad controller not implemented 
*/

#ifndef H_xxFUSA_CONTROLLERSxx_H
#define H_xxFUSA_CONTROLLERSxx_H

#include "fusa_vectormath.h"
#include "fusa_mathfuncs.h"
#include <list>

//FIX_ME not tested thoroughly.
namespace fusa
{
	/*!
	@class cBaseController
	@brief cBaseController is the base class for controllers. a controller returns animated/interpolated data given a t in range 0 < t < 1
	*/
	class cBaseController
	{
	public:
		
		enum ControllerType
		{
			///linearController
			linearControllerType,
			///tcbController (curves with tension, continuity and bias control)
			tcbControllerType,
			///circleController type.
			circleControllerType
		};

		cBaseController();
		cBaseController(const cBaseController &rhs);
		const cBaseController& operator=(const cBaseController &rhs);
		
		
		///returns the interpolated data
		/*!
		Takes parameter t, which should be in the rage 0 to 1.
		if ignoreRepeat is false, it will repeat the controllers animation m_repeatCount times
		during the range 0 < t < 1. if ignoreRepeat is true, it will not repeat the controller.
		*/
		virtual fusa::cVec3f getData(float t, bool ignoreRepeat = false)const =0;

		///returns the controller type
		virtual ControllerType getControllerType()const=0;
		
		///create a clone of the controller.
		virtual cBaseController* clone()const=0;

		///load function, formatID allows for differnt load-paths for different formats 
		///so that old formats still can be loaded
		virtual bool loadControllerFile(std::ifstream &in,bool binFormat=false,unsigned int formatVersion=0);
		
		///saves controller file to ofstream out.
		/*!
		if binFormat is true, it saves binary version, else it saves ascii format.
		binary format is currently not supported.
		*/
		virtual bool saveControllerFile(std::ofstream &out,bool binFormat=false)const;

		static unsigned int getFileFormatVersion();

		///sets nr of components to update, 1 means x is updated, 2 means xy, 3 means xyz.
		void setNrOfComponentsToUpdate(unsigned int doWorkOn);
		unsigned int getNrOfComponentsToUpdate()const;

		virtual void flushKeys()const
		{
			
		}

		///sets the repeatCount for the animation/interpolation
		/*!
		example. if you have a controller which goes from 0 to 100 in the range 0 - 1, you can with repeatCount
		make it go from 0 to 100 x amount of times in the range 0 - 1.
		*/
		void setRepeatCount(unsigned int val)
		{
			m_repeatCount = val;
		}

		unsigned int getRepeatCount()
		{
			return m_repeatCount;
		}

	virtual ~cBaseController();
	protected:
		unsigned int m_doWorkOn;
		unsigned int m_repeatCount;
	};


	/*!
	@class cLinearController
	@brief cLinearController provides a linear controller, which lineararly interpolates keys
	*/
	class cLinearController : public cBaseController
	{
		/*!
		@class sLinearKey
		@brief sLinearKey holds the linearKeyData
		*/
		struct sLinearKey
		{
			bool operator<(const sLinearKey &key)
			{
				return (t<key.t);
			}

			///time key
			float t;
			///key data.
			cVec3f data;
		};
	public:
		cLinearController();
		cLinearController(const cLinearController &rhs);
		
		const cLinearController& operator=(const cLinearController &rhs);
		

		ControllerType getControllerType()const
		{
			return linearControllerType;
		}

		cBaseController* clone()const;
		
		fusa::cVec3f getData(float t, bool ignoreRepeat = false)const;
		void addData(const fusa::cVec3f &data,float t);
		

		unsigned short getNumberOfKeys()const;
		
		///get key at index keyNum
		sLinearKey& KeyAccess(unsigned short keyNum);
		const sLinearKey& KeyAccess(unsigned short keyNum)const;
		
		///gets max values per chan. (val.x is highest possible x value in any point that is returned etc.. )
		cVec3f getMaxValues()const;

		///gets min values per chan. (val.x is lowest possible x value in any point that is returned etc.. )
		cVec3f getMinValues()const;

		///after mucking about with keys you might want to sort the list again. (especially if you are changing time values)
		void sortKeys();
		
		///this goes through the list of keys, and for any key that has bigger t than the next key in the list
		///its t is set to same as the next key.
		void fixOverlapping();

		bool loadControllerFile(std::ifstream &in,bool binFormat=false,unsigned int formatVersion=0);
		bool saveControllerFile(std::ofstream &out,bool binFormat=false)const;

		virtual void flushKeys()const
		{
			//m_controllerKeys.clear();
		}
	protected:
		std::list<sLinearKey> m_controllerKeys;
	};

	/*!
	@class cTcbController
	@brief cTcbController provides an interface for a tcbController.
	*/
	class cTcbController : public cBaseController
	{
		public:
		/*!
		@class sTcbKey
		@brief sTcbKey holds the key data for a tcbKey
		*/
		struct sTcbKey
		{
			///key time.
			float t;
			///key data
			fusa::cVec3f data;
			///key tension
			float tens;
			///key continiuity
			float cont;
			///key bias.
			float bias;

			bool operator<(const sTcbKey &key)
			{
				return (t<key.t);
			}
		};
	

		cTcbController();
		cTcbController(const cTcbController &rhs);
		const cTcbController& operator=(const cTcbController &rhs);
		ControllerType getControllerType()const;
		
		cBaseController* clone()const;
		

		fusa::cVec3f getData(float t, bool ignoreRepeat = false)const;
		
//add data without caring about time.
		//all keys timestamps are automatically calculated.
		void addData(const fusa::cVec3f &data,float tension,float bias,float cont);
		void addData(const fusa::cVec3f &data,float t,float tension,float bias,float cont);
		
		unsigned short getNumberOfKeys()const;
		
		sTcbKey& KeyAccess(unsigned short keyNum);
		const sTcbKey& KeyAccess(unsigned short keyNum)const;
		
		///gets max values per chan. (val.x is highest possible x value in any point that is returned etc.. )
		cVec3f getMaxValues()const;


		///gets min values per chan. (val.x is lowest possible x value in any point that is returned etc.. )
		cVec3f getMinValues()const;

		///after mucking about with keys you might want to sort the list again.
		void sortKeys();
		void fixOverlapping();

		bool loadControllerFile(std::ifstream &in,bool binFormat=false,unsigned int formatVersion=0);
		bool saveControllerFile(std::ofstream &out,bool binFormat=false)const;

		void flushKeys()
		{
			m_controllerKeys.clear();
		}
		
		

	protected:
		std::list<sTcbKey> m_controllerKeys;

	};
}


#endif
