/*
 * This file is part of newduel.
 *
 * newduel is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * newduel is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with newduel.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdlib.h>
#include <stdint.h>
#include <sys/io.h>
#include <sys/tms9918a.h>
#include <sys/sn79489.h>
#include <sms.h>
#include <vdp.h>

#include <player_palette.h>
#include <player_tilemap.h>
#include <player_tiles.h>

#include <env_palette.h>
#include <env_tilemap.h>
#include <env_tiles.h>

/* Port 0xDC io_ab_reg */
#define A_UP      (1 << 0)
#define A_DOWN    (1 << 1)
#define A_LEFT    (1 << 2)
#define A_RIGHT   (1 << 3)
#define A_TL      (1 << 4)
#define A_TR      (1 << 5)
#define B_UP      (1 << 6)
#define B_DOWN    (1 << 7)

/* Port 0xDD io_bmisc_reg */
#define B_LEFT    (1 << 0)
#define B_RIGHT   (1 << 1)
#define B_TL      (1 << 2)
#define B_TR      (1 << 3)

const unsigned char vdp_registers[] = {
	0x06, /* register 0x0 */ /* 6 M4 */
	0x20, /* register 0x1 */ /* 1 IE0 */
	0xFF, /* register 0x2 */ /* 0x3700 is the name table bass address */
	0xFF, /* register 0x3 */ /* Must be 0xFF */
	0xFF, /* register 0x4 */ /* Must be 0xFF */
	0xFF, /* register 0x5 */
	0xFB, /* register 0x6 */
	0xF0, /* register 0x7 Overscan/Backdrop Color */
	0x00, /* register 0x8 Background X scroll value */
	0x00, /* register 0x9 Background Y scroll value */
	0x00, /* register 0xa Line counter */
};

const unsigned char level[24][32] = {
	{  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0, 5, 6, 0, 0, 0, 0, 0, 0, 5, 6, 0, 0, 0, 0, 0, 0, 5, 6, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0, 7, 8, 0, 0, 0, 0, 0, 0, 7, 8, 0, 0, 0, 0, 0, 0, 7, 8, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 1, 1, 1, 1, 1, 3, 4, 1, 0, 0, 1, 1, 1, 3, 4, 1, 1, 1, 0, 0, 1, 3, 4, 1, 1, 1, 1, 1, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0,11,12, 0, 0, 0, 0, 0, 0,11,12, 0, 0, 0, 0, 0, 0,11,12, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0, 7, 8, 0, 0, 0, 0, 0, 0, 7, 8, 0, 0, 0, 0, 0, 0, 7, 8, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0, 9,10, 0, 0, 0, 0, 0, 0, 9,10, 0, 0, 0, 0, 0, 0, 9,10, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0,11,12, 0, 0, 0, 0, 0, 0,11,12, 0, 0, 0, 0, 0, 0,11,12, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 0, 0, 1, 1, 1, 3, 4, 1, 0, 0, 0, 1, 1, 3, 4, 1, 1, 0, 0, 0, 1, 3, 4, 1, 1, 1, 0, 0, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0,11,12, 0, 0, 0, 0, 0, 0, 9,10, 0, 0, 0, 0, 0, 0,11,12, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0, 7, 8, 0, 0, 0, 0, 0, 0,11,12, 0, 0, 0, 0, 0, 0, 7, 8, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0, 9,10, 0, 0, 0, 0, 0, 0, 7, 8, 0, 0, 0, 0, 0, 0, 9,10, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0,11,12, 0, 0, 0, 0, 0, 0, 9,10, 0, 0, 0, 0, 0, 0,11,12, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 1, 1, 1, 1, 1, 3, 4, 1, 0, 0, 1, 1, 1, 2, 2, 1, 1, 1, 0, 0, 1, 3, 4, 1, 1, 1, 1, 1, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0, 9,10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9,10, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0,11,12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,11,12, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0, 7, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 8, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0, 9,10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9,10, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 1, 1, 1, 1, 1, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 2, 1, 1, 1, 1, 1, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
	{  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, },
	{ 13,14,13,14,13,14,13,14,13,14,13,14,13,14,13,14,13,14,13,14,13,14,13,14,13,14,13,14,13,14,13,14, },
};

#define STAND 0
#define RUN 1
#define ROLL 2
#define CLIMB 3
#define HIT 4
#define DUCK 5
#define JUMP 6
#define LEAP 7
#define FALLING 8

uint8_t player_stand[] = {
	/* up */
	0x10, 0x11, 0x12, 0x12,
	0x13, 0x14, 0x12, 0x12,
	0x15, 0x16, 0x12, 0x12,
	0x17, 0x18, 0x12, 0x12,
	/* down */
	0x19, 0x1A, 0x12, 0x12,
	0x1B, 0x1C, 0x12, 0x12,
	0x1D, 0x1E, 0x12, 0x12,
	0x1F, 0x20, 0x12, 0x12,
};

uint8_t player_climb[] = {
	0x45, 0x46, 0x12, 0x12,
	0x47, 0x48, 0x12, 0x12,
	0x49, 0x4A, 0x12, 0x12,
	0x4B, 0x4C, 0x12, 0x12,
};

uint8_t player_duck[] = {
	0x12, 0x12, 0x12, 0x12,
	0x12, 0x66, 0x67, 0x12,
	0x68, 0x69, 0x6A, 0x12,
	0x6B, 0x6C, 0x6D, 0x12,
};

uint8_t player_fall[] = {
	0x5F, 0x60, 0x12, 0x12,
	0x58, 0x61, 0x12, 0x12,
	0x62, 0x63, 0x12, 0x12,
	0x64, 0x65, 0x12, 0x12,
};

uint8_t player_run[] = {
	/* frame 1 */
	0x21, 0x22, 0x12, 0x12,
	0x23, 0x24, 0x12, 0x12,
	0x25, 0x26, 0x12, 0x12,
	0x27, 0x28, 0x12, 0x12,
	/* frame 2 */
	0x29, 0x2A, 0x12, 0x12,
	0x2B, 0x2C, 0x2D, 0x12,
	0x2E, 0x2F, 0x30, 0x12,
	0x31, 0x32, 0x33, 0x12,
	/* frame 3 */
	0x34, 0x35, 0x12, 0x12,
	0x36, 0x37, 0x12, 0x12,
	0x38, 0x39, 0x12, 0x12,
	0x3A, 0x3B, 0x12, 0x12,
	/* frame 4 */
	0x3C, 0x3D, 0x12, 0x12,
	0x3E, 0x3F, 0x40, 0x12,
	0x41, 0x42, 0x12, 0x12,
	0x31, 0x43, 0x44, 0x12,
};

/**
 * Define our two players
 * x, y the position on the screen
 * vx, vy the velocity the character is moving at
 */
typedef struct {
	uint8_t x, y;
	int8_t vx, vy;
	uint8_t current_animation, current_frame;
	uint8_t delay, animation_delay;
	uint8_t sprites[16]; // we potentially have 16 sprites in the SAT
	uint8_t animation_frames;
	uint8_t falling, climbing, jumping;
	uint8_t *animation_sprites;

#if 0
		struct running {
			uint8_t frames = 4;
			uint8_t tiles[] = { 8, 10, 8, 10 }; /* number of tiles per frame */
			uint8_t run[] = {
				/* frame 1 */
				/* frame 2 */
				/* frame 3 */
				/* frame 4 */
			};
		};

		struct climbing {
			uint8_t frames = 1;
			uint8_t tiles[] = { 8 };
			uint8_t clumb[] = {
				/* frame 1 */
			};
		};

		struct jumping {
			uint8_t frames = 3;
			uint8_t tiles[][2] = { 8, 10, 8 };
			uint8_t jump[] = {
				/* frame 1 */
				/* frame 2 */
				/* frame 3 */
			};
		};

		struct rolling {
			uint8_t frames = 4;
			uint8_t tiles[] = { 9, 16, 10, };
		};
#endif

} player_t;

player_t player1, player2;

uint8_t checkSysHW() {
	/* Check hardware info */
	uint8_t sys = in(0);
	/* 5 means it's a PAL system */
	if(sys & 5) {
		return 1;
	}

	return 0;
}

void setup_vdp_registers() {
	int x;

	for(x = 0; x < 10; ++x) {
		tms9918a_register_write(x, vdp_registers[x]);
	}
}

void clear_vram() {
	uint16_t x;

	/* clear palette */
	tms9918a_cram_set_address(0x00);
	for(x = 0; x < 0x20; ++x) {
		tms9918a_write(0);
	}

	/* clear the entire vram */
	tms9918a_vram_set_write_address(0);
	for(x = 0; x < 0x4000; ++x)
		tms9918a_write(0);
}

void setup_palette() {
	int x, size;

	/* load the palette */
	tms9918a_cram_set_address(0x00);

	/* clear the background palette of the cram */
	size = sizeof(env_palette);
	for(x = 0; x < size; ++x) {
		tms9918a_write(env_palette[x]);
	}

	/* setup our sprite palette */
	tms9918a_cram_set_address(0x10);
	size = sizeof(player_palette);
	for(x = 0; x < size; ++x) {
		tms9918a_write(player_palette[x]);
	}
}

void load_tiles() {
	int x, size;

	/* load the tiles */
	tms9918a_vram_set_write_address(0x0000);

	size = sizeof(env_tiles);
	for(x = 0; x < size; ++x) {
		tms9918a_write(env_tiles[x]);
	}

	// for 8x16 tiles, we need to push the start address by 1 8x16 tile
	tms9918a_vram_set_write_address(size + 32);
	size = sizeof(player_tiles);
	for(x = 0; x < size; ++x) {
		tms9918a_write(player_tiles[x]);
	}
}

/**
 * Draw a tile a sprite at x,y
 *
 * sprite is a number 0-63 for the sprite to draw
 * x is the x to draw sprite at
 * y is the y to draw sprite at
 * tile is the VDP tile # to draw
 */
void draw_sprite(uint8_t sprite, uint8_t x, uint8_t y, uint8_t tile) {
	tms9918a_vram_set_write_address(0x3F00 + sprite);
	tms9918a_write(y);

	tms9918a_vram_set_write_address(0x3F80 + sprite + sprite);
	tms9918a_write(x);
	tms9918a_write(tile);
}

void draw_level() {
	int x, y, addr = 0x3800;

	tms9918a_vram_set_write_address(addr);

	for(y = 0; y < 24; ++y) {
		for(x = 0; x < 32; ++x) {
			tms9918a_write(level[y][x]);
			tms9918a_write(0);
		}
	}
}

void draw_lava() {
	static unsigned char lava[] = { 13, 14 };
	static uint8_t frame = 0, delay = 0;
	int x, y = 0;

	if(delay == 0) {
		frame ^= 2;
		tms9918a_vram_set_write_address(0x3DC0);
		for(x = 0; x < 32; ++x) {
			tms9918a_write(lava[y]);
			tms9918a_write(frame);
			y ^= 1;
		}
	}
	delay += 0x04;
}

void draw_player_animation(player_t *player, uint8_t *animation, uint8_t frame) {
	int x, y, vsprite = 0, sprite = 0, cntX, cntY;

	if(frame > 0) {
		sprite += 16 * frame;
	}

	for(cntY = 0, y = player->y; cntY < 4; ++cntY, y += 8) {
		for(cntX = 0, x = player->x; cntX < 4; ++cntX, x += 8, ++sprite, ++vsprite) {
			draw_sprite(player->sprites[vsprite], x, y, animation[sprite]);
		}
	}
}

void change_player_animation(player_t *player, uint8_t animation) {
	if(animation == STAND) {
		player->animation_frames = 2;
		player->current_frame = 0;
		player->animation_delay = 15;
		player->current_animation = STAND;
		player->delay = player->animation_delay;
		player->animation_sprites = player_stand;
	}
	else if(animation == CLIMB) {
		player->animation_frames = 1;
		player->current_frame = 0;
		player->animation_delay = 15;
		player->current_animation = CLIMB;
		player->delay = player->animation_delay;
		player->animation_sprites = player_climb;
	}
	else if(animation == RUN) {
		player->animation_frames = 4;
		player->current_frame = 0;
		player->animation_delay = 10;
		player->current_animation = RUN;
		player->delay = player->animation_delay;
		player->animation_sprites = player_run;
	}
	else if(animation == DUCK) {
		player->animation_frames = 1;
		player->current_frame = 0;
		player->animation_delay = 15;
		player->current_animation = FALLING;
		player->delay = player->animation_delay;
		player->animation_sprites = player_duck;
	}
	else if(animation == FALLING) {
		player->animation_frames = 1;
		player->current_frame = 0;
		player->animation_delay = 15;
		player->current_animation = FALLING;
		player->delay = player->animation_delay;
		player->animation_sprites = player_fall;
	}
}

void setup_player(player_t *player, uint8_t which) {
	int cnt = 0, end = 16, x = 16, y = 0x80-1;

	if(which == 2) {
		cnt += end;
		end += end;
		x = 224;
	}

	player->x = x;
	player->y = y;
	change_player_animation(player, STAND);

	for(x = 0; cnt < end; ++x, ++cnt) {
		player->sprites[x] = cnt;
	}
}

void draw_player() {
	uint8_t vdp = io_vdp_ctrl;

	player1.delay--;
	if(player1.delay == 0) {
		player1.delay = player1.animation_delay;
		++player1.current_frame = player1.current_frame == player1.animation_frames ? 0 : player1.current_frame;
	}


	draw_player_animation(&player1, player1.animation_sprites, player1.current_frame);
}

void update_screen() {
	draw_lava();
	draw_player();
}

void off_screen() {
	static uint8_t off = 0;
	off ^= 0x40;
	/* turn on screen */
	tms9918a_register_write(1, vdp_registers[1] | off);
}

void main() {
	int delay = 0;
// 	player_t player1, player2;
	uint8_t input1, input2;
	uint8_t player1_level_posX = 0, player1_level_posY = 0;
	uint8_t left_tile_under_player, right_tile_under_player;

// 	if(checkSysHW() == 1) {
// 		/* Then we exit because we hate PAL systems here in NTSC country */
// 		return;
// 	}

	setup_vdp_registers();
	clear_vram();
	setup_palette();
	setup_player(&player1, 1);
	setup_player(&player2, 2);
	load_tiles();
	draw_level();
	draw_lava();

	vdp_add_vsync_handler(&update_screen);
	sms_add_pause_handler(&off_screen);

	/* turn on screen */
	tms9918a_register_write(1, vdp_registers[1] | 0x40);

	/* Now we're ready to turn interrupts on */
	EI

	while(1) {
		if(delay < 0x00FF) {
			delay++;
			continue;
		}
		delay = 0;

		/* get input */
		input1 = ~io_ab_reg;
		input2 = ~io_bmisc_reg;

		player1_level_posX = (uint8_t)(player1.x / 8);
		player1_level_posY = (uint8_t)(player1.y + 0x21) / 8;
		left_tile_under_player = level[player1_level_posY][player1_level_posX];
		right_tile_under_player = level[player1_level_posY][player1_level_posX+1];

		if(input1 & A_LEFT) {
			if(player1.current_animation != DUCK) {
				if(player1.current_animation != RUN) {
					change_player_animation(&player1, RUN);
				}
				player1.x--;
			}
		}
		else if(input1 & A_RIGHT) {
			if(player1.current_animation != DUCK) {
				if(player1.current_animation != RUN) {
					change_player_animation(&player1, RUN);
				}
				player1.x++;
			}
		}

		// player is falling
		if((left_tile_under_player == 0 && right_tile_under_player == 0) || (player1.current_animation == FALLING && (left_tile_under_player != 1 && left_tile_under_player != 2 && left_tile_under_player != 3 && left_tile_under_player != 4))) {
			if(player1.current_animation != FALLING) {
				change_player_animation(&player1, FALLING);
			}
			player1.y++;
		}
		else if (!(input1 & A_DOWN || input1 & A_UP || input1 & A_LEFT || input1 & A_RIGHT)) {
			if(player1.current_animation != STAND) {
				change_player_animation(&player1, STAND);
			}
		}

		if((left_tile_under_player != 0 && right_tile_under_player != 0) && input1 & A_UP) {
			// player jump
			if(left_tile_under_player == 1) {
				if(player1.current_animation != JUMP) {
					change_player_animation(&player1, JUMP);
				}
			}
			// player climbing
			else if((left_tile_under_player >= 2 && left_tile_under_player <= 4 )|| (left_tile_under_player >= 7 && left_tile_under_player <= 12)) {
				if(player1.current_animation != CLIMB) {
					change_player_animation(&player1, CLIMB);
				}
				player1.y--;
			}
		}
		else if((left_tile_under_player != 0 && right_tile_under_player != 0) && input1 & A_DOWN) {
			// player duck
			if(left_tile_under_player <= 2) {
				if(player1.current_animation != DUCK) {
					change_player_animation(&player1, DUCK);
				}
			}
			else if(left_tile_under_player >= 2 || left_tile_under_player >= 7 || left_tile_under_player <= 12) {
				if(player1.current_animation != CLIMB) {
					change_player_animation(&player1, CLIMB);
				}
				player1.y++;
			}
		}

		// hit lava
		if(left_tile_under_player > 12) {
			setup_player(&player1, 1);
		}
	}
}
