;=============================================================================
; zoom.asm - Mandelbrot Set Zoom Demostration.
;                                                    File created: 10-11-93
; Copyright (C) 1993, Carlos Hasan                  Last modified: 11-21-93
;
; Description:
;  This file implements the Runtime MandelZoom demostration. Needs the
;  Mandelbrot Set calculation routines and the zoom-in path include file.
;
; Portability:
;  Requires Turbo Assembler 3.2 or better to be assembler.
;  Dependent on the IBM PC 386 and the VGA graphics card.
;
; Modifications:
;  10/22/93 - Startup Code.
;  11/21/93 - Now this thing Shows the Frame Creation.
;=============================================================================

                .model  small,pascal
                .386

                dosseg                          ; used to link like
                .stack  1024                    ; an standalone program.

                global  MandelZoom:proc
                global  DrawMandel:proc         ; in MANDEL.ASM

;===================== Demo Equates ==========================================

MAXWIDTH        equ     320                     ; VGA Screen Dimensions.
MAXHEIGHT       equ     200

;===================== Demo Data =============================================

                .data

                include ZoomPath.Inc            ; Zoom-In Information.
                                                ; and Color Palette.

GridX           dw      WINSIZEX dup (?)        ; Used to compute the
GridY           dw      WINSIZEY dup (?)        ; Zoom-In Window.

;===================== Demo Routines =========================================

                .code

;-----------------------------------------------------------------------------
; WaitVRT - Waits the next VGA Vertical Retrace Period.
;-----------------------------------------------------------------------------

WaitVRT         proc near

                mov     dx,3DAh
WaitVRT1:       in      al,dx
                test    al,8
                jz      WaitVRT1
WaitVRT2:       in      al,dx
                test    al,8
                jnz     WaitVRT2
                ret

WaitVRT         endp

;-----------------------------------------------------------------------------
; BlackPalette - Set a Black Palette.
;-----------------------------------------------------------------------------

BlackPalette    proc near

                call    WaitVRT
                mov     dx,3C8h
                xor     al,al
                out     dx,al
                inc     dx
                mov     cx,768
BlackLoop:      out     dx,al
                loop    BlackLoop
                ret

BlackPalette    endp


;-----------------------------------------------------------------------------
; SetPalette - Sets the Color Palette.
;-----------------------------------------------------------------------------

SetPalette      proc near

                call    WaitVRT
                mov     dx,3C8h
                xor     al,al
                out     dx,al
                inc     dx
                lea     si,[Palette]
                mov     cx,768
                rep     outsb
                ret

SetPalette      endp

;-----------------------------------------------------------------------------
; FadeInPalette - Fade-In the palette.
;-----------------------------------------------------------------------------

FadeInPalette   proc near

                xor     bx,bx
FadeInLoop:     lea     si,[Palette]
                mov     dx,3C8h
                xor     al,al
                out     dx,al
                mov     cx,768
                call    WaitVRT
                mov     dx,3C9h
FadeIn:         lodsb
                mul     bl
                mov     al,ah
                out     dx,al
                loop    FadeIn
                add     bl,4
                jnc     FadeInLoop
                ret

FadeInPalette   endp


;-----------------------------------------------------------------------------
; FadeOutPalette - Fade-Out the palette.
;-----------------------------------------------------------------------------

FadeOutPalette  proc near

                mov     bl,252
FadeOutLoop:    call    WaitVRT
                mov     dx,3C8h
                xor     al,al
                out     dx,al
                inc     dx
                lea     si,[Palette]
                mov     cx,768
FadeOut:        lodsb
                mul     bl
                mov     al,ah
                out     dx,al
                loop    FadeOut
                sub     bl,4
                jnc     FadeOutLoop
                ret

FadeOutPalette  endp


;-----------------------------------------------------------------------------
; ClearScreen - Clears the Video Screen using a Spray-like effect.
;-----------------------------------------------------------------------------

ClearScreen     proc near

                push    ds
                push    es

                mov     ax,0A000h
                mov     es,ax

                xor     cx,cx
SprayLoop:      push    cx
                xor     bx,bx
SprayLine:      push    bx
                mov     ah,cs:[RandTable+bx]
                mov     bx,cx
                mov     al,cs:[RandTable+bx]
                mov     bx,ax
                mov     di,ax
                xor     al,al
                mov     es:[di],al
                pop     bx
                inc     cl
                inc     bl
                jne     SprayLine
                pop     cx
                inc     cl
                jne     SprayLoop

                pop     es
                pop     ds
                ret

RandTable label byte
      db   83,   8,  18, 177,  13, 241, 149, 157
      db   75, 248, 254,  23,  16,  66, 207,  31
      db  211, 183,  80, 242, 218,  27,  15, 128
      db   94,  98,   4,  36, 139, 110,  85, 230
      db  212,  26,  12, 249, 169, 233, 200, 150
      db   95, 114, 130, 167, 202, 187,  76, 145
      db   62, 117, 115, 190, 209,  42, 185, 224
      db  129, 104, 108, 192, 174, 137,  44,  41
      db  141,  53, 179,  81, 181,  57, 147, 210
      db   50, 134, 156, 125, 133, 126, 106, 162
      db   20,  59,  84,   0, 151, 143, 101, 215
      db   68, 246, 189, 197,  99, 213, 112, 144
      db   28, 235, 240,  90, 154,  56, 138, 165
      db   19, 166, 159,  92, 127, 208, 105, 118
      db  119, 153, 191, 184,  87,  70,   9, 164
      db   60, 252,  96,   3, 171,  38, 136,  14
      db    7, 160,  71, 146, 102, 229, 227,  43
      db  221, 182, 217,  30, 131, 219,  61, 180
      db  195, 245, 109, 203,  24,  49, 170, 247
      db   46, 148, 122, 250, 173, 107, 255, 194
      db    6,  37,  93,  22, 168,  97, 193,   5
      db   51, 223, 116,  86,   1,  89, 121, 243
      db  140, 220,  39, 222,  65,  55,  17,  54
      db  175, 206, 214, 155, 142, 163,  25, 188
      db  178,  11, 204, 135, 201, 238,  79, 132
      db  198,  40,  21,  45, 237, 253, 152,  74
      db   32, 111,  52,  47, 236,   2, 176, 239
      db  234,  58, 100,  91, 172,  73,  82, 205
      db   34,  88,  78, 231, 232, 225,  48, 251
      db   67, 123, 244,  29, 216,  64, 196,  69
      db  199,  33,  72,  35,  10,  63, 161, 228
      db  113, 158, 120, 103, 124, 186, 226,  77

ClearScreen   endp


;-----------------------------------------------------------------------------
; ZoomBox - Draws the Zoom Window using the precalculated Mandelbrot
;  pictures and coordinates in the ZoomPath include file.
;-----------------------------------------------------------------------------

ZoomBox         proc near XStart:word,YStart:word,XStop:word,YStop:word, \
                     PicSeg:word

                mov     si,[XStart]             ; compute the GridX array.
                mov     di,[XStop]
                lea     bx,[GridX]
                mov     cx,WINSIZEX
                mov     ax,di
                sub     ax,si
                cwd
                div     cx
                xchg    si,ax
                mov     di,dx
                mov     dx,cx
                shr     dx,1
                neg     dx
MakeGridX:      mov     [bx],ax
                add     ax,si
                add     dx,di
                jl      SkipX
                sub     dx,WINSIZEX
                inc     ax
SkipX:          add     bx,2
                loop    MakeGridX

                mov     si,[YStart]             ; compute the GridY array.
                mov     di,[YStop]
                lea     bx,[GridY]
                mov     cx,WINSIZEY
                mov     ax,di
                sub     ax,si
                cwd
                div     cx
                xchg    si,ax
                mov     di,dx
                mov     dx,cx
                shr     dx,1
                neg     dx
MakeGridY:      mov     [bx],ax
                add     ax,si
                add     dx,di
                jl      SkipY
                sub     dx,WINSIZEY
                inc     ax
SkipY:          add     bx,2
                loop    MakeGridY

                mov     dx,[PicSeg]
                mov     cx,ds
                mov     ax,0A000h
                mov     es,ax
                mov     di,WINPOSX+320*WINPOSY
                cld


                xor     bx,bx
ZoomLoopY:      push    bx
                mov     ax,[GridY+bx]
                imul    ax,ax,MAXWIDTH
                xor     bx,bx
ZoomLoopX:      mov     si,ax
                add     si,[GridX+bx]
                mov     ds,dx
                movsb
                mov     ds,cx
                add     bx,2
                cmp     bx,2*WINSIZEX
                jb      ZoomLoopX
                pop     bx
                add     di,MAXWIDTH-WINSIZEX
                add     bx,2
                cmp     bx,2*WINSIZEY
                jb      ZoomLoopY
                ret

ZoomBox         endp


;-----------------------------------------------------------------------------
; MandelZoom - Performs the demonstration. Because this routine is called
;   from a High-Level language the allocation and deallocation of memory
;   is done by the caller.
; In:
;   MemSegs - Array of NUMPICS segments of 64K to store the pictures.
;-----------------------------------------------------------------------------

MandelZoom      proc    MemSegs:dword

                mov     ax,13h                  ; set VGA 320x200x256 mode.
                int     10h

                call    SetPalette              ; Sets Color Palette

                les     si,[MemSegs]            ; Creates the Mandelbrot
                mov     cx,NUMPICS              ; piccys into the virtual
                xor     bx,bx                   ; screens.
MakePics:       pusha
                push    ds 
                push    es
                mov     eax,[bx+PicTable+0]     ; get fixedpoint X,Y,DX,DY.
                mov     edx,[bx+PicTable+4]
                mov     esi,[bx+PicTable+8]
                mov     edi,[bx+PicTable+12]    ; Draw Mandelbrot Set.
                call    DrawMandel,eax,edx,esi,edi,0A000h
                pop     es
                pop     ds
                popa

                pusha                           ; Copy Picture:
                push    ds
                push    es
                mov     ax,0A000h               ; get screen segment
                mov     ds,ax
                mov     es,es:[si]              ; get vscreen segment
                xor     di,di
                xor     si,si
                mov     cx,32000                ; copy screen
                cld
                rep     movsw
                call    ClearScreen             ; clears screen
                pop     es
                pop     ds
                popa

                add     si,2                    ; Next Picture Gap.
                add     bx,16
                loop    MakePics

                call    BlackPalette            ; set Black Palette.

DemoLoop:
                mov     cx,NUMFRAMES            ; Zoom-In.
                xor     bx,bx
ZoomIN:         call    WaitVRT
                push    bx
                push    cx
                les     si,[MemSegs]
                mov     ax,[bx+FrameTable+8]    ; PicNo.
                add     si,ax
                add     si,ax
                mov     cx,es:[si]              ; PicSeg.
                mov     ax,[bx+FrameTable+0]    ; XStart.
                mov     dx,[bx+FrameTable+2]    ; YStart.
                mov     si,[bx+FrameTable+4]    ; XStop.
                mov     di,[bx+FrameTable+6]    ; YStop.
                call    ZoomBox,ax,dx,si,di,cx
                pop     cx
                pop     bx
                test    bx,bx
                jne     DontFadeIn
                push    bx
                push    cx
                call    FadeInPalette
                pop     cx
                pop     bx
DontFadeIn:     add     bx,10
                loop    ZoomIN

                mov     cx,NUMFRAMES            ; Zoom-Out.
                mov     bx,10*NUMFRAMES-10
ZoomOUT:        call    WaitVRT
                push    bx
                push    cx
                les     si,[MemSegs]
                mov     ax,[bx+FrameTable+8]    ; PicNo.
                add     si,ax
                add     si,ax
                mov     cx,es:[si]              ; PicSeg.
                mov     ax,[bx+FrameTable+0]    ; XStart.
                mov     dx,[bx+FrameTable+2]    ; YStart.
                mov     si,[bx+FrameTable+4]    ; XStop.
                mov     di,[bx+FrameTable+6]    ; YStop.
                call    ZoomBox,ax,dx,si,di,cx
                pop     cx
                pop     bx
                test    bx,bx
                jne     DontFadeOut
                push    bx
                push    cx
                call    FadeOutPalette
                pop     cx
                pop     bx
DontFadeOut:    sub     bx,10
                loop    ZoomOUT

                mov     ax,03h                  ; set VGA 80x25x16 mode.
                int     10h
                ret

MandelZoom      endp

;-----------------------------------------------------------------------------
; Start - Startup Code called from DOS.
; In:
;   ES - Program Segment Prefix.
;-----------------------------------------------------------------------------

Start           proc
                local   PicSegs:word:NUMPICS

                mov     ax,@Data
                mov     ds,ax
                mov     ax,ss                   ; shrink memory block.
                mov     bx,es
                sub     ax,bx
                mov     bx,sp
                shr     bx,4
                inc     bx
                add     bx,ax
                mov     ah,4Ah
                int     21h

                lea     bx,[PicSegs]
                mov     cx,NUMPICS
AllocLoop:      push    bx
                push    cx
                mov     ah,48h                  ; alloc 64000 bytes per
                mov     bx,4000                 ; picture.
                int     21h
                jc      Exit
                pop     cx
                pop     bx
                mov     ss:[bx],ax
                add     bx,2
                loop    AllocLoop

                lea     bx,[PicSegs]            ; do demostration.
                call    MandelZoom,ss,bx

                lea     bx,[PicSegs]
                mov     cx,NUMPICS              ; Free 64000 bytes
FreeLoop:       push    bx                      ; per picture.
                push    cx
                mov     ah,49h
                mov     es,ss:[bx]
                int     21h
                pop     cx
                pop     bx
                add     bx,2
                loop    FreeLoop

Exit:           mov     ax,4C00h
                int     21h

Start           endp

                end     Start
