// Helper program to find good seeds for the heightmap texture
// used in the 256-byte intro "Grey Tunnel With No Music".
// The program scores all maps according to their minimum,
// maximum and spread, and displays the best maps.
// Each of these can then be further inspected using the
// viewer at https://www.shadertoy.com/view/3dSSRR and by
// trying them out in the intro.

use std::f32::consts::PI;
use std::io::{self, Write};

const CAPACITY: usize = 15;

#[derive(Clone, Copy)]
struct Map {
	cosine: bool,
	seed: u16,
	min: i16,
	max: i16,
	spread: i16,
}

impl Map {
	// Scoring function. Smaller is better. Maps with limits close to +/- 29 are good,
	// and having a high spread is good.
	fn score(&self) -> i16 {
		((self.max - 29).abs() + (-self.min - 29).abs()) * 20 - self.spread
	}
}

// Calculate min, max and spread for the two maps (sine and cosine) with the given seed.
fn limits_for_map(seed: u16, sintab: &[f32]) -> [Map; 2] {
	let mut maps = [
		Map { cosine: false, seed, min: 0, max: 0, spread: 0 },
		Map { cosine: true,  seed, min: 0, max: 0, spread: 0 },
	];
	let mut var = [0i32; 2];
	for pos in 0..0x10000 {
		let mut sums = [0f32; 2];
		for layer in 0..0x100 {
			let r = layer * seed;
			let a = ((r & 0x1f07) - 0x1000) * pos as u16 + r;
			sums[0] += sintab[a as usize];
			sums[1] += sintab[(a + 0x4000) as usize];
		}
		for i in 0..2 {
			let s = sums[i].round() as i16;
			maps[i].min = maps[i].min.min(s);
			maps[i].max = maps[i].max.max(s);
			var[i] += (s as i32).pow(2);
		}
	}
	for i in 0..2 {
		maps[i].spread = (var[i] as f32).sqrt().round() as i16;
	}

	maps
}

fn main() {
	let mut sintab = vec![0f32; 0x10000];
	for v in 0..0x10000 {
		sintab[v] = ((v as f32) * (PI / 0x8000 as f32)).sin();
	}

	let mut best: Vec<Map> = vec![];

	for seed in (1..0x2000).step_by(2) {
		let mut new = false;
		for &map in &limits_for_map(seed, &sintab) {
			if map.min < -99 || map.min >= 0 || map.max <= 0 || map.max > 99 { continue; }
			let mut i = best.len();
			while i > 0 && best[i - 1].score() > map.score() { i -= 1; }
			best.insert(i, map);
			best.truncate(CAPACITY);
			new |= i < CAPACITY;
		}
		let mut s = format!("{:04X}: ", seed);
		if new {
			for m in best.iter() {
				let seedcol = if m.cosine { 3 } else { 7 };
				// White for sine, yellow for cosine
				s += &format!(" \x1b[1m\x1b[3{:1}m{:04X}\x1b[31m{:02}\x1b[32m{:02}\x1b[0m", seedcol, m.seed, -m.min, m.max);
			}
		}
		print!("\r{}", s);
		io::stdout().flush().ok();
	}
	println!();
}
