-- title: Dark Planets
-- author: Wojciech Graj
-- desc: Lovebyte 2023 512B Submission
-- website: https://github.com/wojciech-graj/lovebyte2023
-- license: BSL-1.0 License
-- script: lua

--          Copyright Wojciech Graj 2022-2023.
-- Distributed under the Boost Software License, Version 1.0.
--    (See accompanying file LICENSE or copy at
--          https://www.boost.org/LICENSE_1_0.txt)

--- Generate a pseudorandom number
-- @return float: number between [0-1]
function hash(x)
   return math.sin(x) * 999 % 1
end

--- mix x and y based on a
function mix(x, y, a)
   return y * a + x * (1 - a)
end

--- calculate the mix ratio for noise function
function mixpct(v)
   v = v % 1
   return v * v * (3 - 2 * v)
end

--- generate 3D noise
function noise(x, y, z)
   n = x // 1 + y // 1 * 57 + z // 1 * 113
   return mix(
      mix(
         mix(hash(n), hash(1 + n), mixpct(x)),
         mix(hash(57 + n), hash(58 + n), mixpct(x)),
         mixpct(y)),
      mix(
         mix(hash(113 + n), hash(114 + n), mixpct(x)),
         mix(hash(170 + n), hash(171 + n), mixpct(x)),
         mixpct(y)),
      mixpct(z))
end

--- draw sphere with lighting and texture
-- @param texf function(x,y,z): returns brightness [0-1] at point
function sphere(x0, y0, r, texf)
   for x = -r, r do
      for y = -r, r do
         z = r ^ 2 - x ^ 2 - y ^ 2
         if z > 0 then
            pix(x + x0, y + y0,
               math.min(math.max(
                  1.3 * (1 - math.abs(x / r - math.tan(t / 5999 - 1.57) / 5)) * texf(x, y, z ^ .5),
               0), 1) * 15
            )
         end
      end
   end
end

--- draw moon
function moon()
   sphere(92 * math.sin(t / 999) + 120,
      28 * math.sin(t / 999 - .5) * math.sin(t / 9999 + math.sin(t / 9999) * math.sin(t / 9999 + 1.57)) + 68,
      24,
      function(x, y, z)
         return noise(x / 3 + t / 999 * 3, y / 3 + t / 999, z / 3 + t / 999)
      end)
end

-- initialize stars array
-- each star has form [t] = {x, y}
stars = {}

-- set grayscale palette
for i = 0, 47 do
   poke(16320 + i, i // 3 * 15)
end

function TIC()
--{
   t = time()
   cls()
--}

   -- add new star
   stars[t // 1] = {hash(t) * 240, hash(t / 999) * 136}

   -- draw stars
   for i, star in pairs(stars) do
      pix(star[1], star[2], (t - i) / 444)
      if t - i > 5999 then
         stars[i] = _
      end
   end

   -- draw sun
   if math.abs(math.tan(t / 5999 - 1.57)) > 4 then
      circ(120 - math.tan(t / 5999) * 499, 68, 8, 15)
   end

   -- draw moon
   -- this if statement is redundant but improves performance
   if (t / 999 - .9) % 6.28 < 3.14 then
      moon()
   end

   -- draw planet
   sphere(120, 68, 46,
      function(x, y, z)
         return (noise(x / 15 + t / 3999, y / 15, z / 15) * 2 % 1) ^ 1.3
      end)

   -- redraw moon if in front of planet
   if (t / 999 - .9) % 6.28 > 3.14 then
      moon()
   end
end
