//-------------------------------------------------------------------------------------
//
// Copyright 2009 Intel Corporation
// All Rights Reserved
//
// Permission is granted to use, copy, distribute and prepare derivative works of this
// software for any purpose and without fee, provided, that the above copyright notice
// and this statement appear in all copies.  Intel makes no representations about the
// suitability of this software for any purpose.  THIS SOFTWARE IS PROVIDED "AS IS."
// INTEL SPECIFICALLY DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, AND ALL LIABILITY,
// INCLUDING CONSEQUENTIAL AND OTHER INDIRECT DAMAGES, FOR THE USE OF THIS SOFTWARE,
// INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PROPRIETARY RIGHTS, AND INCLUDING THE
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  Intel does not
// assume any responsibility for any errors which may appear in this software nor any
// responsibility to update it.
//

#include "GfxMath.h"


//________________________________________________________________________________
const CMatrix& MtxMul(CMatrix* pAtoC, const CMatrix& AtoB, const CMatrix& BtoC)
{
	MTXSET( *pAtoC,
			AtoB.m11 * BtoC.m11 + AtoB.m21 * BtoC.m12 + AtoB.m31 * BtoC.m13 + AtoB.m41 * BtoC.m14 ,
			AtoB.m12 * BtoC.m11 + AtoB.m22 * BtoC.m12 + AtoB.m32 * BtoC.m13 + AtoB.m42 * BtoC.m14 ,
			AtoB.m13 * BtoC.m11 + AtoB.m23 * BtoC.m12 + AtoB.m33 * BtoC.m13 + AtoB.m43 * BtoC.m14 ,
			AtoB.m14 * BtoC.m11 + AtoB.m24 * BtoC.m12 + AtoB.m34 * BtoC.m13 + AtoB.m44 * BtoC.m14 ,
																									
			AtoB.m11 * BtoC.m21 + AtoB.m21 * BtoC.m22 + AtoB.m31 * BtoC.m23 + AtoB.m41 * BtoC.m24 ,
			AtoB.m12 * BtoC.m21 + AtoB.m22 * BtoC.m22 + AtoB.m32 * BtoC.m23 + AtoB.m42 * BtoC.m24 ,
			AtoB.m13 * BtoC.m21 + AtoB.m23 * BtoC.m22 + AtoB.m33 * BtoC.m23 + AtoB.m43 * BtoC.m24 ,
			AtoB.m14 * BtoC.m21 + AtoB.m24 * BtoC.m22 + AtoB.m34 * BtoC.m23 + AtoB.m44 * BtoC.m24 ,
																									
			AtoB.m11 * BtoC.m31 + AtoB.m21 * BtoC.m32 + AtoB.m31 * BtoC.m33 + AtoB.m41 * BtoC.m34 ,
			AtoB.m12 * BtoC.m31 + AtoB.m22 * BtoC.m32 + AtoB.m32 * BtoC.m33 + AtoB.m42 * BtoC.m34 ,
			AtoB.m13 * BtoC.m31 + AtoB.m23 * BtoC.m32 + AtoB.m33 * BtoC.m33 + AtoB.m43 * BtoC.m34 ,
			AtoB.m14 * BtoC.m31 + AtoB.m24 * BtoC.m32 + AtoB.m34 * BtoC.m33 + AtoB.m44 * BtoC.m34 ,
																									
			AtoB.m11 * BtoC.m41 + AtoB.m21 * BtoC.m42 + AtoB.m31 * BtoC.m43 + AtoB.m41 * BtoC.m44 ,
			AtoB.m12 * BtoC.m41 + AtoB.m22 * BtoC.m42 + AtoB.m32 * BtoC.m43 + AtoB.m42 * BtoC.m44 ,
			AtoB.m13 * BtoC.m41 + AtoB.m23 * BtoC.m42 + AtoB.m33 * BtoC.m43 + AtoB.m43 * BtoC.m44 ,
			AtoB.m14 * BtoC.m41 + AtoB.m24 * BtoC.m42 + AtoB.m34 * BtoC.m43 + AtoB.m44 * BtoC.m44 
		);
		
	return *pAtoC;
}

//________________________________________________________________________________
const CMatrix &Transpose(CMatrix* pDest, const CMatrix &Src)
{	/* transpose can work in place */ 
	float a;
	
	pDest->m11=Src.m11; 
	pDest->m22=Src.m22; 
	pDest->m33=Src.m33; 
	pDest->m44=Src.m44;
	
	a=Src.m12; pDest->m12=Src.m21; pDest->m21=a;
	a=Src.m13; pDest->m13=Src.m31; pDest->m31=a;
	a=Src.m14; pDest->m14=Src.m41; pDest->m41=a;
	
	a=Src.m21; pDest->m21=Src.m12; pDest->m12=a;
	a=Src.m23; pDest->m23=Src.m32; pDest->m32=a;
	a=Src.m24; pDest->m24=Src.m42; pDest->m42=a;
	
	a=Src.m31; pDest->m31=Src.m13; pDest->m13=a;
	a=Src.m32; pDest->m32=Src.m23; pDest->m23=a;
	a=Src.m34; pDest->m34=Src.m43; pDest->m43=a;
	
	a=Src.m41; pDest->m41=Src.m14; pDest->m14=a;
	a=Src.m42; pDest->m42=Src.m24; pDest->m24=a;
	a=Src.m43; pDest->m43=Src.m34; pDest->m34=a;
	
	return *pDest;
}

//________________________________________________________________________________
void CMatrix::Set( const CTransform &T)
{
	float lx, ly, lz;
	float lxlxomc, lxlyomc, lxlzomc, lylyomc, lylzomc, lzlzomc;
	float lxs,lys,lzs;
	float c,s,omc;
	
	SinCos(T.Rot.Angle, &c,&s);

	omc= 1.0f - c;
	
	lx = T.Rot.Axis.x;
	ly = T.Rot.Axis.y;
	lz = T.Rot.Axis.z;

	lxlxomc	= lx*lx * omc;
	lxlyomc	= lx*ly * omc;
	lxlzomc = lx*lz * omc;
	lylyomc	= ly*ly * omc;
	lylzomc	= ly*lz * omc;
	lzlzomc	= lz*lz * omc;
	
	lxs = lx*s;
	lys = ly*s;
	lzs = lz*s;
	
	MTXSETthis(	T.Scl*( lxlxomc + c   ),	T.Scl*( lxlyomc + lzs ),	T.Scl*( lxlzomc - lys  ),	T.Pos.x,
				T.Scl*( lxlyomc - lzs ),	T.Scl*( lylyomc + c   ),	T.Scl*( lylzomc + lxs  ),	T.Pos.y,
				T.Scl*( lxlzomc + lys ),	T.Scl*( lylzomc - lxs ),	T.Scl*( lzlzomc + c    ),	T.Pos.z,
				0.0f,						0.0f,						0.0f,						1.0f	);
}

//________________________________________________________________________________
const CMatrix& MtxLookat(CMatrix* pDest, const CVector3 &Eye, const CVector3 &At, const CVector3 &Up)
{
	CVector3 Vx, Vy, Vz;
	
	Vz = Normalize( At - Eye );
	Vx = Normalize( Cross(Up, Vz) );
	Vy = Cross(Vz, Vx);
	
	MTXSET(	*pDest,
			Vx.x,	Vx.y,	Vx.z,	-Dot(Vx, Eye),
			Vy.x,	Vy.y,	Vy.z,	-Dot(Vy, Eye),
			Vz.x,	Vz.y,	Vz.z,	-Dot(Vz, Eye),
			0.0f,	0.0f,	0.0f,	1.0f			);
			
	return *pDest;
}

// 
const CMatrix& MtxPerspective(CMatrix* pDest, float yFOV, float Aspect, float Near, float Far)
{
	float  HalfFov;
	float  TanHalfFov;
	float  xScale;
	float  yScale;
	
	HalfFov = yFOV*0.5f;

	TanHalfFov = Tan( HalfFov );
	
	yScale = 1.0f / TanHalfFov;
	xScale = yScale / Aspect;
	
	MTXSET(	*pDest,
			xScale,		0.0f,		0.0f,			0.0f,
			0.0f,		yScale,		0.0f,			0.0f,
			0.0f,		0.0f,		Far/(Far-Near),	-Near*Far/(Far-Near),
			0.0f,		0.0f,		1.0f,			0.0f	);

	return *pDest;
}

//________________________________________________________________________________
CVector3 Normalize(const CVector3 &V)
{
	return V * rsqrt( Dot(V,V) );
}

const CVector3& MtxMulPoint(CVector3* pOut, const CVector3& AA, const CMatrix& M)
{
	CVector3	A(AA);
	
	pOut->Set(	M.m11*A.x + M.m12*A.y + M.m13*A.y + M.m14,
				M.m21*A.x + M.m22*A.y + M.m23*A.y + M.m24,
				M.m31*A.x + M.m32*A.y + M.m33*A.y + M.m34 );
				
	return *pOut;
}

const CVector4& MtxMulPoint(CVector4* pRes, const CVector3& A, const CMatrix& M)
{
	pRes->x = M.m11*A.x + M.m12*A.y + M.m13*A.z + M.m14;
	pRes->y = M.m21*A.x + M.m22*A.y + M.m23*A.z + M.m24;
	pRes->z = M.m31*A.x + M.m32*A.y + M.m33*A.z + M.m34;
	pRes->w = 1.0f;
	
	return *pRes;
}

const CVector3& MtxMulVector(CVector3* pRes, const CVector3& AA, const CMatrix& M)
{
	CVector3	A(AA);
	
	pRes->Set(	M.m11*A.x + M.m12*A.y + M.m13*A.y,
				M.m21*A.x + M.m22*A.y + M.m23*A.y,
				M.m31*A.x + M.m32*A.y + M.m33*A.y );
				
	return *pRes;
}

const CVector4& MtxMulVector(CVector4* pRes, const CVector3& A, const CMatrix& M)
{
	pRes->x = M.m11*A.x + M.m12*A.y + M.m13*A.z;
	pRes->y = M.m21*A.x + M.m22*A.y + M.m23*A.z;
	pRes->z = M.m31*A.x + M.m32*A.y + M.m33*A.z;
	pRes->w = 0.0f;
	
	return *pRes;
}

const CVector4& MtxTransposeMulPoint(CVector4* pRes, const CVector3& AA, const CMatrix& M)
{
	CVector3	A,R;
	float		Scale;
	
	A.x = AA.x - M.m14;
	A.y = AA.y - M.m24;
	A.z = AA.z - M.m34;
	
	R.x = M.m11*A.x + M.m21*A.y + M.m31*A.z;
	R.y = M.m12*A.x + M.m22*A.y + M.m32*A.z;
	R.z = M.m13*A.x + M.m23*A.y + M.m33*A.z;
	
	Scale = rsqrt( M.m11*M.m11 + M.m21*M.m21 + M.m31*M.m31 );
	pRes->Set(Scale*R.x, Scale*R.y, Scale*R.z, 1.0f);
	
	return *pRes;
}
